﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "ResFontBase.h"
#include "inlines.h"
#include "nw4r/misc.h"
#include <stdint.h>

namespace NW4R
{
namespace Font
{
namespace UnManaged
{
namespace detail
{

namespace
{
    /*!--------------------------------------------------------------------------*
    @brief        オフセットで格納されているポインタを解決します。

    @param[in]      base  オフセットの基準となるアドレス。
    @param[in]      ptr   解決するポインタ。
    *---------------------------------------------------------------------------*/
    template <typename T>
    inline T* ResolveOffset(char* base, uint32_t ptr)
    {
        if (ptr == 0)
        {
            // NULL にはオフセットを加算しない
            return NULL;
        }
        uint32_t offset = *(reinterpret_cast<uint32_t*>(&ptr));
        return reinterpret_cast<T*>(base + offset);
    }
} //     namespace

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::ResFontBase()

  Description:  コンストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
ResFontBase::ResFontBase()
: mResource(NULL),
  mFontInfo(NULL),
  m_pKerningTable(NULL),
  mRgbaSheetImageBase(NULL),
  m_pPtrOffsetBase(NULL),
  m_CharCodeRangeCount(0)
{
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::~ResFontBase()

  Description:  デストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
ResFontBase::~ResFontBase()
{
    // .NET 用画像バッファの解放
    delete[] mRgbaSheetImageBase;
}



/* ------------------------------------------------------------------------
        構築/破棄
   ------------------------------------------------------------------------ */


/*---------------------------------------------------------------------------*
  Name:         ResFontBase::SetResourceBuffer()

  Description:  割り当てられたバッファとそこに配置されている FINF ブロックへの
                ポインタを設定します。

  Arguments:    pUserBuffer:    割り当てられたバッファへのポインタ。
                pFontInfo:      FINF ブロックへのポインタ。
                pKerningTable:  カーニング情報
  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ResFontBase::SetResourceBuffer(void* pUserBuffer, FontInformation* pFontInfo, FontKerningTable*	pKerningTable)
{
    NW4R_POINTER_ASSERT(pUserBuffer);
    NW4R_POINTER_ASSERT(pFontInfo);
    NW4R_ASSERT( mResource == NULL );
    NW4R_ASSERT( mFontInfo == NULL );

    mResource = pUserBuffer;
    mFontInfo = pFontInfo;
    m_pKerningTable = pKerningTable;

    // ここで行っていたリソースバージョンのアサートチェックは、削除しました。
    // (PCツール、マルチプラットフォーム環境での取り回しが煩雑なため、NW4RResFont.ValidateBinaryに統合しました。)

    InitReaderFunc(GetEncoding());
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::RemoveResourceBuffer()

  Description:  SetResourceBuffer で設定されたパラメータをリセットし、
                割り当てられていたバッファへのポインタを返します。

  Arguments:    なし。

  Returns:      設定されていたバッファへのポインタ。
 *---------------------------------------------------------------------------*/
void*
ResFontBase::RemoveResourceBuffer()
{
    void* pUserData = mResource;

    mResource = NULL;
    mFontInfo = NULL;
    m_pKerningTable = NULL;

    return pUserData;
}




/* ------------------------------------------------------------------------
        フォント全体情報アクセサ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetWidth()

  Description:  フォントの幅を取得します。

  Arguments:    なし。

  Returns:      フォントの幅(ピクセル単位)。
 *---------------------------------------------------------------------------*/
int
ResFontBase::GetWidth() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    return mFontInfo->width;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetHeight()

  Description:  フォントの高さを取得します。

  Arguments:    なし。

  Returns:      フォントの高さ(ピクセル単位)。
 *---------------------------------------------------------------------------*/
int
ResFontBase::GetHeight() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    return mFontInfo->height;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetAscent()

  Description:  フォントのアセントを取得します。

  Arguments:    なし。

  Returns:      フォントのアセント(ピクセル単位)
 *---------------------------------------------------------------------------*/
int
ResFontBase::GetAscent() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    return mFontInfo->ascent;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetDescent()

  Description:  フォントのディセントを取得します。

  Arguments:    なし。

  Returns:      フォントのディセント(ピクセル単位)
 *---------------------------------------------------------------------------*/
int
ResFontBase::GetDescent() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    return mFontInfo->height - mFontInfo->ascent;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetBaselinePos()

  Description:  フォントのベースライン位置を取得します。

  Arguments:    なし。

  Returns:      フォントのベースライン位置(セル上端からのピクセル数)
 *---------------------------------------------------------------------------*/
int
ResFontBase::GetBaselinePos() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    return ResolveOffset<FontTextureGlyph>(m_pPtrOffsetBase, mFontInfo->pGlyph)->baselinePos;//
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetCellHeight()

  Description:  フォントのセルの高さを取得します。

  Arguments:    なし。

  Returns:      フォントのセルの高さ(ピクセル単位)。
 *---------------------------------------------------------------------------*/
int
ResFontBase::GetCellHeight() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    return ResolveOffset<FontTextureGlyph>(m_pPtrOffsetBase, mFontInfo->pGlyph)->cellHeight;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetCellWidth()

  Description:  フォントのセルの幅を取得します。

  Arguments:    なし。

  Returns:      フォントのセルの幅を取得します(ピクセル単位)。
 *---------------------------------------------------------------------------*/
int
ResFontBase::GetCellWidth() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    return ResolveOffset<FontTextureGlyph>(m_pPtrOffsetBase, mFontInfo->pGlyph)->cellWidth;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetMaxCharWidth()

  Description:  フォント中に含まれる文字の最大文字幅を取得します

  Arguments:    なし。

  Returns:      最大文字幅(ピクセル単位)。
 *---------------------------------------------------------------------------*/
int
ResFontBase::GetMaxCharWidth() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    return ResolveOffset<FontTextureGlyph>(m_pPtrOffsetBase, mFontInfo->pGlyph)->maxCharWidth;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetType()

  Description:  フォントのタイプを取得します。

  Arguments:    なし。

  Returns:      Font::TYPE_RESOURCE を返します。
 *---------------------------------------------------------------------------*/
Font::Type
ResFontBase::GetType() const
{
    return TYPE_RESOURCE;
}

FontType
ResFontBase::GetFontType() const
{
    return (FontType)mFontInfo->fontType;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetTextureFormat()

  Description:  フォントのテクスチャフォーマットを返します。

  Arguments:    なし。

  Returns:      フォントのテクスチャフォーマット。
 *---------------------------------------------------------------------------*/
GXTexFmt
ResFontBase::GetTextureFormat() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );

    //return static_cast<GXTexFmt>(mFontInfo->pGlyph->sheetFormat);
    //読み込まれたフォントテクスチャはRGBA8に変換して扱っているので
    //常にGX_TF_RGBA8を返しても問題無い
    return GX_TF_RGBA8;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetLineFeed()

  Description:  1行の高さを返します。

  Arguments:    なし。

  Returns:      1行の高さ(ピクセル単位)。
 *---------------------------------------------------------------------------*/
int
ResFontBase::GetLineFeed() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    return mFontInfo->linefeed;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetDefaultCharWidths()

  Description:  デフォルトの文字幅を返します。

  Arguments:    なし。

  Returns:      文字幅情報。
 *---------------------------------------------------------------------------*/
const CharWidths
ResFontBase::GetDefaultCharWidths() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    return mFontInfo->defaultWidth;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::SetDefaultCharWidths( const CharWidths& )

  Description:  デフォルトの文字幅を設定します。

  Arguments:    widths: 新しいデフォルトの文字幅。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ResFontBase::SetDefaultCharWidths( const CharWidths& widths )
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    NW4R_REFERENCE_ASSERT( widths );
    mFontInfo->defaultWidth = widths;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::SetAlternateChar( CharCode )

  Description:  代替文字を設定します。

  Arguments:    c:  新しい代替文字の文字コード。

  Returns:      代替文字の変更に成功した場合に true。
                設定しようとしている代替文字がフォントに含まれていない場合に
                失敗します。
 *---------------------------------------------------------------------------*/
bool
ResFontBase::SetAlternateChar( CharCode c )
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    GlyphIndex index = FindGlyphIndex(c);

    if( index != GLYPH_INDEX_NOT_FOUND )
    {
        mFontInfo->alterCharIndex = index;
        return true;
    }

    return false;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::SetLineFeed( int )

  Description:  1行の高さを設定します。

  Arguments:    linefeed:   新しい1行の高さ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ResFontBase::SetLineFeed( int linefeed )
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    NW4R_S8_RANGE_ASSERT(linefeed);
    mFontInfo->linefeed = static_cast<s8>(linefeed);
}




/* ------------------------------------------------------------------------
        文字単体情報アクセサ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetCharWidth( CharCode )

  Description:  文字の文字幅を取得します。

  Arguments:    c:  文字幅を取得する文字の文字コード。

  Returns:      文字幅(ピクセル単位)
 *---------------------------------------------------------------------------*/
int
ResFontBase::GetCharWidth( CharCode c ) const
{
    return GetCharWidths(c).charWidth;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetCharWidths( CharCode )

  Description:  文字の文字幅情報を取得します。

  Arguments:    c:  文字幅情報を取得する文字の文字コード

  Returns:      文字幅情報。
 *---------------------------------------------------------------------------*/
const CharWidths
ResFontBase::GetCharWidths( CharCode c ) const
{
    GlyphIndex index = GetGlyphIndex(c);
    return GetCharWidthsFromIndex(index);
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetGlyph( Glyph*, CharCode )

  Description:  文字のグリフデータを取得します。

  Arguments:    glyph:  グリフデータを受け取るバッファへのポインタ。
                c:      グリフデータを取得する文字の文字コード。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ResFontBase::GetGlyph( Glyph* glyph, CharCode c ) const
{
    GlyphIndex index = GetGlyphIndex(c);
    GetGlyphFromIndex(glyph, index);
}

// 文字のグリフデータの有無を取得します。
bool
ResFontBase::HasGlyph(CharCode c) const
{
    return (GLYPH_INDEX_NOT_FOUND != FindGlyphIndex(c));
}


/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetKerning

  Description:  文字のカーニング情報を取得します。

 *---------------------------------------------------------------------------*/
int
ResFontBase::GetKerning(CharCode c0, CharCode c1) const
{
    if (m_pKerningTable == NULL)
    {
        return 0;
    }

    if (!CheckCharCodeRange(c0) || !CheckCharCodeRange(c1))
    {
        return 0;
    }

    u32 firstWordNum = m_pKerningTable->firstWordNum;
    const KerningFirstTableElem* firstTable = m_pKerningTable->firstTable;

    const KerningSecondTable* secondTable = NULL;
    // firstTableからc0を二分探索
    {
        u32 middle;
        u32 start = 0;
        u32 end = firstWordNum;
        for( ;; )
        {
            middle = ( start + end ) / 2;
            u32 firstWord = firstTable[ middle ].firstWord;
            if ( firstWord == static_cast<u32>(c0) )
            {
                break;
            }
            else if ( firstWord < static_cast<u32>(c0) )
            {
                if ( start == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                start = middle;
            }
            else
            {
                if ( end == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                end = middle;
            }
        }

        u32 offset = firstTable[ middle ].offset;
        secondTable = reinterpret_cast<const KerningSecondTable*>(&reinterpret_cast<const u8*>(m_pKerningTable)[offset]);
    }

    u32 secondWordNum = secondTable->secondWordNum;
    const KerningSecondTableElem* secondTableElems = secondTable->elems;

    // secondTableからc1を二分探索
    {
        u32 middle;
        u32 start = 0;
        u32 end = secondWordNum;
        for( ;; )
        {
            middle = ( start + end ) / 2;
            u32 secondWord = secondTableElems[ middle ].secondWord;
            if ( secondWord == static_cast<u32>(c1) )
            {
                // 見つかった。カーニングの値を返す
                return secondTableElems[ middle ].kerningValue;
            }
            else if ( secondWord < static_cast<u32>(c1) )
            {
                if ( start == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                start = middle;
            }
            else
            {
                if ( end == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                end = middle;
            }
        }
    }
}

// ------------------------------------------------------------------------

bool
ResFontBase::HasKerning() const
{
    return m_pKerningTable != NULL;
}

/* ------------------------------------------------------------------------
        文字ストリーム
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetEncoding()

  Description:  フォントが対応している文字列エンコーディング形式を取得します。

  Arguments:    なし。

  Returns:      フォントが対応している文字列エンコーディング形式。
 *---------------------------------------------------------------------------*/
FontEncoding
ResFontBase::GetEncoding() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );

    return static_cast<FontEncoding>(mFontInfo->encoding);
}



/* ------------------------------------------------------------------------
        グリフインデックス
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetGlyphIndex( CharCode )

  Description:  文字のグリフインデックスを取得します。

  Arguments:    c:  グリフインデックスを取得する文字の文字コード。

  Returns:      文字のグリフインデックス。
                フォントに対象の文字が含まれていない場合は代替文字の
                グリフインデックスを返します。
 *---------------------------------------------------------------------------*/
ResFontBase::GlyphIndex
ResFontBase::GetGlyphIndex( CharCode c ) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    GlyphIndex index = FindGlyphIndex(c);
    return (index != GLYPH_INDEX_NOT_FOUND) ? index: mFontInfo->alterCharIndex;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::FindGlyphIndex( CharCode )

  Description:  文字のグリフインデックスを取得します。

  Arguments:    c:  グリフインデックスを取得する文字の文字コード。

  Returns:      文字のグリフインデックス。
                フォントに対象の文字が含まれていない場合は
                ResFontBase::GLYPH_INDEX_NOT_FOUND を返します。
 *---------------------------------------------------------------------------*/
ResFontBase::GlyphIndex
ResFontBase::FindGlyphIndex( CharCode c ) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    const FontCodeMap* pMap;

    pMap = ResolveOffset<FontCodeMap>(m_pPtrOffsetBase, mFontInfo->pMap);

    while( pMap != NULL )
    {
        if( (pMap->ccodeBegin <= static_cast<u32>(c)) && (static_cast<u32>(c) <= pMap->ccodeEnd) )
        {
            return FindGlyphIndex(pMap, c);
        }

        pMap = ResolveOffset<FontCodeMap>(m_pPtrOffsetBase, pMap->pNext);
    }

    return GLYPH_INDEX_NOT_FOUND;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::FindGlyphIndex( const FontCodeMap*, CharCode )

  Description:  文字のグリフインデックスをCMAPブロックから探索します。

  Arguments:    pMap:   探索対象のCMAPブロック本体へのポインタ。
                c:      グリフインデックスを取得する文字の文字コード。

  Returns:      文字のグリフインデックス。
                フォントに対象の文字が含まれていない場合は
                ResFontBase::GLYPH_INDEX_NOT_FOUND を返します。
 *---------------------------------------------------------------------------*/
ResFontBase::GlyphIndex
ResFontBase::FindGlyphIndex(
    const FontCodeMap*  pMap,
    CharCode            c
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( pMap );

    if (!CheckCharCodeRange(c))
    {
        return GLYPH_INDEX_NOT_FOUND;
    }

    u16 index = GLYPH_INDEX_NOT_FOUND;

    switch( pMap->mappingMethod )
    {
    //-----------------------------------------------------------
    // インデックス = 文字コード - オフセット
    case FONT_MAPMETHOD_DIRECT:
        {
            u16 offset = reinterpret_cast<const u16*>(&pMap->mapInfo)[0];
            index = static_cast<u16>(static_cast<u32>(c) - pMap->ccodeBegin + static_cast<u32>(offset));
        }
        break;

    //-----------------------------------------------------------
    // インデックス = table[文字コード - 文字コードオフセット]
    case FONT_MAPMETHOD_TABLE:
        {
            const int table_index = static_cast<u32>(c) - pMap->ccodeBegin;

            index = reinterpret_cast<const u16*>(&pMap->mapInfo)[table_index];
        }
        break;

    //-----------------------------------------------------------
    // インデックス = 二分探索(文字コード)
    case FONT_MAPMETHOD_SCAN:
        {
            const CMapInfoScan* const scanInfo
                = reinterpret_cast<const CMapInfoScan*>(&pMap->mapInfo);
            const CMapScanEntry* first  = &scanInfo->entries[0];
            const CMapScanEntry* last   = &scanInfo->entries[scanInfo->num - 1];

            while( first <= last )
            {
                const CMapScanEntry* mid = first + (last - first) / 2;

                if( mid->ccode < static_cast<u32>(c) )
                {
                    first = mid + 1;
                }
                else if( static_cast<u32>(c) < mid->ccode )
                {
                    last = mid - 1;
                }
                else
                {
                    index = mid->index;
                    break;
                }
            }
        }
        break;

    //-----------------------------------------------------------
    // unknown
    default:
        NW4R_ASSERTMSG(false, "unknwon MAPMETHOD");
    }

    return index;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetCharWidthsFromIndex( GlyphIndex )

  Description:  グリフインデックスに対応する文字の文字幅を取得します。

  Arguments:    index:  文字幅情報を取得する文字のグリフインデックス

  Returns:      文字幅情報へのリファレンス。
 *---------------------------------------------------------------------------*/
const CharWidths&
ResFontBase::GetCharWidthsFromIndex( GlyphIndex index ) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    const FontWidth* pWidth;

    pWidth = ResolveOffset<FontWidth>(m_pPtrOffsetBase, mFontInfo->pWidth);

    while( pWidth != NULL )
    {
        if( (pWidth->indexBegin <= index) && (index <= pWidth->indexEnd) )
        {
            return GetCharWidthsFromIndex( pWidth, index );
        }

        pWidth = ResolveOffset<FontWidth>(m_pPtrOffsetBase, pWidth->pNext);
    }

    return mFontInfo->defaultWidth;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetCharWidthsFromIndex( const FontWidth*, GlyphIndex )

  Description:  CWDHブロック本体からグリフインデックスに対応する文字の
                文字幅情報を取得します。

  Arguments:    pWidth: 探索対象のCWDHブロック本体へのポインタ。
                index:  文字幅情報を取得する文字のグリフインデックス

  Returns:      文字幅情報へのリファレンス。
 *---------------------------------------------------------------------------*/
const CharWidths&
ResFontBase::GetCharWidthsFromIndex(
    const FontWidth*    pWidth,
    GlyphIndex          index
) const
{
    NW4R_POINTER_ASSERT( pWidth );
    return reinterpret_cast<const CharWidths*>(&pWidth->widthTable)[index - pWidth->indexBegin];
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetGlyphFromIndex( Glyph*, GlyphIndex )

  Description:  グリフインデックスに対応するグリフデータを取得します。

  Arguments:    glyph:  グリフデータを格納するバッファへのポインタ。
                index:  取得するグリフデータのグリフインデックス。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ResFontBase::GetGlyphFromIndex(
    Glyph*      glyph,
    GlyphIndex  index
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontInfo );
    const FontTextureGlyph& tg = *ResolveOffset<FontTextureGlyph>(m_pPtrOffsetBase, mFontInfo->pGlyph);

#if 0
    const u32 cellsInASheet = static_cast<u32>(tg.sheetRow * tg.sheetLine);
#else
    //const u32 sheetSeqLine = ( tg.sheetFormat ==  (u16)GX_TF_I4 ? tg.sheetLine/2 : tg.sheetLine );
    const u32 sheetSeqLine = tg.sheetLine;
    const u32 cellsInASheet = static_cast<u32>(tg.sheetRow * sheetSeqLine);
#endif
    const u32 sheetNo       = index       / cellsInASheet;
    const u32 cellNo        = index       % cellsInASheet;
    const u32 cellUnitX     = cellNo      % tg.sheetRow;
#if 0
    const u32 cellUnitY     = cellNo      / tg.sheetRow;
#else
    const u32 cellUnitY     = ( cellNo / tg.sheetRow ) % sheetSeqLine;
#endif
    const u32 cellPixelX    = cellUnitX   * (tg.cellWidth + 1);
    const u32 cellPixelY    = cellUnitY   * (tg.cellHeight + 1);
    const u32 offsetBytes   = sheetNo * tg.sheetSize;
    const void* pSheet      = ResolveOffset<const u8>(m_pPtrOffsetBase, tg.sheetImage) + offsetBytes;

    glyph->pTexture     = pSheet;
    glyph->sheetNo      = sheetNo; // 移植の際に、追加しています。
    const CharWidths& widths = GetCharWidthsFromIndex(index);
    glyph->widths.left         = static_cast<s16>(widths.left);
    glyph->widths.glyphWidth   = static_cast<u16>(widths.glyphWidth);
    glyph->widths.charWidth    = static_cast<u16>(widths.charWidth);
    glyph->widths.rawWidth     = static_cast<u16>(widths.glyphWidth);
    glyph->height       = tg.cellHeight;
    glyph->rawHeight    = tg.cellHeight;
#if 0
    glyph->texFormat    = static_cast<GXTexFmt>(tg.sheetFormat);
#else
    glyph->texFormat    = GX_TF_RGBA8;
#endif
    glyph->texWidth     = tg.sheetWidth;
    glyph->texHeight    = tg.sheetHeight;
    glyph->cellX        = static_cast<u16>(cellPixelX + 1);
    glyph->cellY        = static_cast<u16>(cellPixelY + 1);
    glyph->baselineDifference = 0;
    glyph->isBitmapFont = true;
    glyph->sheetFormat = tg.sheetFormat;
}

/*---------------------------------------------------------------------------*
  Name:         ResFontBase::GetFontTextureGlyph()

  Description:  フォント全体情報アクセサ
                移植の際に追加されました。

  Arguments:    なし

  Returns:      フォント全体情報
 *---------------------------------------------------------------------------*/
const FontTextureGlyph*
ResFontBase::GetFontTextureGlyph() const
{
    if( mFontInfo != NULL )
    {
        return ResolveOffset<FontTextureGlyph>(m_pPtrOffsetBase, mFontInfo->pGlyph);
    }
    return NULL;
}

// .NET モジュールでテクスチャの検索を行う際に利用します。
int ResFontBase::GetSheetNum() const
{
    return ResolveOffset<FontTextureGlyph>(m_pPtrOffsetBase, mFontInfo->pGlyph)->sheetNum;
}

void
ResFontBase::SetCharCodeRange(int count, u32* pFirst, u32* pLast)
{
    m_CharCodeRangeCount = count;
    for (int i = 0; i < count; i++)
    {
        m_CharCodeRangeFirst[i] = pFirst[i];
        m_CharCodeRangeLast[i] = pLast[i];
    }
}

bool
ResFontBase::CheckCharCodeRange(u32 code) const
{
    if (m_CharCodeRangeCount == 0)
    {
        // CharCodeRange の数が 0 の場合はすべての文字コードを使用
        return true;
    }

    for (int i = 0; i < m_CharCodeRangeCount; i++)
    {
        if (m_CharCodeRangeFirst[i] <= code && code <= m_CharCodeRangeLast[i])
        {
            return true;
        }
    }
    return false;
}

            } /* namespace detail */
        } /* UnManaged */
    } /* namespace ut */
} /* namespace nw4r */
