﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifdef NW_LAYOUTEDITOR_CAFE

#define NW_ASSERTION_ENABLE

#include <scfont_TextureCache.h>

#if defined(NW_PLATFORM_WIN32)
// .NET ではテクスチャは利用しません
//#include <gl/glew.h>
#endif

#include <nn/fontll/fontll_ScalableFontEngine.h>

namespace {

bool IsPos2(u32 n)
{
    return (((n - 1) & n) == 0);
}

template <typename ValueT>
inline ValueT
RoundUp(ValueT x, u32 base)
{
    u32 mask = base - 1;
    return static_cast<ValueT>( x + mask ) & ~mask;
}

u32 BSwap( u32 val )
{
    const u32 MASK = 0xFF00FF00;
    val = ( (val & MASK) >> 8 ) | ( (val << 8) & MASK );
    return (val >> 16) | (val << 16);
}

void* DecodeBfttf(void* pData, u32 dataSize)
{
    if (dataSize < 8)
    {
        // データサイズが8より小さい場合は不正
        return NULL;
    }
    static const u32 SCRAMBLE_SIGNETURE = 0x7f9a0218;
    u32* data = static_cast<u32*>(pData);
    if (BSwap(data[0]) == SCRAMBLE_SIGNETURE)
    {
        // デコード済み、何もせずに返す
        return reinterpret_cast<void*>(&data[2]);
    }
    else
    {
        // デコードしていない
        u32 scrambleKey = BSwap(data[0]) ^ SCRAMBLE_SIGNETURE;
        {
            // 正しいbffntのデータ
            // ttfのサイズを取得
            u32 ttfSize = BSwap(data[1]) ^ scrambleKey;
            if (dataSize < ttfSize)
            {
                // ttfのサイズがデータサイズより大きい場合は不正
                return NULL;
            }
            // スクランブル解除
            data[0] = BSwap(SCRAMBLE_SIGNETURE);
            u32 loopEnd = RoundUp(ttfSize, 4) / 4 + 2;
            for (u32 i = 2; i < loopEnd; i++)
            {
                data[i] = BSwap(BSwap(data[i]) ^ scrambleKey);
            }
            return reinterpret_cast<void*>(&data[2]);
        }
    }
}

void* AllocateForFontll(size_t size, size_t alignment, void* pUserData)
{
    NW_UNUSED_VARIABLE(pUserData);
    nw::scfont::IAllocator* pAllocator = static_cast<nw::scfont::IAllocator*>(pUserData);
    return pAllocator->Alloc(size, static_cast<u32>(alignment));
}

void FreeForFontll(void* ptr, void* pUserData)
{
    NW_UNUSED_VARIABLE(pUserData);
    nw::scfont::IAllocator* pAllocator = static_cast<nw::scfont::IAllocator*>(pUserData);
    pAllocator->Free(ptr);
}

}

namespace nw
{
namespace scfont
{

TextureCache::InitializeArg::InitializeArg()
 : textureCacheWidth(1024)
 , textureCacheHeight(1024)
 , allocator(NULL)
 , fontDatas(NULL)
 , fontDataSizes(NULL)
 , boldWeights(NULL)
 , borderWidths(NULL)
 , scaleWidths(NULL)
 , scaleHeights(NULL)
 , fontDataTypes(NULL)
 , ignorePalt(NULL)
 , deleteBearingX(NULL)
 , bearingOffsetX(NULL)
 , forceMonospacedEnabled(NULL)
 , forceMonospacedWidth(NULL)
 , baselineOffset(NULL)
 , fontCount(1)
 , fontListCount(NULL)
 , workMemorySize(WORK_MEMORY_SIZE_DEFAULT)
 , noPlotWorkMemorySize(WORK_MEMORY_SIZE_NO_PLOT_DEFAULT)
 , glyphNodeNumMax(GLYPH_NODE_NUM_MAX_DEFALUT)
 , isDrawMultiCore(false)
{
}

TextureCache::TextureCache()
 : mFontEngine(NULL)
 , mFontEngineNoPlot(NULL)
 , mFontNameBufs(NULL)
 , mTextureCacheBitMap(NULL)
 , mTextureCacheWidth(0)
 , mTextureCacheHeight(0)
 , mLineCurrentPos(0)
 , mFontCount(0)
 , mCurrentFontFace(0)
 , mNoSpaceError(false)
 , mIsDrawMultiCore(false)
 , mFontMetrics(NULL)
{
    for (int i = 0; i < FONT_FACE_COUNT_MAX; i++)
    {
        mFontListCount[i] = 0;
        mFontFaceTable[i] = 0;
}
}

TextureCache::~TextureCache()
{
}

void TextureCache::Initialize(const InitializeArg& arg)
{
    // 引数のチェック
    if ( ! IsPos2(arg.textureCacheWidth))
    {
        NW_ERR("arg.width[%u] is not power of 2.", arg.textureCacheWidth);
        return;
    }
    if (arg.allocator == NULL)
    {
        NW_ERR("arg.allocator must be set.");
        return;
    }
    if (arg.fontDatas == NULL)
    {
        NW_ERR("arg.fontDatas must be set.");
        return;
    }
    if (arg.fontCount == 0 || arg.fontCount > 32)
    {
        NW_ERR("arg.fontNum must be 0 and more, 32 and less.");
        return;
    }
    if (arg.fontDataTypes != NULL && arg.fontDataSizes == NULL)
    {
        for (u32 i = 0; i < arg.fontCount; i++)
        {
            for (u32 j = 0; j < arg.fontListCount[i]; j++)
            {
                if (arg.fontDataTypes[i][j] == FONT_DATA_TYPE_BFTTF || arg.fontDataTypes[i][j] == FONT_DATA_TYPE_BFOTF)
                {
                NW_ERR("arg.fontDataSizes must not be NULL when one of arg.fontDataTypes is FONT_DATA_TYPE_BFTTF or FONT_DATA_TYPE_BFOTF.");
                return;
            }
        }
    }
    }

    int fontFaceCount = InitializeFontFaceTable(arg);
    mFontCount = arg.fontCount;
    mIsDrawMultiCore = arg.isDrawMultiCore;
    mFontNameBufs = static_cast<char*>(arg.allocator->Alloc(fontFaceCount * FONT_NAME_LEN_MAX, 4));
    mFontMetrics = static_cast<FontMetrics*>(arg.allocator->Alloc(fontFaceCount * sizeof(FontMetrics), 4));
    mBoldWeights = static_cast<nn::fontll::ScFixed*>(arg.allocator->Alloc(fontFaceCount * sizeof(nn::fontll::ScFixed), 4));
    mBorderWidths = static_cast<u8*>(arg.allocator->Alloc(fontFaceCount * sizeof(u8), 4));
    mDeleteBearingX = static_cast<bool*>(arg.allocator->Alloc(fontFaceCount * sizeof(bool), 4));
    mBearingOffsetX = static_cast<s16*>(arg.allocator->Alloc(fontFaceCount * sizeof(s16), 4));
    mForceMonospacedEnabled = static_cast<bool*>(arg.allocator->Alloc(fontFaceCount * sizeof(bool), 4));
    mForceMonospacedWidth = static_cast<s16*>(arg.allocator->Alloc(fontFaceCount * sizeof(s16), 4));
    mOtfKerningTable = static_cast<nn::fontll::OtfKerningTable**>(arg.allocator->Alloc(fontFaceCount * sizeof(nn::fontll::OtfKerningTable*), 4));
    mCharCodeRangeCount = static_cast<int*>(arg.allocator->Alloc(fontFaceCount * sizeof(int), 4));
    mCharCodeRangeFirst = static_cast<u32(*)[CHAR_CODE_RANGE_COUNT_MAX]>(arg.allocator->Alloc(fontFaceCount * sizeof(u32[CHAR_CODE_RANGE_COUNT_MAX]), 4));
    mCharCodeRangeLast = static_cast<u32(*)[CHAR_CODE_RANGE_COUNT_MAX]>(arg.allocator->Alloc(fontFaceCount * sizeof(u32[CHAR_CODE_RANGE_COUNT_MAX]), 4));


    mTextureCacheWidth = arg.textureCacheWidth;
    mTextureCacheHeight = arg.textureCacheHeight;
    mLineCurrentPos = 0;
    mCurrentFontFace = UINT_MAX;
    for (u32 k = 0; k < CORE_NUM; k++)
    {
        mCurrentFontFacesNoPlot[k] = UINT_MAX;
    }

    // ステート構造体の確保
    mFontEngine = static_cast<nn::fontll::ScalableFontEngine*>(arg.allocator->Alloc(sizeof(nn::fontll::ScalableFontEngine), 4));
    mFontEngineNoPlot = static_cast<nn::fontll::ScalableFontEngine*>(arg.allocator->Alloc(sizeof(nn::fontll::ScalableFontEngine) * GetFontEngineNoPlotCount(), 4));

    // ステート構造体の初期化
    mFontEngine->Initialize(arg.allocator->Alloc(arg.workMemorySize, 4), arg.workMemorySize);
    AssertFsError("Initialize");
    // このステート構造体ではプロットは行わないので、ワークエリアは固定の小さい領域にする。フォントを複数扱う場合は消費されるサイズはその数の倍数必要。
    for (u32 k = 0; k < GetFontEngineNoPlotCount(); k++)
    {
        mFontEngineNoPlot[k].Initialize(arg.allocator->Alloc(arg.noPlotWorkMemorySize * fontFaceCount, 4), arg.noPlotWorkMemorySize * mFontCount);
        AssertFsErrorNoPlot("Initialize", k);
    }

    // フォントの登録
    {
        int idx = 0;
        for (u32 i = 0; i < mFontCount; i++)
        {
            for (u32 j = 0; j < static_cast<u32>(mFontListCount[i]); j++)
            {
                void* fontData;
                if (arg.fontDataTypes != NULL && (arg.fontDataTypes[i][j] == FONT_DATA_TYPE_BFTTF || arg.fontDataTypes[i][j] == FONT_DATA_TYPE_BFOTF))
                {
                    fontData = DecodeBfttf(arg.fontDatas[i][j], arg.fontDataSizes[i][j]);
                    NW_ASSERT(fontData != NULL);
                }
                else
                {
                            fontData = arg.fontDatas[i][j];
                }
                if (arg.boldWeights != NULL)
                {
                            mBoldWeights[idx] = static_cast<nn::fontll::ScFixed>(static_cast<f32>(1 << 16) * arg.boldWeights[i][j]);
                }
                else
                {
                            mBoldWeights[idx] = 0;
                }
                if (arg.borderWidths != NULL)
                {
                            mBorderWidths[idx] = arg.borderWidths[i][j];
                }
                else
                {
                            mBorderWidths[idx] = 0;
                }
                if (arg.deleteBearingX != NULL)
                {
                            mDeleteBearingX[idx] = arg.deleteBearingX[i][j];
                }
                else
                {
                            mDeleteBearingX[idx] = false;
                }
                if (arg.bearingOffsetX != NULL)
                {
                            mBearingOffsetX[idx] = static_cast<s16>(arg.bearingOffsetX[i][j]);
                }
                else
                {
                            mBearingOffsetX[idx] = 0;
                }
                if (arg.forceMonospacedEnabled != NULL)
                {
                            mForceMonospacedEnabled[idx] = arg.forceMonospacedEnabled[i][j];
                }
                else
                {
                            mForceMonospacedEnabled[idx] = false;
                }
                if (arg.forceMonospacedWidth != NULL)
                {
                            mForceMonospacedWidth[idx] = static_cast<s16>(arg.forceMonospacedWidth[i][j]);
                }
                else
                {
                            mForceMonospacedWidth[idx] = 0;
                }
                if (arg.scaleWidths != NULL)
                {
                            mFontMetrics[idx].scaleWidth = arg.scaleWidths[i][j];
                }
                else
                {
                            mFontMetrics[idx].scaleWidth = 1.0f;
                }
                if (arg.scaleHeights != NULL)
                {
                            mFontMetrics[idx].scaleHeight = arg.scaleHeights[i][j];
                }
                else
                {
                            mFontMetrics[idx].scaleHeight = 1.0f;
                }

                mFontEngine->LoadFont(GetFontNameBuf(idx), fontData, 0, FONT_NAME_LEN_MAX);
                if (arg.charCodeRangeCount != NULL)
                {
                    NW_ASSERT(arg.charCodeRangeCount[i][j] <= CHAR_CODE_RANGE_COUNT_MAX);
                    mCharCodeRangeCount[idx] = arg.charCodeRangeCount[i][j];
                    if (mCharCodeRangeCount[idx] > CHAR_CODE_RANGE_COUNT_MAX)
                    {
                        mCharCodeRangeCount[idx] = CHAR_CODE_RANGE_COUNT_MAX;
                    }
                    for (int k = 0; k < CHAR_CODE_RANGE_COUNT_MAX; k++)
                    {
                        if (k >= mCharCodeRangeCount[idx])
                        {
                            mCharCodeRangeFirst[idx][k] = 0;
                            mCharCodeRangeLast[idx][k] = 0;
                        }
                        else
                        {
                            mCharCodeRangeFirst[idx][k] = arg.charCodeRangeFirst[i][j][k];
                            mCharCodeRangeLast[idx][k] = arg.charCodeRangeLast[i][j][k];
                        }
                    }
                }
                else
                {
                    mCharCodeRangeCount[idx] = 0;
                    for (int k = 0; k < CHAR_CODE_RANGE_COUNT_MAX; k++)
                    {
                        mCharCodeRangeFirst[idx][k] = 0;
                        mCharCodeRangeLast[idx][k] = 0;
                    }
                }

                mFontEngine->LoadFont(GetFontNameBuf(idx), fontData, 0, FONT_NAME_LEN_MAX);
                AssertFsError("LoadFont");
                for (u32 k = 0; k < GetFontEngineNoPlotCount(); k++)
                {
                    mFontEngineNoPlot[k].LoadFont(GetFontNameBuf(idx), fontData, 0, FONT_NAME_LEN_MAX);
                    AssertFsErrorNoPlot("LoadFont", k);
                }

                // フォントのアセントと高さを求めておく
                SetFontFace(idx);
                nn::fontll::Metrics metrics;
                mFontEngine->GetFontMetrics(&metrics);
                AssertFsError("GetFontMetrics");

                mFontMetrics[idx].boundingBoxAscentRatio = static_cast<float>(metrics.fontBoundingBox.yMax) / metrics.metricsResolution;
                mFontMetrics[idx].boundingBoxHeightRatio = static_cast<float>(metrics.fontBoundingBox.yMax - metrics.fontBoundingBox.yMin) / metrics.metricsResolution;
                mFontMetrics[idx].ascent_ratio = static_cast<float>(metrics.os2WinAscent) / metrics.metricsResolution;
                mFontMetrics[idx].height_ratio = static_cast<float>(metrics.os2WinAscent + metrics.os2WinDescent) / metrics.metricsResolution;
                mFontMetrics[idx].baselineOffset = arg.baselineOffset[i][j];

                bool ignorePalt = false;
                if (arg.ignorePalt != NULL)
                {
                    ignorePalt = arg.ignorePalt[i][j];
                }

                if (CheckNecessaryToReadOtfKerningTable(arg, i, j))
                {
                    mOtfKerningTable[idx] = mFontEngine->InitializeOtfKerningTable(AllocateForFontll, arg.allocator, ignorePalt);
                }
                else
                {
                    mOtfKerningTable[idx] = NULL;
                }

                idx++;
            }
        }
    }

    // 高さが最大のものにベースラインを合わせる
    {
        int idx = 0;
        for (uint32_t i = 0; i < mFontCount; i++)
        {
            float maxBoundingBoxAscentRatio = 0.0f;
            float maxBoundingBoxHeightRatio = 0.0f;
            float maxAscentRatio = 0.0f;
            float maxHeightRatio = 0.0f;
            for (uint32_t j = 0; j < static_cast<uint32_t>(mFontListCount[i]); j++)
            {
                if (maxBoundingBoxHeightRatio < mFontMetrics[idx + j].boundingBoxHeightRatio)
                {
                    maxBoundingBoxAscentRatio = mFontMetrics[idx + j].boundingBoxAscentRatio;
                    maxBoundingBoxHeightRatio = mFontMetrics[idx + j].boundingBoxHeightRatio;
                    maxAscentRatio = mFontMetrics[idx + j].ascent_ratio;
                    maxHeightRatio = mFontMetrics[idx + j].height_ratio;
                }
            }
            for (uint32_t j = 0; j < static_cast<uint32_t>(mFontListCount[i]); j++)
            {
                mFontMetrics[idx + j].boundingBoxAscentRatio = maxBoundingBoxAscentRatio;
                mFontMetrics[idx + j].boundingBoxHeightRatio = maxBoundingBoxHeightRatio;
                mFontMetrics[idx + j].ascent_ratio = maxAscentRatio;
                mFontMetrics[idx + j].height_ratio = maxHeightRatio;
            }
            idx += mFontListCount[i];
        }
    }

#if defined(NW_PLATFORM_WIN32)
    mTextureCacheBitMap = static_cast<u8*>(arg.allocator->Alloc(mTextureCacheWidth * mTextureCacheHeight, 4));
    std::memset(mTextureCacheBitMap, 0x0, mTextureCacheWidth * mTextureCacheHeight);
    {
        //GLuint textureCacheId;
        // .NET ではテクスチャは利用しません。（別途作成します）
        //glGenTextures( 1, &textureCacheId );
        //glBindTexture( GL_TEXTURE_2D_ARRAY, textureCacheId );
        //glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        //glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        //glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_WRAP_S, GL_CLAMP );
        //glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_WRAP_T, GL_CLAMP );
        //glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_R, GL_ONE );
        //glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_G, GL_ONE );
        //glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_B, GL_ONE );
        //glTexImage3D(GL_TEXTURE_2D_ARRAY, 0, GL_ALPHA8, mTextureCacheWidth, mTextureCacheHeight, 1, 0, GL_ALPHA, GL_UNSIGNED_BYTE, mTextureCacheBitMap);
        //NW_GL_ASSERT();
        m_TexObj.Set(0, mTextureCacheBitMap, GX_TF_I8, static_cast<u16>(mTextureCacheWidth), static_cast<u16>(mTextureCacheHeight), 1, true);
    }
#elif defined(NW_PLATFORM_CAFE)
    GX2InitTexture(m_TexObj.GetTexture(), mTextureCacheWidth, mTextureCacheHeight, 1, 1, GX2_SURFACE_FORMAT_TC_R8_UNORM, GX2_SURFACE_DIM_2D_ARRAY);
    m_TexObj.GetTexture()->surface.tileMode = GX2_TILE_MODE_LINEAR_ALIGNED;
    // GX2InitTextureで設定されたアラインメントに従ってメモリ確保。DCZeroRangeを使うために、最低64バイトアラインメントになるようにする
    mTextureCacheBitMap = static_cast<u8*>(arg.allocator->Alloc(mTextureCacheWidth * mTextureCacheHeight, ut::RoundUp(m_TexObj.GetTexture()->surface.alignment, 64)));
    // テクスチャキャッシュを更新するときにキャッシュストアされるので、この時点ではDCZeroRangeでよい
    DCZeroRange(mTextureCacheBitMap, mTextureCacheWidth * mTextureCacheHeight);
    DCFlushRange(mTextureCacheBitMap, mTextureCacheWidth * mTextureCacheHeight);
    GX2InitTexturePtrs(m_TexObj.GetTexture(), mTextureCacheBitMap, 0);
    GX2InitTextureCompSel(m_TexObj.GetTexture(), GX2_GET_COMP_SEL(GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_X_R));
    GX2InitTextureRegs(m_TexObj.GetTexture());
    m_TexObj.Set(1, mTextureCacheBitMap, font::FONT_SHEET_FORMAT_A8, mTextureCacheWidth, mTextureCacheHeight, 1, true);
#endif

    mGlyphTreeMap.Initialize(arg.allocator, arg.glyphNodeNumMax);

    // ツール利用では、利用されるフォントが想定できないので常にONにします。
    mFontEngine->SetAutoHint(true);
}

void TextureCache::Finalize(IAllocator* allocator)
{
    void* work = reinterpret_cast<void*>(mFontEngine->GetPointerToWorkBuffer());
    void* noPlotWorks[CORE_NUM];
    for (u32 k = 0; k < GetFontEngineNoPlotCount(); k++)
    {
        noPlotWorks[k] = reinterpret_cast<void*>(mFontEngineNoPlot[k].GetPointerToWorkBuffer());
    }

    {
        int idx = 0;
        for (uint32_t i = 0; i < mFontCount; i++)
        {
            for (uint32_t j = 0; j < static_cast<uint32_t>(mFontListCount[i]); j++)
            {
                if (mOtfKerningTable[idx] != NULL)
                {
                    mFontEngine->FinalizeOtfKerningTable(mOtfKerningTable[idx], FreeForFontll, allocator);
                    mOtfKerningTable[idx] = NULL;
                }
                idx++;
            }
        }
    }

    mFontEngine->Finalize();
    AssertFsError("Finalize");
    for (u32 k = 0; k < GetFontEngineNoPlotCount(); k++)
    {
        mFontEngineNoPlot[k].Finalize();
        AssertFsErrorNoPlot("Finalize", k);
    }

    SafeFree( mFontEngine, allocator );
    SafeFree( mFontEngineNoPlot, allocator );
    SafeFree( work, allocator );
    for (u32 k = 0; k < GetFontEngineNoPlotCount(); k++)
    {
        SafeFree( noPlotWorks[k], allocator );
    }
    SafeFree( mTextureCacheBitMap, allocator );
    mGlyphTreeMap.Finalize(allocator);
    SafeFree( mFontNameBufs, allocator );
    SafeFree( mFontMetrics, allocator );
    SafeFree( mBoldWeights, allocator );
    SafeFree( mBorderWidths, allocator );
    SafeFree( mDeleteBearingX, allocator );
    SafeFree( mBearingOffsetX, allocator );
    SafeFree( mForceMonospacedEnabled, allocator );
    SafeFree( mForceMonospacedWidth, allocator );
    SafeFree( mOtfKerningTable, allocator );
    SafeFree( mCharCodeRangeCount, allocator );
    SafeFree( mCharCodeRangeFirst, allocator );
    SafeFree( mCharCodeRangeLast, allocator );

#if defined(NW_PLATFORM_WIN32)
    // .NET ではテクスチャは利用しません
    //if (m_TexObj.GetName() != 0)
    //{
    //	GLuint textureCacheId = m_TexObj.GetName();
    //    glDeleteTextures(1, &textureCacheId);
    //}
#endif
}

bool TextureCache::RegisterGlyph(char16 code, u32 fontSize, u16 fontFace, s32 lockGroup /* = -1 */)
{
    u32 innerFontFace;
    if (!GetInnerFontFace(&innerFontFace, static_cast<u32>(fontFace), code))
    {
        // 文字が存在しなかった場合は登録を行わずに抜ける。
        return false;
    }
    GlyphNode* node = mGlyphTreeMap.Find(code, static_cast<u16>(fontSize), static_cast<u16>(innerFontFace));

    if (node)
    {
        // 見つかった -> 既に登録されている文字
        node->SetRequested(true);
        if (lockGroup >= 0)
        {
            node->GetLockGroup().SetBitOn(lockGroup);
        }
        // 既に登録されている文字の場合、IsPlottingがtrueのときだけまだ描画できない
        return node->IsPlotting();
    }
    else
    {
        // 見つからない -> まだ登録されていない文字
        node = mGlyphTreeMap.Insert(code, static_cast<u16>(fontSize), static_cast<u16>(innerFontFace));
        if (node)
        {
            node->SetRequested(true);
            node->SetPlotting(true);
            if (lockGroup >= 0)
            {
                node->GetLockGroup().SetBitOn(lockGroup);
            }
            // 未プロットリストに入れる
            mNeedPlotGlyphList.PushBack(node);
            return true;
        }
        else
        {
            // 平衡二分木のノードに空きがない。
            if (EraseNotInFontGlyphs() > 0) {
                // フォントにない文字のノードを削除し、空きができたら登録を行う。(次は登録できるはず)
                return RegisterGlyph(code, fontSize, static_cast<u16>(innerFontFace), lockGroup);
            } else {
                // 空きができなかったら、文字の登録を行わずに抜ける。
                NW_LOG("nw::scfont::TextureCache: Node number max over. Please increase InitializeArg::glyphNodeNumMax.\n");
                mNoSpaceError = true;
                return false;
            }
        }
    }
}


u32 TextureCache::RegisterGlyphsWithLength(const char16* codes, u32 codeNum, u32 fontSize, u16 fontFace, s32 lockGroup /* = -1 */)
{
    u32 count = 0;
    for (u32 i = 0; i < codeNum; i++)
    {
        if (codes[i] == 0)
        {
            break;
        }

        count += static_cast<u32>(RegisterGlyph(codes[i], fontSize, fontFace, lockGroup));
    }
    return count;
}

void TextureCache::UpdateTextureCache()
{
    // 結局テクスチャキャッシュ内の文字の配置はUpdateTextureCache内で行わなければならない。
    // 文字の横幅を取る処理が重く、GetGlyphmap() で取ってきてその場で配置したほうがトータルのコストが小さいため。
    GlyphList::Iterator step_it = mNeedPlotGlyphList.GetBeginIter();
    GlyphList::Iterator end_it = mNeedPlotGlyphList.GetEndIter();
    for (;step_it != end_it;)
    {
        GlyphList::Iterator it = step_it;
        ++step_it;
        mNeedPlotGlyphList.Erase(it);

        SetFontFace(it->GetFace());

        // まず、文字がフォントにあるかどうかを調べる。ここで調べているのは、RegisterGlyphでやると時間がかかり、
        // メインスレッドの遅延に繋がるため。UpdateTextureCacheの中であれば、プロットの処理に紛れる
        if ( ! mFontEngine->CheckGlyphExist(it->GetCode())) {
            // ない場合、削除は行わない。(削除してしまうとその文字のRegisterGlyphが常にtrueになってしまうため。)
            // フォントにない文字フラグを立て、フォントにない文字リストに入れておく。
            it->SetNotInFont(true);
            mNotInFontGlyphList.PushBack(&(*it));
            continue;
        }

        // フォントサイズの設定
        mFontEngine->SetScale(it->GetSize() << 16, 0, 0, it->GetSize() << 16);
        AssertFsError("SetScale");

        // このフォントのアセントを求める
        const int font_ascent = static_cast<int>(mFontMetrics[it->GetFace()].boundingBoxAscentRatio * it->GetSize() + 0.5f);

        // GRAYMAP8固定にする。そのままA8のテクスチャにコピーできるという利点がある。
        // GRAYMAP8の場合は、ScGlyphMapのメンバbplとwidthは同じ値になる。
        nn::fontll::GlyphMap *map = mFontEngine->AcquireGlyphmap(it->GetCode(), nn::fontll::FormatGrayMap8);
        AssertFsError("GetGlyphmap");

        it->SetLineKind(GlyphNode::CalcLineKind(static_cast<u16>(map->height + font_ascent - map->hiY)));

        // 文字をプロットするラインを決める
        LineInfo* matched_line = NULL;
        for (u32 i = 0; i < mLineCurrentPos; i++)
        {
            if (mLineInfos[i].kind == it->GetLineKind() && (static_cast<u32>(mLineInfos[i].currentX + map->width + GLYPH_PADDING) < mTextureCacheWidth))
            {
                matched_line = &mLineInfos[i];
                break;
            }
        }
        // マッチするラインがなかった場合
        if ( ! matched_line)
        {
            if (mLineCurrentPos >= TEXTURE_CACHE_LINE_NUM_MAX)
            {
                // ラインの最大数を超えた。エラー扱い
                NW_LOG("nw::scfont::TextureCache: line num[%d] exceeds\n", mLineCurrentPos);
                mNeedEraseGlyphList.PushBack(&(*it));
                mFontEngine->ReleasesGlyph(map);
                AssertFsError("ReleasesGlyph");
                mNoSpaceError = true;
                continue;
            }

            // 新しいラインを作成する
            matched_line = CreateNewLineImpl(static_cast<LineHeight>(it->GetLineKind()));

            if (matched_line == NULL)
            {
                // 新しいラインを作成できなかった。
                // 現在プロットされているものを消して、その上にプロットする必要がある。

                // 消去可能な文字を探す
                GlyphNode* node = NULL;
                node = FindAndReserveEraseGlyph(it->GetLineKind(), map->width);
                if (node)
                {
                    // 元々あったグリフの領域に上書きする
                    GlyphLineList* list = &mLineInfos[node->GetLineNo()].list;
                    it->SetCachePosX(node->GetCachePosX());
                    it->SetCachePosY(node->GetCachePosY());
                    it->SetLineNo(node->GetLineNo());
                    list->Insert(GlyphLineList::GetIteratorFromPointer(node), &(*it));
                }
                else
                {
                    // 入るスペースがないので抜ける
                    NW_LOG("nw::scfont::TextureCache: no erasable glyph. charCode = 0x%x lineHeight = %d\n", it->GetCode(), 1 << it->GetLineKind());
                    mNeedEraseGlyphList.PushBack(&(*it));
                    mFontEngine->ReleasesGlyph(map);
                    AssertFsError("ReleasesGlyph");
                    mNoSpaceError = true;
                    continue;
                }
            }
        }

        if (matched_line)
        {
            it->SetCachePosX(matched_line->currentX);
            it->SetCachePosY(matched_line->y);
            it->SetLineNo(matched_line->no);
            matched_line->list.PushBack(&(*it));
            matched_line->currentX += map->width + GLYPH_PADDING;
        }

        const int offset_from_top = font_ascent - map->hiY;
        const int render_origin_y = it->GetCachePosY() + offset_from_top;

        const f32 scaleWidth = mFontMetrics[it->GetFace()].scaleWidth;
        const f32 scaleHeight = mFontMetrics[it->GetFace()].scaleHeight;
        const int baselineOffset = mFontMetrics[it->GetFace()].baselineOffset;
        it->SetGlyphWidth(static_cast<u16>(map->width * scaleWidth));
        it->SetGlyphHeight(static_cast<u16>((map->height + offset_from_top) * scaleHeight));
        it->SetCacheWidth(map->width);
        it->SetCacheHeight(static_cast<u16>(map->height + offset_from_top));
        if (mDeleteBearingX[it->GetFace()] && map->width != 0)
        {
            // 文字幅ではなくグリフ幅で描画する。
            // スペースなどの幅 0 のグリフは文字幅で描画する。
            it->SetAdvanceX(static_cast<u16>((map->width + mBearingOffsetX[it->GetFace()]) * scaleWidth));
            it->SetLeftOffset(static_cast<s16>(mBearingOffsetX[it->GetFace()] / 2 * scaleWidth));
        }
        else if (mForceMonospacedEnabled[it->GetFace()])
        {
            // 強制的に等幅で描画する。
            it->SetAdvanceX(static_cast<u16>(mForceMonospacedWidth[it->GetFace()] * scaleWidth));
            it->SetLeftOffset(static_cast<s16>((mForceMonospacedWidth[it->GetFace()] - map->width) / 2 * scaleWidth));
        }
        else
        {
            it->SetAdvanceX(static_cast<u16>((map->idX + mBearingOffsetX[it->GetFace()]) * scaleWidth));
            it->SetLeftOffset(static_cast<s16>((map->loX + mBearingOffsetX[it->GetFace()] / 2) * scaleWidth));
        }
        it->SetBaseline(static_cast<s16>(font_ascent * scaleHeight + baselineOffset));

        const u32 copy_size = map->width + 2;
#if defined(NW_PLATFORM_CAFE)
        u32 cache_flush_size = map->width + 2 + 32;
#endif

        // 前の文字が描画されている場合があるので、文字の上のスペースをクリアする
        for (s32 l = it->GetCachePosY(); l < render_origin_y; l++) {
            // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
            u8* line_start = &mTextureCacheBitMap[(mTextureCacheHeight - l - 1) * mTextureCacheWidth + it->GetCachePosX()];
            std::memset(line_start, 0, copy_size);
#if defined(NW_PLATFORM_CAFE)
            // DCFlushRange(line_start, copy_size)と同等の処理
            for (u32 dot = 0; dot < cache_flush_size; dot += 32)
            {
                asm ( "dcbf		%0, %1" : "+g"(line_start), "+g"(dot) );
            }
#endif
        }

        const s32 copy_start_y = ut::Max(0, - render_origin_y);
        if (mBorderWidths[it->GetFace()])
        {
            // フチを含めた全領域を白で塗りつぶす
            for (s32 l = copy_start_y; l < map->height; l++) {
                // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
                u8* line_start = &mTextureCacheBitMap[(mTextureCacheHeight - (l + render_origin_y) - 1) * mTextureCacheWidth + it->GetCachePosX()];

                std::memcpy(line_start, &map->bits[l * map->width], map->width);
                // コピーした部分の右の2ピクセルは0でクリアする
                *reinterpret_cast<u16*>(&line_start[map->width]) = 0;
            }

            mFontEngine->ReleasesGlyph(map);
            mFontEngine->SetFlags(nn::fontll::ScalableFontEngine::Flags_OutlinedUnFilled);
            map = mFontEngine->AcquireGlyphmap(it->GetCode(), nn::fontll::FormatGrayMap8);
            AssertFsError("GetGlyphmap");
            mFontEngine->SetFlags(nn::fontll::ScalableFontEngine::Flags_OutlinedFilled);

            // フチの部分だけを黒くする
            for (s32 l = copy_start_y; l < map->height; l++) {
                // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
                u8* line_start = &mTextureCacheBitMap[(mTextureCacheHeight - (l + render_origin_y) - 1) * mTextureCacheWidth + it->GetCachePosX()];

                for (int x = 0; x < map->width; ++x)
                {
                    // フチ描画の場合は、値を半分未満にすることでシェーダ内でフチとして扱っている
                    auto b = (int)line_start[x] - (map->bits[l * map->width + x] >> 1);
                    line_start[x] = (b >= 0)? (u8)b : 0;
                }
#if defined(NW_PLATFORM_CAFE)
                // DCFlushRange(line_start, copy_size)と同等の処理
                for (u32 dot = 0; dot < cache_flush_size; dot += 32)
                {
                    asm ( "dcbf		%0, %1" : "+g"(line_start), "+g"(dot) );
                }
#endif
            }
        }
        else
        {
            for (s32 l = copy_start_y; l < map->height; l++) {
                // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
                u8* line_start = &mTextureCacheBitMap[(mTextureCacheHeight - (l + render_origin_y) - 1) * mTextureCacheWidth + it->GetCachePosX()];

                std::memcpy(line_start, &map->bits[l * map->width], map->width);
                // コピーした部分の右の2ピクセルは0でクリアする
                *reinterpret_cast<u16*>(&line_start[map->width]) = 0;
#if defined(NW_PLATFORM_CAFE)
                // DCFlushRange(line_start, copy_size)と同等の処理
                for (u32 dot = 0; dot < cache_flush_size; dot += 32)
                {
                    asm ( "dcbf		%0, %1" : "+g"(line_start), "+g"(dot) );
                }
#endif
            }
        }
        // heightがそのラインの高さより小さい場合は、最大2ドット0でクリアする
        const u32 line_height = 1 << it->GetLineKind();
        for (u32 l = map->height; l < static_cast<u32>(map->height + 2) && l < line_height; l++)
        {
            // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
            u8* line_start = &mTextureCacheBitMap[(mTextureCacheHeight - (l + render_origin_y) - 1) * mTextureCacheWidth + it->GetCachePosX()];
            std::memset(line_start, 0, copy_size);
#if defined(NW_PLATFORM_CAFE)
            // DCFlushRange(line_start, copy_size)と同等の処理
            for (u32 dot = 0; dot < cache_flush_size; dot += 32)
            {
                asm ( "dcbf		%0, %1" : "+g"(line_start), "+g"(dot) );
            }
#endif
        }

        mFontEngine->ReleasesGlyph(map);
        AssertFsError("ReleasesGlyph");
    }
}

TextureCache::LineInfo* TextureCache::CreateNewLineImpl(LineHeight lineHeight)
{
    u32 next_line_y;
    if (mLineCurrentPos == 0)
    {
        next_line_y = 0;
    }
    else
    {
        next_line_y = mLineInfos[mLineCurrentPos - 1].y + (1 << mLineInfos[mLineCurrentPos - 1].kind) + GLYPH_PADDING;
    }
    u32 endY = next_line_y + static_cast<u32>(1 << static_cast<u32>(lineHeight));

    if (endY > mTextureCacheHeight)
    {
        // 全てのラインが埋まっていて、新たにラインを作ることができない。
        return NULL;
    }
    else
    {
        // 新しいラインを作成できる
        LineInfo* new_line = &mLineInfos[mLineCurrentPos];
        new_line->currentX = 0;
        new_line->y = static_cast<u16>(next_line_y);
        new_line->kind = static_cast<u8>(lineHeight);
        new_line->no = static_cast<u8>(mLineCurrentPos);

        mLineCurrentPos++;
        return new_line;
    }
}

GlyphNode* TextureCache::FindAndReserveEraseGlyph(u8 lineKind, u16 glyphWidth)
{
    for (u32 l = 0; l < mLineCurrentPos; l++)
    {
        LineInfo* lineInfo = &mLineInfos[l];
        if (lineInfo->kind == lineKind)
        {
            // 種類が一致するラインを見つけた、消去可能な文字がないか探す
            GlyphLineList::Iterator it = lineInfo->list.GetBeginIter();
            GlyphLineList::Iterator begin_it = lineInfo->list.GetBeginIter();
            GlyphLineList::Iterator end_it = lineInfo->list.GetEndIter();
            GlyphLineList::Iterator mostleft_erasable_node;
            u32 erasable_num = 0;
            for (;it != end_it; ++it)
            {
                if (it->IsErasable())
                {
                    u32 erasable_width;
                    if (erasable_num == 0)
                    {
                        mostleft_erasable_node = it;
                        erasable_width = it->GetGlyphWidth();
                    }
                    else
                    {
                        erasable_width = it->GetGlyphWidth() + it->GetCachePosX() - mostleft_erasable_node->GetCachePosX();
                    }
                    s32 mostleft_move_val = 0;
                    if (mostleft_erasable_node != begin_it)
                    {
                        // mostleft_erasable_nodeが最初のノードでなければ、左側の空きスペースを使うことができる
                        GlyphLineList::Iterator mostleft_prev_it = mostleft_erasable_node;
                        do {
                            --mostleft_prev_it;
                        } while (mostleft_prev_it->IsErased() && begin_it != mostleft_prev_it);

                        mostleft_move_val = mostleft_erasable_node->GetCachePosX() - (mostleft_prev_it->GetCachePosX() + mostleft_prev_it->GetGlyphWidth() + GLYPH_PADDING);
                        if (mostleft_move_val > 0)
                        {
                            erasable_width += mostleft_move_val;
                        }
                    }
                    if (erasable_width >= glyphWidth)
                    {
                        // 消去可能な幅が必要な文字幅を超えた
                        // もう一度消去されることがないように消去フラグを立て、消去予約済みリストに入れる
                        GlyphLineList::Iterator erasable_it = mostleft_erasable_node;
                        GlyphLineList::Iterator erasable_end_it = it;
                        ++erasable_end_it;
                        for (; erasable_it != erasable_end_it; ++erasable_it)
                        {
                            if ( ! erasable_it->IsErased())
                            {
                                erasable_it->SetErased(true);
                                mNeedEraseGlyphList.PushBack(&(*erasable_it));
                            }
                        }
                        if (mostleft_move_val > 0)
                        {
                            // mostleft_erasable_nodeの左側の空きスペースを使っている場合はCachePosXを動かす
                            mostleft_erasable_node->SetCachePosX(static_cast<u16>(mostleft_erasable_node->GetCachePosX() - mostleft_move_val));
                        }
                        return &(*mostleft_erasable_node);
                    }
                    erasable_num += 1;
                }
                else
                {
                    erasable_num = 0;
                }
            }
        }
    }
    return NULL;
}

void TextureCache::CompleteTextureCache()
{
#if defined(NW_PLATFORM_WIN32)
    // .NET ではテクスチャは利用しません
    //glBindTexture(GL_TEXTURE_2D_ARRAY, m_TexObj.GetName());
    //glTexImage3D(GL_TEXTURE_2D_ARRAY, 0, GL_ALPHA8, mTextureCacheWidth, mTextureCacheHeight, 1, 0, GL_ALPHA, GL_UNSIGNED_BYTE, mTextureCacheBitMap);
    //NW_GL_ASSERT();
#endif

    // 消去予約済みリストに入っているノードを破棄する
    // 消去予約済みリストに入るのは、新しい文字を表示する必要があって削除する場合と、
    // プロットすることができなかったために削除する場合の二通りある。
    // 前者はIsErasedがtrueになり、後者はfalseになるので区別できる。後者はラインの
    // リストから削除する必要がない。
    GlyphList::Iterator step_it = mNeedEraseGlyphList.GetBeginIter();
    GlyphList::Iterator end_it = mNeedEraseGlyphList.GetEndIter();
    for (;step_it != end_it;)
    {
        GlyphList::Iterator it = step_it;
        ++step_it;
        mNeedEraseGlyphList.Erase(it);
        if (it->IsErased())
        {
            GlyphLineList& list = mLineInfos[it->GetLineNo()].list;
            list.Erase(&(*it));
            // グリフが消えたことによりラインの右側が空く可能性があるので、currentXを設定する
            if (list.GetSize() > 0)
            {
                mLineInfos[it->GetLineNo()].currentX = list.GetBack().GetCachePosX() + list.GetBack().GetGlyphWidth() + GLYPH_PADDING;
            }
            else
            {
                mLineInfos[it->GetLineNo()].currentX = 0;
            }
        }
        mGlyphTreeMap.Erase(it->GetCode(), it->GetSize(), it->GetFace());
    }

    mGlyphTreeMap.UpdateFlagsForCompleteTextureCache();
}

void TextureCache::ClearLockAllGlyphs(s32 lockGroup)
{
    mGlyphTreeMap.ClearLockGroup(1 << lockGroup);
}

void TextureCache::ClearLockGlyphsWithLength(const char16* codes, u32 codeLength, u32 fontSize, u16 fontFace, s32 lockGroup)
{
    u32 groupLockMask = 1 << lockGroup;

    for (u32 i = 0; i < codeLength; i++)
    {
        if (codes[i] == 0)
        {
            break;
        }

        u32 innerFontFace;
        if (!GetInnerFontFace(&innerFontFace, static_cast<u32>(fontFace), codes[i]))
        {
            break;
        }

        GlyphNode* node = mGlyphTreeMap.Find(codes[i], static_cast<u16>(fontSize), static_cast<u16>(innerFontFace));
        if (node)
        {
            node->GetLockGroup().SetMaskOff(groupLockMask);
        }
    }
}

void TextureCache::ResetTextureCache()
{
#if defined(NW_PLATFORM_WIN32)
    std::memset(mTextureCacheBitMap, 0x0, mTextureCacheWidth * mTextureCacheHeight);
#elif defined(NW_PLATFORM_CAFE)
    // テクスチャキャッシュを更新するときにキャッシュストアされるので、この時点ではDCZeroRangeでよい
    DCZeroRange(mTextureCacheBitMap, mTextureCacheWidth * mTextureCacheHeight);
#endif
    mNeedPlotGlyphList.Clear();
    mNeedEraseGlyphList.Clear();
    mNotInFontGlyphList.Clear();

    // ラインをリセット
    for (u32 i = 0; i < mLineCurrentPos; i++)
    {
        mLineInfos[i].list.Clear();
    }
    mLineCurrentPos = 0;

    // 平衡二分木をリセット
    mGlyphTreeMap.Reset();
}

bool TextureCache::IsBorderEffectEnabled(u16 fontFace) const
{
    // 1 つでも BorderWidth が設定されていれば true を返す
    NW_ASSERT(fontFace < mFontCount);
    u32 innerFontFace = static_cast<u32>(mFontFaceTable[fontFace]);
    for (u32 i = 0; i < static_cast<u32>(mFontListCount[fontFace]); i++)
    {
        if (0 < mBorderWidths[innerFontFace + i])
        {
            return true;
        }
    }
    return false;
}

bool TextureCache::IsGlyphExistInFont(char16 code, u16 fontFace)
{
    u32 innerFontFace;
    return GetInnerFontFace(&innerFontFace, static_cast<u32>(fontFace), code);
}

const TextureCache::FontMetrics& TextureCache::GetFontMetrics(u16 fontFace)
{
    // 高さが最大のものを返す。
    NW_ASSERT(fontFace < mFontCount);
    u32 innerFontFace = static_cast<u32>(mFontFaceTable[fontFace]);
    float maxHeight = mFontMetrics[innerFontFace].height_ratio;
    u32 result = innerFontFace;
    for (u32 i = 1; i < static_cast<u32>(mFontListCount[fontFace]); i++)
    {
        float height = mFontMetrics[innerFontFace + i].height_ratio;
        if (maxHeight < height)
        {
            result = innerFontFace + i;
            maxHeight = height;
}
    }
    return mFontMetrics[result];
}

int TextureCache::CalcCharWidth(char16 code, u32 fontSize, u16 fontFace)
{
    u32 innerFontFace;
    if (!GetInnerFontFace(&innerFontFace, static_cast<u32>(fontFace), code))
    {
        return 0;
    }
    s16 i_dx, i_dy;
    nn::fontll::ScFixed dx, dy;
    u32 coreId = GetCoreId();
    SetFontFaceNoPlot(innerFontFace, coreId);
    mFontEngineNoPlot[coreId].SetScale(fontSize << 16, 0, 0, fontSize << 16);
    AssertFsErrorNoPlot("SetScale", coreId);
    if (mFontEngineNoPlot[coreId].GetAdvance(&i_dx, &i_dy, &dx, &dy, code, nn::fontll::FormatGrayMap8) == nn::fontll::Success)
    {
        return i_dx;
    }
    else
    {
        return 0;
    }
}

int TextureCache::CalcKerning(char16 c0, char16 c1, u32 fontSize, u16 fontFace)
{
    if (c0 == 0)
    {
        // 先頭の文字のカーニング処理
        u32 innerFontFace;
        if (!GetInnerFontFace(&innerFontFace, static_cast<u32>(fontFace), c1))
        {
            return 0;
        }
        if (mDeleteBearingX[innerFontFace] || mForceMonospacedEnabled[innerFontFace])
        {
            return 0;
        }

        u32 coreId = GetCoreId();
        SetFontFaceNoPlot(innerFontFace, coreId);
        mFontEngineNoPlot[coreId].SetScale(fontSize << 16, 0, 0, fontSize << 16);
        AssertFsErrorNoPlot("SetScale", coreId);

        if (mOtfKerningTable[innerFontFace] != NULL)
        {
            const float scaleWidth = mFontMetrics[innerFontFace].scaleWidth;
            return static_cast<int>(mFontEngineNoPlot[coreId].AcquireOtfKerningFirst(mOtfKerningTable[innerFontFace], c1, fontSize) * scaleWidth);
        }
    }
    else if (c1 == 0)
    {
        // 末尾の文字のカーニング処理
        u32 innerFontFace;
        if (!GetInnerFontFace(&innerFontFace, static_cast<u32>(fontFace), c0))
        {
            return 0;
        }
        if (mDeleteBearingX[innerFontFace] || mForceMonospacedEnabled[innerFontFace])
        {
            return 0;
        }

        u32 coreId = GetCoreId();
        SetFontFaceNoPlot(innerFontFace, coreId);
        mFontEngineNoPlot[coreId].SetScale(fontSize << 16, 0, 0, fontSize << 16);
        AssertFsErrorNoPlot("SetScale", coreId);

        if (mOtfKerningTable[innerFontFace] != NULL)
        {
            const float scaleWidth = mFontMetrics[innerFontFace].scaleWidth;
            return static_cast<int>(mFontEngineNoPlot[coreId].AcquireOtfKerningLast(mOtfKerningTable[innerFontFace], c0, fontSize) * scaleWidth);
        }
    }
    else
    {
        // 通常のカーニング処理
        u32 innerFontFace0;
        u32 innerFontFace1;
        if (!GetInnerFontFace(&innerFontFace0, static_cast<u32>(fontFace), c0))
        {
            return 0;
        }
        if (!GetInnerFontFace(&innerFontFace1, static_cast<u32>(fontFace), c1))
        {
            return 0;
        }

        if (innerFontFace0 != innerFontFace1)
        {
            return 0;
        }
        if (mDeleteBearingX[innerFontFace0] || mForceMonospacedEnabled[innerFontFace0])
        {
            return 0;
        }

        nn::fontll::ScFixed dx, dy;
        u32 coreId = GetCoreId();
        SetFontFaceNoPlot(innerFontFace0, coreId);
        mFontEngineNoPlot[coreId].SetScale(fontSize << 16, 0, 0, fontSize << 16);
        AssertFsErrorNoPlot("SetScale", coreId);

        // OTF 用のカーニングテーブルが存在していればそちらから取得する
        if (mOtfKerningTable[innerFontFace0] != NULL)
        {
            int kerning = mFontEngineNoPlot[coreId].AcquireOtfKerning(mOtfKerningTable[innerFontFace0], c0, c1, fontSize);
            if (kerning != 0)
            {
                const float scaleWidth = mFontMetrics[innerFontFace0].scaleWidth;
                return static_cast<int>(kerning * scaleWidth);
            }
        }

        // OTF 用のカーニングに存在しなければペアカーニングから取得する。
        if (mFontEngineNoPlot[coreId].GetKerning(&dx, &dy, c0, c1) == nn::fontll::Success)
        {
            const float scaleWidth = mFontMetrics[innerFontFace0].scaleWidth;
            return static_cast<int>(dx * scaleWidth) >> 16;
        }
    }

    // カーニング情報が存在しなければ 0 とする
    return 0;
}

bool TextureCache::HasKerning()
{
    const u32 coreId = GetCoreId();

    // 内部に格納されているすべてのフォントを操作します。
    int innerFontIdx = 0;
    for (uint32_t i = 0; i < mFontCount; i++)
    {
        // 各フォント中の、内部フォントの数を取得します。
        const uint32_t fontListCount = static_cast<uint32_t>(mFontListCount[i]);
        for (uint32_t j = 0; j < fontListCount; j++)
        {
            // OTF カーニングの有無をチェック
            if (mOtfKerningTable[innerFontIdx] != NULL)
            {
                return true;
            }

            // TTF カーニングの有無をチェック
            SetFontFaceNoPlot(innerFontIdx, coreId);
            if (mFontEngineNoPlot[coreId].CheckTtfKernTableExist())
            {
                return true;
            }
            innerFontIdx++;
        }
    }

    return false;
}

GlyphNode* TextureCache::FindGlyphNode(char16 code, u32 fontSize, u16 fontFace)
{
    u32 innerFontFace;
    if (!GetInnerFontFace(&innerFontFace, static_cast<u32>(fontFace), code))
    {
        return NULL;
    }
    return mGlyphTreeMap.Find(code, static_cast<u16>(fontSize), static_cast<u16>(innerFontFace));
}


void TextureCache::SetFontFace(u32 innerFontFace)
{
    if (innerFontFace != mCurrentFontFace)
    {
        // 今から使用するフォントの設定
        // 指定する文字のエンコーディングを設定
        mFontEngine->SetFont(GetFontNameBuf(innerFontFace));
        AssertFsError("SetFont");

        // ボールドのウェイトの設定
        mFontEngine->SetBoldWeight(mBoldWeights[innerFontFace]);
        AssertFsError("SetBoldWeight");

        // 縁取りの設定
        if (mBorderWidths[innerFontFace])
        {
            mFontEngine->SetOutlineWidth(mBorderWidths[innerFontFace]);
            mFontEngine->SetFlags(nn::fontll::ScalableFontEngine::Flags_OutlinedFilled);
        }
        else
        {
            mFontEngine->SetFlags(nn::fontll::ScalableFontEngine::Flags_NoEffect);
        }
        AssertFsError("SetFlags");

        mCurrentFontFace = innerFontFace;
    }
}

void TextureCache::SetFontFaceNoPlot(u32 innerFontFace, u32 coreId)
{
    if (innerFontFace != mCurrentFontFacesNoPlot[coreId])
    {
        // 今から使用するフォントの設定
        // 指定する文字のエンコーディングを設定
        mFontEngineNoPlot[coreId].SetFont(GetFontNameBuf(innerFontFace));
        AssertFsErrorNoPlot("SetFont", coreId);

        // ボールドのウェイトの設定
        mFontEngineNoPlot[coreId].SetBoldWeight(mBoldWeights[innerFontFace]);
        AssertFsErrorNoPlot("SetBoldWeight", coreId);

        // 縁取りの設定
        if (mBorderWidths[innerFontFace])
        {
            mFontEngineNoPlot[coreId].SetOutlineWidth(mBorderWidths[innerFontFace]);
            mFontEngineNoPlot[coreId].SetFlags(nn::fontll::ScalableFontEngine::Flags_OutlinedFilled);
        }
        else
        {
            mFontEngineNoPlot[coreId].SetFlags(nn::fontll::ScalableFontEngine::Flags_NoEffect);
        }
        AssertFsErrorNoPlot("SetFlags", coreId);

        mCurrentFontFacesNoPlot[coreId] = innerFontFace;
    }
}

void TextureCache::AssertFsError(const char* api_name)
{
    NW_UNUSED_VARIABLE(api_name);
    NW_ASSERTMSG(mFontEngine->GetError() == nn::fontll::Success, "%s error: %d", api_name, mFontEngine->GetError());
}

void TextureCache::AssertFsErrorNoPlot(const char* api_name, u32 coreId)
{
    NW_UNUSED_VARIABLE(api_name);
    NW_ASSERTMSG(mFontEngineNoPlot[coreId].GetError() == nn::fontll::Success, "%s error: %d", api_name, mFontEngineNoPlot[coreId].GetError());
}

u32 TextureCache::EraseNotInFontGlyphs()
{
    u32 count = 0;
    GlyphList::Iterator step_it = mNotInFontGlyphList.GetBeginIter();
    GlyphList::Iterator end_it = mNotInFontGlyphList.GetEndIter();
    for (;step_it != end_it;)
    {
        GlyphList::Iterator it = step_it;
        ++step_it;
        // 現在使われている文字については削除しない
        if ( ! it->IsRequestedOrKeeped()) {
            mNotInFontGlyphList.Erase(it);
            mGlyphTreeMap.Erase(it->GetCode(), it->GetSize(), it->GetFace());
            count++;
        }
    }
    return count;
}

int TextureCache::InitializeFontFaceTable(const InitializeArg& arg)
{
    // FontFace の総数を求める
    int fontFaceCount = 0;
    {
        for (u32 i = 0; i < arg.fontCount; i++)
        {
            if (arg.fontListCount[i] == 0 || arg.fontListCount[i] > 32)
            {
                NW_ERR("arg.fontListCount must be 0 and more, 32 and less.");
                return 0;
            }
            fontFaceCount += arg.fontListCount[i];
        }
        if (fontFaceCount <= 0 || fontFaceCount > 32)
        {
            NW_ERR("The sum of arg.fontListCount must be 0 and more, 32 and less.");
            return 0;
        }
    }

    {
        // 外部用 FontFace と内部用 FontFace をマッピングするテーブルの作成
        u8 idx = 0;
        for (int i = 0; i < FONT_FACE_COUNT_MAX; i++)
        {
            if (static_cast<u32>(i) >= arg.fontCount)
            {
                mFontListCount[i] = 0;
                mFontFaceTable[i] = idx;
                continue;
            }
            mFontListCount[i] = static_cast<u8>(arg.fontListCount[i]);
            mFontFaceTable[i] = idx;
            for (u32 j = 0; j < arg.fontListCount[i]; j++)
            {
                idx++;
            }
        }
    }
    return fontFaceCount;
}

bool TextureCache::GetInnerFontFace(u32* innerFontFace, u32 outerFontFace, u16 code)
{
    NW_ASSERT(outerFontFace < mFontCount);
    *innerFontFace = static_cast<u32>(mFontFaceTable[outerFontFace]);
    for (int i = 0; i < static_cast<int>(mFontListCount[outerFontFace]); i++)
    {
        if (CheckCharCodeRange(*innerFontFace, code))
        {
            // CharCodeRange の中に該当する場合はグリフが存在するかを判別
            u32 coreId = GetCoreId();
            SetFontFaceNoPlot(*innerFontFace, coreId);
            bool existing = mFontEngineNoPlot[coreId].CheckGlyphExist(code);
            if (existing)
            {
                return true;
            }
        }

        (*innerFontFace)++;
    }
    return false;
}

bool TextureCache::CheckCharCodeRange(u32 fontFace, u32 code)
{
    int charCodeRangeCount = mCharCodeRangeCount[fontFace];
    if (charCodeRangeCount == 0)
    {
        // CharCodeRange の数が 0 の場合はすべての文字コードを使用
        return true;
    }

    for (int j = 0; j < charCodeRangeCount; j++)
    {
        if (mCharCodeRangeFirst[fontFace][j] <= code && code <= mCharCodeRangeLast[fontFace][j])
        {
            return true;
        }
    }
    return false;
}

} // namespace nw::scfont
} // namespace nw

#endif NW_LAYOUTEDITOR_CAFE
