﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifdef NW_LAYOUTEDITOR_CAFE
#include <scfont_GlyphTreeMap.h>

namespace nw
{
namespace scfont
{

u8 GlyphNode::CalcLineKind(u16 size)
{
    if (size > 1024)
    {
        NW_LOG("The font height(%d) is too large.", size);
        return 0;
    }
    else if (size > 512)
    {
        return 10;
    }
    else if (size > 256)
    {
        return 9;
    }
    else if (size > 128)
    {
        return 8;
    }
    else if (size > 64)
    {
        return 7;
    }
    else if (size > 32)
    {
        return 6;
    }
    else if (size > 16)
    {
        return 5;
    }
    else
    {
        return 4;
    }
}

GlyphTreeMap::GlyphTreeMap()
 : mRoot(NULL)
 , mNodeNumMax(0)
{
}

void GlyphTreeMap::Initialize(IAllocator* allocator, u32 nodeNumMax)
{
    mNodeNumMax = nodeNumMax;
    mFreeList.Init(allocator->Alloc(sizeof(GlyphNode) * nodeNumMax, IAllocator::DEFAULT_ALIGNMENT), sizeof(GlyphNode), nodeNumMax);
}

void GlyphTreeMap::Finalize(IAllocator* allocator)
{
    allocator->Free(mFreeList.GetWork());
    mFreeList.Cleanup();
}

GlyphNode* GlyphTreeMap::Find(char16 code, u16 size, u16 face) const
{
    GlyphNode::KeyType key;
    key.detail.code = code;
    key.detail.size = size + (face << 11);
    return Find(mRoot, key.raw);
}

GlyphNode* GlyphTreeMap::Insert(char16 code, u16 size, u16 face)
{
    if (mFreeList.IsEmpty())
    {
        return NULL;
    }
    else
    {
        GlyphNode* node = new(mFreeList.Get()) GlyphNode(code, size, face);
        mRoot = Insert(mRoot, node);
        mRoot->mColor = GlyphNode::BLACK;
        return node;
    }
}

void GlyphTreeMap::Erase(char16 code, u16 size, u16 face)
{
    GlyphNode::KeyType key;
    key.detail.code = code;
    key.detail.size = size + (face << 11);
    mRoot = Erase(mRoot, key.raw);
    if (mRoot) mRoot->mColor = GlyphNode::BLACK;
}

GlyphNode* GlyphTreeMap::Find(GlyphNode* node, u32 key_raw) const
{
    while (node)
    {
        if (key_raw < node->mKey.raw) {
            node = node->mLeft;
        } else if (key_raw > node->mKey.raw) {
            node = node->mRight;
        } else {
            return node;
        }
    }
    return NULL;
}

GlyphNode* GlyphTreeMap::Insert(GlyphNode* h, GlyphNode* node)
{
    if (h == NULL) {
        node->mColor = GlyphNode::RED;
        node->mLeft = NULL;
        node->mRight = NULL;
        return node;
    }
    {
        if (node->mKey.raw < h->mKey.raw) {
            h->mLeft = Insert(h->mLeft, node);
        } else if (node->mKey.raw > h->mKey.raw) {
            h->mRight = Insert(h->mRight, node);
        } else {
            // 置き換わり
            if (h != node) {
                // 全く同じオブジェクトの場合はこの処理は行う必要がない
                node->mRight = h->mRight;
                node->mLeft = h->mLeft;
                node->mColor = h->mColor;
                mFreeList.Put(h);
            }
            h = node;
        }
        if (IsRed(h->mRight) && ! IsRed(h->mLeft)) h = RotateLeft(h);
        if (IsRed(h->mLeft) && IsRed(h->mLeft->mLeft)) h = RotateRight(h);
        if (IsRed(h->mLeft) && IsRed(h->mRight)) FlipColors(h);
    }
    return h;
}

GlyphNode* GlyphTreeMap::Erase(GlyphNode* h, u32 key_raw)
{
    if (key_raw < h->mKey.raw) {
        if ( ! IsRed(h->mLeft) && ! IsRed(h->mLeft->mLeft)) {
            h = MoveRedLeft(h);
        }
        h->mLeft = Erase(h->mLeft, key_raw);
    } else {
        if (IsRed(h->mLeft)) {
            h = RotateRight(h);
        }
        if (key_raw == h->mKey.raw && (h->mRight == NULL)) {
            mFreeList.Put(h);
            return NULL;
        }
        if ( ! IsRed(h->mRight) && ! IsRed(h->mRight->mLeft)) {
            h = MoveRedRight(h);
        }
        if (key_raw == h->mKey.raw) {
            GlyphNode* newh = Find(h->mRight, GetMin(h->mRight)->mKey.raw);
            newh->mRight = EraseMin(h->mRight);
            newh->mLeft = h->mLeft;
            newh->mColor = h->mColor;
            mFreeList.Put(h);
            h = newh;
        } else {
            h->mRight = Erase(h->mRight, key_raw);
        }
    }
    return FixUp(h);
}

void GlyphTreeMap::Dump()
{
#if ! defined(NW_RELEASE)
    u32 nodeNum = 0;
    if (mRoot)
    {
        nodeNum = DumpRecursive(mRoot, 0);
    }
    NW_LOG("dump %d nodes.\n", nodeNum);
#endif
}

void GlyphTreeMap::Reset()
{
    mRoot = NULL;
    void* work = mFreeList.GetWork();
    mFreeList.Init(work, sizeof(GlyphNode), mNodeNumMax);
}

void GlyphTreeMap::UpdateFlagsForCompleteTextureCacheRecursive(GlyphNode* node)
{
    u32 flag = node->mFlag.GetDirect();
    u32 isKeep = static_cast<u32>((flag & (GlyphNode::FLAG_MASK_REQUESTED | GlyphNode::FLAG_MASK_USED)) != 0);
    node->mFlag.SetDirect(static_cast<u8>((isKeep << GlyphNode::FLAG_BIT_KEEPED) | (flag & (GlyphNode::FLAG_MASK_SYSTEM_RESERVED | GlyphNode::FLAG_MASK_NOT_IN_FONT))));
    if (node->mLeft)
    {
        UpdateFlagsForCompleteTextureCacheRecursive(node->mLeft);
    }
    if (node->mRight)
    {
        UpdateFlagsForCompleteTextureCacheRecursive(node->mRight);
    }
}

void GlyphTreeMap::ClearLockGroupRecursive(GlyphNode* node, u32 groupMask)
{
    node->GetLockGroup().SetMaskOff(groupMask);
    if (node->mLeft)
    {
        ClearLockGroupRecursive(node->mLeft, groupMask);
    }
    if (node->mRight)
    {
        ClearLockGroupRecursive(node->mRight, groupMask);
    }
}

u32 GlyphTreeMap::DumpRecursive(GlyphNode* node, u32 level)
{
    u32 nodeNum = 1;
    for (u32 i = 0; i < level; i++)
    {
        NW_LOG("  ");
    }
    NW_LOG("%d 0x%x %d %s%s%s%s%s%s%s", node->GetFace(), node->GetCode(), node->GetSize(),
        node->IsRequested() ? "r" : "",
        node->IsPlotting() ? "p" : "",
        node->IsUsed() ? "u" : "",
        node->IsErased() ? "e" : "",
        node->IsKeeped() ? "k" : "",
        node->IsSystemReserved() ? "s" : "",
        node->IsNotInFont() ? "n" : ""
    );
    if (node->GetLockGroup() != 0)
    {
        NW_LOG(" lock:0x%x\n", node->GetLockGroup().GetDirect());
    }
    else
    {
        NW_LOG("\n");
    }
    if (node->mLeft)
    {
        nodeNum += DumpRecursive(node->mLeft, level + 1);
    }
    if (node->mRight)
    {
        nodeNum += DumpRecursive(node->mRight, level + 1);
    }
    return nodeNum;
}

} // namespace nw::scfont
} // namespace nw

#endif // NW_LAYOUTEDITOR_CAFE
