﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 *  小物なユーティリティインライン関数を定義します。
 */

#ifndef NW_UT_INLINES_H_
#define NW_UT_INLINES_H_

#include <nw/types.h>
#include <nw/config.h>
#include <cstring>
#include <cstdio>
#include <cstdarg>
#include <limits>
#include <functional>
#include <algorithm>

#if defined( NW_COMPILER_MSVC )
#include <iosfwd>
#else
#include <string>
#endif

//! x を base の倍数に切り上げるマクロです。
//! このマクロでは、base が二回評価されますので、base は副作用のない式としてください。
#define NW_UT_ROUNDUP(x, base) (((x) + ((base)-1)) & ~((base)-1))

//! x を base の倍数に切り下げるマクロです。
//! このマクロでは、base が二回評価されますので、base は副作用のない式としてください。
#define NW_UT_ROUNDDOWN(x, base) ((x) & ~((base)-1))

//! x を y で割って端数を切り上げるマクロです。
//! このマクロでは、y が二回評価されますので、y は副作用のない式としてください。
#define NW_UT_DIV_UP(x, y) (((x) + ((y) - 1)) / (y))


namespace nw {
namespace ut {

//----------------------------------------
//! @name 数値関連
//@{

//---------------------------------------------------------------------------
//! @brief       引数の絶対値を返します。
//!
//! @param[in]   a
//!
//! @return      a の絶対値
//---------------------------------------------------------------------------
template <typename T>
NW_INLINE T
Abs(T a)
{
    return (a < 0) ? static_cast<T>(-a) : a;
}

//---------------------------------------------------------------------------
//! @brief       引数の絶対値を返します。
//!
//! @details :private
//!
//! @param[in]   a
//!
//! @return      a の絶対値
//---------------------------------------------------------------------------
template<>
NW_INLINE float
Abs<float>(register float a)
{
#if defined(GEKKO) && defined(NW_COMPILER_CWCC)
    register float ret;
    asm{ fabs   ret, a }
    return ret;
#else
    return (a < 0) ? -a : a;
#endif
}


//---------------------------------------------------------------------------
//! @brief       2つの数値を比較して小さい方の値を返します。
//!
//! @param[in]   a, b    比較する数値
//!
//! @return      a, b のうち小さい方の値を返します。
//---------------------------------------------------------------------------
template <typename T>
NW_INLINE T
Min(T a, T b)
{
    return (a > b) ? b: a;
}



//---------------------------------------------------------------------------
//! @brief       2つの数値を比較して大きい方の値を返します。
//!
//! @param[in]   a, b    比較する数値
//!
//! @return      a, b のうち大きい方の値を返します。
//---------------------------------------------------------------------------
template <typename T>
NW_INLINE T
Max(T a, T b)
{
    return (a < b) ? b: a;
}


//---------------------------------------------------------------------------
//! @brief       low から high の範囲に収まる値を返します。
//!
//! @param[in]   x       比較する数値
//! @param[in]   low     最小値
//! @param[in]   high    最大値
//!
//! @return      x < low ならば low, x > high ならば high それ以外は x を返します。
//---------------------------------------------------------------------------
template <typename T>
NW_INLINE T
Clamp(T x, T low, T high)
{
    return (x > high) ? high : ( (x < low) ? low : x );
}

//---------------------------------------------------------------------------
//! @brief       x を base の倍数に切り上げます。
//!
//! @param[in]   x       切り上げる数。
//! @param[in]   base    切り上げの基準となる値。
//!                      2の累乗でなければなりません。
//!
//! @return      x を base の倍数に切り上げた値を返します。
//---------------------------------------------------------------------------
template <typename ValueT>
NW_INLINE ValueT
RoundUp(ValueT x, std::size_t base)
{
    std::size_t mask = base - 1;
    return static_cast<ValueT>(static_cast<ValueT>( x + mask ) & ~mask);
}

//---------------------------------------------------------------------------
//! @brief       x を base の倍数に切り上げます。(ポインタ用の特殊化)
//---------------------------------------------------------------------------
template <typename ValueT>
NW_INLINE void*
RoundUp(ValueT* x, std::size_t base)
{
    IntPtr value = GetIntPtr( x );
    return reinterpret_cast<void*>( RoundUp(value, base) );
}

//---------------------------------------------------------------------------
//! @brief       x を base の倍数に切り上げます。(const ポインタ用の特殊化)
//---------------------------------------------------------------------------
template <typename ValueT>
NW_INLINE const void*
RoundUp(const ValueT* x, std::size_t base)
{
    IntPtr value = GetIntPtr( x );
    return reinterpret_cast<const void*>( RoundUp(value, base) );
}

//---------------------------------------------------------------------------
//! @brief        ポインタを base の倍数に切り上げ、指定した型のポインタにキャストします。
//!
//! @tparam       TPtr    キャストする型です。
//! @param[out]   x       切り上げるポインタです。
//! @param[in]    base    切り上げの基準となる値です。
//!                       2の累乗でなければなりません。
//!
//! @return       x を base の倍数に切り上げた値を TPtr にキャストして返します。
//---------------------------------------------------------------------------
template <typename TPtr>
NW_INLINE TPtr
RoundUpTo(void* x, std::size_t base)
{
    IntPtr value = GetIntPtr( x );
    void* rounded = reinterpret_cast<void*>( RoundUp(value, base) );

    return static_cast<TPtr>(rounded);
}

//---------------------------------------------------------------------------
//! @brief        ポインタを base の倍数に切り上げ、指定した型のポインタにキャストします。(const 版)
//!
//! @tparam       TPtr    キャストする型です。
//! @param[out]   x       切り上げるポインタです。
//! @param[in]    base    切り上げの基準となる値です。
//!                       2の累乗でなければなりません。
//!
//! @return       x を base の倍数に切り上げた値を TPtr にキャストして返します。
//---------------------------------------------------------------------------
template <typename TPtr>
NW_INLINE TPtr
RoundUpTo(const void* x, std::size_t base)
{
    IntPtr value = GetIntPtr( x );
    const void* rounded = reinterpret_cast<const void*>( RoundUp(value, base) );

    return static_cast<TPtr>(rounded);
}


//---------------------------------------------------------------------------
//! @brief       x を base の倍数に切り下げます。
//!
//! @param[in]   x       切り下げる数。
//! @param[in]   base    切り下げの基準となる値。2の累乗でなければなりません。
//!
//! @return      x を base の倍数に切り下げた値を返します。
//---------------------------------------------------------------------------
template <typename ValueT>
NW_INLINE ValueT
RoundDown(ValueT x, std::size_t base)
{
    return static_cast<ValueT>( x & ~(base - 1) );
}

//---------------------------------------------------------------------------
//! @brief       x を base の倍数に切り下げます。(ポインタ用特殊化)
//---------------------------------------------------------------------------
template <typename ValueT>
NW_INLINE void*
RoundDown(ValueT* x, std::size_t base)
{
    IntPtr value = GetIntPtr( x );
    return reinterpret_cast<void*>( RoundDown(value, base) );
}

//---------------------------------------------------------------------------
//! @brief       x を base の倍数に切り下げます。(const ポインタ用特殊化)
//---------------------------------------------------------------------------
template <typename ValueT>
NW_INLINE const void*
RoundDown(const ValueT* x, std::size_t base)
{
    IntPtr value = GetIntPtr( x );
    return reinterpret_cast<const void*>( RoundDown(value, base) );
}


//---------------------------------------------------------------------------
//! @brief        ポインタを base の倍数に切り下げ、指定した型のポインタにキャストします。
//!
//! @tparam       TPtr    キャストする型です。
//! @param[out]   x       切り下げるポインタです。
//! @param[in]    base    切り下げの基準となる値です。
//!                       2の累乗でなければなりません。
//!
//! @return       x を base の倍数に切り下げた値を TPtr にキャストして返します。
//---------------------------------------------------------------------------
template <typename TPtr>
NW_INLINE TPtr
RoundDownTo(void* x, std::size_t base)
{
    IntPtr value = GetIntPtr( x );
    void* rounded = reinterpret_cast<void*>( RoundDown(value, base) );

    return static_cast<TPtr>(rounded);
}

//---------------------------------------------------------------------------
//! @brief        const ポインタを base の倍数に切り下げ、指定した型のポインタにキャストします。
//---------------------------------------------------------------------------
template <typename TPtr>
NW_INLINE TPtr
RoundDownTo(const void* x, std::size_t base)
{
    IntPtr value = GetIntPtr( x );
    const void* rounded = reinterpret_cast<const void*>( RoundDown(value, base) );

    return static_cast<TPtr>(rounded);
}

//---------------------------------------------------------------------------
//! @brief       x を y で割って端数を切り上げます。
//!
//! @param[in]   x   割られる数
//! @param[in]   y   割る数
//!
//! @return      x を y で割って端数を切り上げた値を返します。
//---------------------------------------------------------------------------
template <typename ValueT>
NW_INLINE ValueT
DivUp(ValueT x, int y)
{
    return (x + (static_cast<ValueT>(y) - 1)) / static_cast<ValueT>(y);
}

//---------------------------------------------------------------------------
//! @brief       ビット列の一部を抜き出します。
//!
//! @param[in]   v   ビット列
//! @param[in]   pos 抜き出す最下位ビット位置
//! @param[in]   len 抜き出す長さ
//!
//! @return      抜き出されたビット列を返します。
//---------------------------------------------------------------------------
template <typename T>
NW_INLINE T
ExtractBits(bit32 v, int pos, int len)
{
    return static_cast<T>( v & (((1u << len) - 1) << pos) );
}

//---------------------------------------------------------------------------
//! @brief       ビット列の一部を抜き出します。
//!
//! @param[in]   v   ビット列
//! @param[in]   pos 抜き出す最下位ビット位置
//! @param[in]   len 抜き出す長さ
//!
//! @return      抜き出されたビット列を返します。
//---------------------------------------------------------------------------
template <typename T>
NW_INLINE T
ExtractBits(bit64 v, int pos, int len)
{
    return static_cast<T>( v & (((1ull << len) - 1) << pos) );
}

//---------------------------------------------------------------------------
//! @brief       ビット列の一部を取得します。
//!
//! @param[in]   v   ビット列
//! @param[in]   pos 取得する最下位ビット位置
//! @param[in]   len 取得する長さ
//!
//! @return      取得したビット列を返します。
//---------------------------------------------------------------------------
template <typename T>
NW_INLINE T
GetBits(bit32 v, int pos, int len)
{
    return static_cast<T>( (v >> pos) & ((1u << len) - 1) );
}

//---------------------------------------------------------------------------
//! @brief       ビット列の一部を取得します。
//!
//! @param[in]   v   ビット列
//! @param[in]   pos 取得する最下位ビット位置
//! @param[in]   len 取得する長さ
//!
//! @return      取得したビット列を返します。
//---------------------------------------------------------------------------
template <typename T>
NW_INLINE T
GetBits(bit64 v, int pos, int len)
{
    return static_cast<T>( (v >> pos) & ((1ull << len) - 1) );
}

//---------------------------------------------------------------------------
//! @brief        浮動少数点型を比較するための等値比較関数オブジェクトです。
//!
//! @tparam       ValueT 比較する型です。
//---------------------------------------------------------------------------
template<typename ValueT>
class
CloseAtTolerance
{
public:
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    tolerance 誤差許容値です。
    //!
    CloseAtTolerance(ValueT tolerance) : m_Tolerance(tolerance) {}

    //! @brief        コンストラクタです。
    //!
    //! @param[in]    numberOfRoundingError 丸め誤差の数です。
    //!
    CloseAtTolerance(int numberOfRoundingError)
    : m_Tolerance(numberOfRoundingError * std::numeric_limits<ValueT>::epsilon() * 0.5f) {}

    //! @brief        許容範囲内かどうか比較します。
    //!
    //! @param[in]    lhs 左辺です。
    //! @param[in]    rhs 右辺です。
    //! @return       lhs と rhs の値が許容範囲内であれば true を返します。
    //!
    bool operator()(ValueT lhs, ValueT rhs) const
    {
        ValueT diff = Abs(lhs - rhs);
        return
            diff <= (m_Tolerance * Abs(lhs)) &&
            diff <= (m_Tolerance * Abs(rhs));
    }

private:
    ValueT m_Tolerance;
};

//---------------------------------------------------------------------------
//! @brief        浮動少数点型を比較するための等値比較関数オブジェクトです。
//!
//! @tparam       ValueT 比較する型です。
//---------------------------------------------------------------------------
template<typename ValueT>
class
CloseAtToleranceWeak
{
public:
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    tolerance 誤差許容値です。
    //!
    CloseAtToleranceWeak(ValueT tolerance) : m_Tolerance(tolerance) {}

    //! @brief        コンストラクタです。
    //!
    //! @param[in]    numberOfRoundingError 丸め誤差の数です。
    //!
    CloseAtToleranceWeak(int numberOfRoundingError)
    : m_Tolerance(numberOfRoundingError * std::numeric_limits<ValueT>::epsilon() * 0.5f) {}

    //! @brief        許容範囲内かどうか比較します。
    //!
    //! @param[in]    lhs 左辺です。
    //! @param[in]    rhs 右辺です。
    //! @return       lhs と rhs の値が許容範囲内であれば true を返します。
    //!
    bool operator()(ValueT lhs, ValueT rhs) const
    {
        ValueT diff = Abs(lhs - rhs);
        return
            diff <= (m_Tolerance * Abs(lhs)) ||
            diff <= (m_Tolerance * Abs(rhs));
    }

private:
    ValueT m_Tolerance;
};

//---------------------------------------------------------------------------
//! @brief       浮動少数点型の誤差範囲を考慮した等値比較です。
//!
//! @param[in]   lhs   左辺です。
//! @param[in]   rhs   右辺です。
//!
//! @return      等値と判断されたら true を返します。
//---------------------------------------------------------------------------
template<typename ValueT>
NW_INLINE bool
FloatEquals(ValueT lhs, ValueT rhs)
{
    return CloseAtTolerance<ValueT>(std::numeric_limits<ValueT>::epsilon())(lhs, rhs);
}

//---------------------------------------------------------------------------
//! @brief       浮動少数点型の誤差範囲を考慮したほぼ等値比較です。
//!
//! @param[in]   lhs   左辺です。
//! @param[in]   rhs   右辺です。
//!
//! @return      等値と判断されたら true を返します。
//---------------------------------------------------------------------------
template<typename ValueT>
NW_INLINE bool
FloatEqualsWeak(ValueT lhs, ValueT rhs)
{
    return CloseAtToleranceWeak<ValueT>(std::numeric_limits<ValueT>::epsilon())(lhs, rhs);
}

namespace internal {

typedef union
{
    f32 f;
    u32 u;
} f32_u32_convert_t;

typedef union
{
    f64 f;
    u64 u;
} f64_u64_convert_t;

} // namespace internal

//---------------------------------------------------------------------------
//! @brief        f32 型の値をビット列を変更することなく u32 型にします
//!
//! @param[in]    x       u32型 にする値。
//!
//! @return       x と等しいビット表現を持つ u32 型の値。
//---------------------------------------------------------------------------
NW_INLINE u32
F32AsU32(f32 x)
{
#if 1
    internal::f32_u32_convert_t val;
    val.f = x;
    return val.u;
#else
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    char* ptr = reinterpret_cast<char*>( &x );
    return *reinterpret_cast<u32*>(ptr);
#endif
}

//---------------------------------------------------------------------------
//! @brief        f32 型のポインタを u32 型のポインタにキャストします
//!
//! @param[in]    x       f32型 のポインタ。
//!
//! @return       u32 型のポインタを返します。
//---------------------------------------------------------------------------
NW_INLINE u32*
F32AsU32(f32* x)
{
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    char* ptr = reinterpret_cast<char*>( x );
    return reinterpret_cast<u32*>(ptr);
}

//---------------------------------------------------------------------------
//! @brief        f32 型のポインタを u32 型のポインタにキャストします
//!
//! @param[in]    x       f32型 のポインタ。
//!
//! @return       u32 型のポインタを返します。
//---------------------------------------------------------------------------
NW_INLINE const u32*
F32AsU32(const f32* x)
{
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    const char* ptr = reinterpret_cast<const char*>( x );
    return reinterpret_cast<const u32*>(ptr);
}


//---------------------------------------------------------------------------
//! @brief        u32 型の値をビット列を変更することなく f32 型にします
//!
//! @param[in]    x       f32型にする値。
//!
//! @return       x と等しいビット表現を持つ f32 型の値。
//---------------------------------------------------------------------------
NW_INLINE f32
U32AsF32(u32 x)
{
#if 1
    internal::f32_u32_convert_t val;
    val.u = x;
    return val.f;
#else
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    char* ptr = reinterpret_cast<char*>( &x );
    return *reinterpret_cast<f32*>(ptr);
#endif
}

//---------------------------------------------------------------------------
//! @brief        u32 型のポインタを f32 型のポインタにキャストします
//!
//! @param[in]    x       u32型 のポインタ。
//!
//! @return       f32 型のポインタを返します。
//---------------------------------------------------------------------------
NW_INLINE f32*
U32AsF32(u32* x)
{
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    char* ptr = reinterpret_cast<char*>( x );
    return reinterpret_cast<f32*>(ptr);
}

//---------------------------------------------------------------------------
//! @brief        u32 型のポインタを f32 型のポインタにキャストします
//!
//! @param[in]    x       u32型 のポインタ。
//!
//! @return       f32 型のポインタを返します。
//---------------------------------------------------------------------------
NW_INLINE const f32*
U32AsF32(const u32* x)
{
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    const char* ptr = reinterpret_cast<const char*>( x );
    return reinterpret_cast<const f32*>(ptr);
}

//---------------------------------------------------------------------------
//! @brief        f64 型の値をビット列を変更することなく u64 型にします
//!
//! @param[in]    x       u64型 にする値。
//!
//! @return       x と等しいビット表現を持つ u64 型の値。
//---------------------------------------------------------------------------
NW_INLINE u64
F64AsU64(f64 x)
{
#if 1
    internal::f64_u64_convert_t val;
    val.f = x;
    return val.u;
#else
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    char* ptr = reinterpret_cast<char*>( &x );
    return *reinterpret_cast<u64*>(ptr);
#endif
}

//---------------------------------------------------------------------------
//! @brief        f64 型のポインタを u64 型のポインタにキャストします
//!
//! @param[in]    x       f64 型 のポインタ。
//!
//! @return       u64 型のポインタを返します。
//---------------------------------------------------------------------------
NW_INLINE u64*
F64AsU64(f64* x)
{
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    char* ptr = reinterpret_cast<char*>( x );
    return reinterpret_cast<u64*>(ptr);
}

//---------------------------------------------------------------------------
//! @brief        f64 型のポインタを u64 型のポインタにキャストします
//!
//! @param[in]    x       f64 型 のポインタ。
//!
//! @return       u64 型のポインタを返します。
//---------------------------------------------------------------------------
NW_INLINE const u64*
F64AsU64(const f64* x)
{
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    const char* ptr = reinterpret_cast<const char*>( x );
    return reinterpret_cast<const u64*>(ptr);
}


//---------------------------------------------------------------------------
//! @brief        u64 型の値をビット列を変更することなく f64 型にします
//!
//! @param[in]    x       f64型にする値。
//!
//! @return       x と等しいビット表現を持つ f64 型の値。
//---------------------------------------------------------------------------
NW_INLINE f64
U64AsF64(u64 x)
{
#if 1
    internal::f64_u64_convert_t val;
    val.u = x;
    return val.f;
#else
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    char* ptr = reinterpret_cast<char*>( &x );
    return *reinterpret_cast<f64*>(ptr);
#endif
}

//---------------------------------------------------------------------------
//! @brief        u64 型のポインタを f64 型のポインタにキャストします
//!
//! @param[in]    x       u64 型 のポインタ。
//!
//! @return       f64 型のポインタを返します。
//---------------------------------------------------------------------------
NW_INLINE f64*
U64AsF64(u64* x)
{
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    char* ptr = reinterpret_cast<char*>( x );
    return reinterpret_cast<f64*>(ptr);
}

//---------------------------------------------------------------------------
//! @brief        u64 型のポインタを f64 型のポインタにキャストします
//!
//! @param[in]    x       u64 型 のポインタ。
//!
//! @return       f64 型のポインタを返します。
//---------------------------------------------------------------------------
NW_INLINE const f64*
U64AsF64(const u64* x)
{
    // NOTE: char ポインタは、ANSI aliasing rule では例外扱いなので、これでも -ansi_aliasing で通ります。
    const char* ptr = reinterpret_cast<const char*>( x );
    return reinterpret_cast<const f64*>(ptr);
}

//@}


//----------------------------------------
//! @name ポインタ操作関連
//@{

//---------------------------------------------------------------------------
//! @brief       ポインタからIntPtrへのキャストをおこないます。
//!
//! @param[in]   ptr ポインタ
//!
//! @return      ptrをIntPtrにキャストした値を返します。
//---------------------------------------------------------------------------
NW_INLINE IntPtr
GetIntPtr( const void* ptr )
{
    return reinterpret_cast<IntPtr>( ptr );
}


//---------------------------------------------------------------------------
//! @brief       ２つのポインタアドレスのオフセット値をlongで取得します。
//!
//! @param[in]   start   開始アドレス
//! @param[in]   end     終了アドレス
//!
//! @return      ２つのポインタのオフセット値
//---------------------------------------------------------------------------
template <typename TDiff>
NW_INLINE TDiff
GetOffsetFromPtr( const void* start, const void* end )
{
    return static_cast<TDiff>(GetIntPtr(end)) - static_cast<TDiff>(GetIntPtr(start));
}

NW_INLINE PtrDiff
GetOffsetFromPtr( const void* start, const void* end )
{
    return GetOffsetFromPtr<PtrDiff>(start, end);
}


//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。
//!
//! @tparam      TResult キャストする型を指定します。
//! @param[in]   ptr     ポインタ
//! @param[in]   offset  オフセット値
//! @param@in]   align   アライメント値です。
//!
//! @return      voidポインタ型にoffsetバイトを加えたアドレスを引数と同型のポインタとして返します。
//---------------------------------------------------------------------------
template <typename TOffset>
NW_INLINE void*
AddOffsetToPtr( void* ptr, TOffset offset )
{
    return reinterpret_cast<void*>( GetIntPtr(ptr) + offset );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(const版)
//---------------------------------------------------------------------------
template <typename TOffset>
NW_INLINE const void*
AddOffsetToPtr( const void* ptr, TOffset offset )
{
    return reinterpret_cast<const void*>( GetIntPtr(ptr) + offset );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(キャスト付)
//---------------------------------------------------------------------------
template <typename TResult, typename TOffset>
NW_INLINE TResult
AddOffsetToPtr( void* ptr, TOffset offset )
{
    return static_cast<TResult>( AddOffsetToPtr(ptr, offset) );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(キャスト付き, const版)
//---------------------------------------------------------------------------
template <typename TResult, typename TOffset>
NW_INLINE TResult
AddOffsetToPtr( const void* ptr, TOffset offset )
{
    return static_cast<TResult>( AddOffsetToPtr(ptr, offset) );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(アライメント付き)
//---------------------------------------------------------------------------
template <typename TOffset>
NW_INLINE void*
AddOffsetToPtr( void* ptr, TOffset offset, u32 align )
{
    return RoundUp(reinterpret_cast<void*>( GetIntPtr(ptr) + offset ), align);
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(アライメント付き, const版)
//---------------------------------------------------------------------------
template <typename TOffset>
NW_INLINE const void*
AddOffsetToPtr( const void* ptr, TOffset offset, u32 align )
{
    return RoundUp(reinterpret_cast<const void*>( GetIntPtr(ptr) + offset ), align);
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(キャスト付き, アライメント付き)
//---------------------------------------------------------------------------
template <typename TResult, typename TOffset>
NW_INLINE TResult
AddOffsetToPtr( void* ptr, TOffset offset, u32 align )
{
    return static_cast<TResult>( AddOffsetToPtr(ptr, offset, align) );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(キャスト付き, アライメント付き, const 版)
//---------------------------------------------------------------------------
template <typename TResult, typename TOffset>
NW_INLINE TResult
AddOffsetToPtr( const void* ptr, TOffset offset, u32 align )
{
    return static_cast<TResult>( AddOffsetToPtr(ptr, offset, align) );
}


//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。
//!
//! @param[in]   ptr     ポインタ
//! @param[in]   offset  オフセット値
//! @param[in]   align   結果に必要なアライメント値
//!
//! @return      voidポインタ型にoffsetバイトを加えたアドレスを引数と同型のポインタとして返します。
//---------------------------------------------------------------------------
NW_INLINE void*
AddU32ToPtr( void* ptr, u32 offset )
{
    return AddOffsetToPtr( ptr, offset );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(const 版)
//---------------------------------------------------------------------------
NW_INLINE const void*
AddU32ToPtr( const void* ptr, u32 offset )
{
    return AddOffsetToPtr( ptr, offset );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(キャスト付き)
//---------------------------------------------------------------------------
template <typename TResult>
NW_INLINE TResult
AddU32ToPtr( void* ptr, u32 offset )
{
    return static_cast<TResult>( AddU32ToPtr(ptr, offset) );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(キャスト付き, const 版)
//---------------------------------------------------------------------------
template <typename TResult>
NW_INLINE TResult
AddU32ToPtr( const void* ptr, u32 offset )
{
    return static_cast<TResult>( AddU32ToPtr(ptr, offset) );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(アライメント付き)
//---------------------------------------------------------------------------
NW_INLINE void*
AddU32ToPtr( void* ptr, u32 offset, u32 align )
{
    return AddOffsetToPtr( ptr, offset, align );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(アライメント付き, const 版)
//---------------------------------------------------------------------------
NW_INLINE const void*
AddU32ToPtr( const void* ptr, u32 offset, u32 align )
{
    return AddOffsetToPtr( ptr, offset, align );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(アライメント付き, キャスト付き)
//---------------------------------------------------------------------------
template <typename TResult>
NW_INLINE TResult
AddU32ToPtr( void* ptr, u32 offset, u32 align )
{
    return static_cast<TResult>( AddU32ToPtr(ptr, offset, align) );
}

//---------------------------------------------------------------------------
//! @brief       ポインタにオフセット値を加えます。(アライメント付き, キャスト付き, const 版)
//---------------------------------------------------------------------------
template <typename TResult>
NW_INLINE TResult
AddU32ToPtr( const void* ptr, u32 offset, u32 align )
{
    return static_cast<TResult>( AddU32ToPtr(ptr, offset, align) );
}



//---------------------------------------------------------------------------
//! @brief       ２つのポインタアドレスを比較します。
//!
//! @param[in]   a   比較元のポインタ
//! @param[in]   b   比較対象となるポインタ
//!
//! @return      > 0 aの指すアドレスがbの指すアドレスよりも大きい場合
//!              = 0 aの指すアドレスとbの指すアドレスが等しい場合
//!              < 0 aの指すアドレスがbの指すアドレスよりも小さい場合
//---------------------------------------------------------------------------
NW_INLINE int
ComparePtr( const void* a, const void* b )
{
    if ( GetIntPtr( a ) == GetIntPtr( b ) )
    {
        return 0;
    }
    else
    {
        return ( GetIntPtr( a ) > GetIntPtr( b ) ) ? 1 : -1;
    }
}

//@}


 //----------------------------------------
//! @name ビット操作関連
//@{

//---------------------------------------------------------------------------
//! @brief       ビット列から部分ビット列を抜き出します。
//!
//! @param[in]   bits    抜き出し元のビット列
//! @param[in]   pos     抜き出すビット列の最下位ビットのbits中での位置
//! @param[in]   len     抜き出すビット長
//!
//! @return      抽出したビット列。
//---------------------------------------------------------------------------
template <typename OutputT>
NW_INLINE OutputT
BitExtract(OutputT bits, int pos, int len=1)
{
    const OutputT mask = static_cast<OutputT>( ( 1UL << len ) - 1 );
    return static_cast<OutputT>( (bits >> pos) & mask );
}


//---------------------------------------------------------------------------
//! @brief       エンディアンを反転します。
//!
//! @param[in]   x       反転する数。
//!
//! @return      x のエンディアンを反転して返します。
//---------------------------------------------------------------------------
NW_INLINE u32
ReverseEndian( u32 x )
{
    return static_cast<u32>(
        BitExtract( x,  0, 8 ) << 24 |
        BitExtract( x,  8, 8 ) << 16 |
        BitExtract( x, 16, 8 ) <<  8 |
        BitExtract( x, 24, 8 ) <<  0
    );
}

//---------------------------------------------------------------------------
//! @brief       エンディアンを反転します。
//---------------------------------------------------------------------------
NW_INLINE s32
ReverseEndian( s32 x )
{
    return static_cast<s32>(
        BitExtract( x,  0, 8 ) << 24 |
        BitExtract( x,  8, 8 ) << 16 |
        BitExtract( x, 16, 8 ) <<  8 |
        BitExtract( x, 24, 8 ) <<  0
    );
}

//---------------------------------------------------------------------------
//! @brief       エンディアンを反転します。
//---------------------------------------------------------------------------
NW_INLINE u16
ReverseEndian( u16 x )
{
    return static_cast<u16>(
        BitExtract( x,  0, 8 ) << 8 |
        BitExtract( x,  8, 8 ) << 0
    );
}

//---------------------------------------------------------------------------
//! @brief       エンディアンを反転します。
//---------------------------------------------------------------------------
NW_INLINE s16
ReverseEndian( s16 x )
{
    return static_cast<s16>(
        BitExtract( x,  0, 8 ) << 8 |
        BitExtract( x,  8, 8 ) << 0
    );
}

//---------------------------------------------------------------------------
//! @brief       エンディアンを反転します。
//---------------------------------------------------------------------------
NW_INLINE u64
ReverseEndian( u64 x )
{
    return static_cast<u64>(
        BitExtract( x,  0, 8 ) << 56 |
        BitExtract( x,  8, 8 ) << 48 |
        BitExtract( x, 16, 8 ) << 40 |
        BitExtract( x, 24, 8 ) << 32 |
        BitExtract( x, 32, 8 ) << 24 |
        BitExtract( x, 40, 8 ) << 16 |
        BitExtract( x, 48, 8 ) <<  8 |
        BitExtract( x, 56, 8 ) <<  0
    );
}

//---------------------------------------------------------------------------
//! @brief       エンディアンを反転します。
//---------------------------------------------------------------------------
NW_INLINE s64
ReverseEndian( s64 x )
{
    return static_cast<s64>(
        BitExtract( x,  0, 8 ) << 56 |
        BitExtract( x,  8, 8 ) << 48 |
        BitExtract( x, 16, 8 ) << 40 |
        BitExtract( x, 24, 8 ) << 32 |
        BitExtract( x, 32, 8 ) << 24 |
        BitExtract( x, 40, 8 ) << 16 |
        BitExtract( x, 48, 8 ) <<  8 |
        BitExtract( x, 56, 8 ) <<  0
    );
}

//---------------------------------------------------------------------------
//! @brief       エンディアンを反転します。
//!
//! @param[in]   x       反転する数。
//!
//! @return      x のエンディアンを反転して返します。
//---------------------------------------------------------------------------
NW_INLINE f32
ReverseEndian( f32 x )
{
    u32 ux = F32AsU32( x );
    u32 rux = ReverseEndian( ux );

    return U32AsF32( rux );
}

//@}

namespace internal {

//--------------------------------------------------------------------------
//! @brief        ビット列から値を取り出します。
//!
//! @param[in]    bits    元のビット列です。
//! @param[in]    width   ビット幅です。
//! @param[in]    shift   シフト値です。
//!
//! @return       ビット長とシフト値から取得した値を返します。
//---------------------------------------------------------------------------
NW_INLINE u32
ReadBits(u32 bits, s32 width, s32 shift)
{
    return (u32(bits) >> shift) & ((0x1U << width) - 1);
}

//--------------------------------------------------------------------------
//! @brief        ビット長とシフト値を指定してマスクされたビット列を取得します。
//!
//! @param[in]    value   ビット列に対して設定する値です。
//! @param[in]    width   ビット幅です。
//! @param[in]    shift   シフト値です。
//!
//! @return       ビット長とシフト値から生成したビット列を返します。
//---------------------------------------------------------------------------
template <typename T>
NW_INLINE u32
MakeBits(T value, s32 width, s32 shift)
{
    return (u32(value) & ((0x1U << width) - 1)) << shift;
}

//--------------------------------------------------------------------------
//! @brief        ビット列に特定の値を書き込みます。
//!
//! @param[in]    bits    元ビット列です。
//! @param[in]    value   ビット列の一部を置き換える為の値です。
//! @param[in]    width   ビット幅です。
//! @param[in]    shift   シフト値です。
//!
//! @return       置き換え後のビット列を返します。
//---------------------------------------------------------------------------
template <typename T>
NW_INLINE u32
WriteBits(u32 bits, T value, s32 width, s32 shift)
{
    u32 mask = (0x1U << width) - 1;

    return (bits & ~(mask << shift)) | ((u32(value) & mask) << shift);
}

template <typename T>
NW_FORCE_INLINE
bool
TestBit(
    T       bits,
    int     pos
)
{
    const T mask = T(1 << pos);

    return 0 != (bits & mask);
}

template <typename T>
NW_FORCE_INLINE
T
GetBits(
    T       bits,
    int     pos,
    int     len
)
{
    NW_ASSERT(len <= 32);
    const u32 mask = ~(0xFFFFFFFFU << len);

    return T((bits >> pos) & mask);
}

NW_FORCE_INLINE
void* Memset(void* dst, u8 val, size_t size)
{
#if defined(NW_PLATFORM_CAFE)
    return OSBlockSet(dst, val, size);
#else
    return ::std::memset(dst, val, size);
#endif
}

} // internal

//---------------------------------------------------------------------------
//! @brief        配列のサイズを取得します。
//!
//! @param[in]    array サイズを取得したい配列
//!
//! @return       配列のサイズ
//---------------------------------------------------------------------------
template<class T, std::size_t S>
NW_INLINE int
GetArrayLength(
    const T (&array)[S]
)
{
    NW_UNUSED_VARIABLE(array);
    return static_cast<int>(S);
}

//----------------------------------------
//! @name オブジェクト破棄
//@{

//---------------------------------------------------------------------------
//! @brief        オブジェクトを削除後に０を設定するためのインライン関数です。
//!
//! @tparam       TObject 削除するオブジェクトの型です。
//!
//! @param[in]    object 削除するオブジェクトです。
//---------------------------------------------------------------------------
template<typename TObject>
NW_INLINE void
SafeDestroy(
    TObject*& object
)
{
    if (object == NULL) { return; }
    object->Destroy();
    object = NULL;
}

//---------------------------------------------------------------------------
//! @brief        SafeDestroy でオブジェクトを破棄するためのデリーターです。
//!
//! @tparam       TObject 削除するオブジェクトの型です。
//---------------------------------------------------------------------------
template<typename TObject>
struct SafeDestroyer : public std::unary_function<TObject&, void>
{
    //! オブジェクトを破棄します。
    void operator()(TObject& object) const
    {
        SafeDestroy(object);
    }
};

//---------------------------------------------------------------------------
//! @brief        SafeDestroy でコンテナ要素の全てのオブジェクトを破棄するための関数です。
//!
//! @tparam       TArray 削除するオブジェクトの型です。
//!
//! @param[in]    array 削除するオブジェクトの配列です。
//---------------------------------------------------------------------------
template<typename TArray>
NW_INLINE void
SafeDestroyAll(
    TArray& array
)
{
    std::for_each(array.begin(), array.end(), SafeDestroyer<typename TArray::value_type>());
    array.clear();
}

//@}

} /* namespace ut */
} /* namespace nw */

#endif //  NW_UT_INLINES_H_
