﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW4R_UT_TEXTWRITERBASE_H_
#define NW4R_UT_TEXTWRITERBASE_H_

// 何故か CW3 ではデフォルトで WideChar が使えない
//#ifdef NW4R_CW3
// ↑↓CW3のバグ回避
#ifndef NW4R_CW2
 #ifdef  _MSL_WIDE_CHAR
  #undef  _MSL_WIDE_CHAR
 #endif // _MSL_WIDE_CHAR

 #define _MSL_WIDE_CHAR 1
#endif  // NW4R_CW3


#include <cstddef>
#include <cwchar>
#include <cstdio>
#include <cstring>
#include <dolphin/types.h>
#include "Color.h"
#include "Font.h"
#include "CharWriter.h"
#include "TagProcessorBase.h"


namespace NW4R
{
namespace Font
{
namespace UnManaged
{

template <typename CharType>
class TextWriterBase: public CharWriter
{
public:
    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    enum PositionFlag
    {
  HORIZONTAL_ALIGN_LEFT       = 0x0,
        HORIZONTAL_ALIGN_CENTER     = 0x1,
        HORIZONTAL_ALIGN_RIGHT      = 0x2,
        HORIZONTAL_ALIGN_MASK       = 0x3,

        HORIZONTAL_ORIGIN_LEFT      = 0x00,
        HORIZONTAL_ORIGIN_CENTER    = 0x10,
        HORIZONTAL_ORIGIN_RIGHT     = 0x20,
        HORIZONTAL_ORIGIN_MASK      = 0x30,

        VERTICAL_ORIGIN_TOP         = 0x000,
        VERTICAL_ORIGIN_MIDDLE      = 0x100,
        VERTICAL_ORIGIN_BOTTOM      = 0x200,
        VERTICAL_ORIGIN_BASELINE    = 0x300,
        VERTICAL_ORIGIN_MASK        = 0x300,

        // 以下の定義は削除予定です。
        VERTICAL_BASE_TOP           = 0x0000,
        VERTICAL_BASE_BASELINE      = VERTICAL_ORIGIN_BASELINE  // VERTICAL_ORIGIN_TOP と共に使用していた場合のみ同じになります。
    };

    enum ContextFlag
    {
        CONTEXT_NO_CHAR_SPACE       = 0x1
    };

    typedef CharType                    CharType;
    typedef const CharType*             StreamType;
    typedef TagProcessorBase<CharType>  TagProcessor;
    typedef TextWriterBase<CharType>    SelfType;

    /* ------------------------------------------------------------------------
            変数
       ------------------------------------------------------------------------ */
    static const u32 DEFAULT_DRAWFLAG = (HORIZONTAL_ALIGN_LEFT | HORIZONTAL_ORIGIN_LEFT | VERTICAL_ORIGIN_TOP);

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- コンストラクタ/デストラクタ
                        TextWriterBase();
                        ~TextWriterBase();

    //---- 行間/文字間/タブ幅
    void                SetLineHeight(float height);
    float               GetLineHeight() const;
    void                SetLineSpace(float space);
    void                SetCharSpace(float space);
    float               GetLineSpace() const;
    float               GetCharSpace() const;
    void                SetTabWidth( int tabWidth );
    int                 GetTabWidth() const;
    void                SetDrawFromRightToLeftEnabled(bool isDrawFromRightToLeftEnabled);
    bool                GetDrawFromRightToLeftEnabled() const;

    //! @name 文字列の自動折り返し
    //! @brief      自動的に折り返す幅を指定します。
    void                SetWidthLimit(f32 limit)    { m_WidthLimit  = limit; }

    //! @brief      設定されている折り返し幅を取得します。
    //!
    //! @return     設定されている折り返し幅を返します。@n
    //!             折り返し幅が無効に設定されているときは正の無限大を返します。
    //!
    f32                 GetWidthLimit() const       { return m_WidthLimit; }

    //! 自動折り返し機能を無効に設定します。
    void                ResetWidthLimit()           { SetWidthLimit(FLT_MAX); }


    // 描画フラグ
    void                SetDrawFlag(u32 flags);
    u32                 GetDrawFlag() const;

    //---- タグ処理
    void                SetTagProcessor(TagProcessor* tagProcessor);
    void                ResetTagProcessor();
    TagProcessor&       GetTagProcessor() const;


    //---- 文字列サイズ計算
    float               CalcFormatStringWidth(
                            StreamType      format,
                            ...
                        ) const;

    float               CalcFormatStringHeight(
                            StreamType      format,
                            ...
                        ) const;

    void                CalcFormatStringRect(
                            Rect*           pRect,
                            StreamType      format,
                            ...
                        ) const;

    void                CalcVStringRect(
                            Rect*           pRect,
                            StreamType      format,
                            va_list    args
                        ) const;

    float               CalcStringWidth(
                            StreamType      str,
                            int             length
                        ) const;

    float               CalcStringHeight(
                            StreamType      str,
                            int             length
                        ) const;

    void                CalcStringRect(
                            Rect*           pRect,
                            StreamType      str,
                            int             length
                        ) const;

    float               CalcStringWidth(
                            StreamType      str
                        ) const;

    float               CalcStringHeight(
                            StreamType      str
                        ) const;

    void                CalcStringRect(
                            Rect*           pRect,
                            StreamType      str
                        ) const;

    //---- 文字列描画
    float               Printf(
                            StreamType      format,
                            ... );

    float               VPrintf(
                            StreamType      format,
                            va_list    args );

    float               Print(
                            StreamType      str,
                            int             length );

    float               Print(
                            StreamType      str );

    float               Print(
                            StreamType      str,
                            int             length,
                            int             lineOffsetCount,
                            const float*    pLineOffset );




    //---- sprintfバッファ
    static void*        SetBuffer(
                            CharType* buffer,
                            std::size_t size );

    static void*        SetBuffer(
                            std::size_t size
                        );

    static std::size_t  GetBufferSize();
    static const void*  GetBuffer();


    //---- vsnprintf の 分岐
    static int          VSNPrintf(
                            char*                   buffer,
                            std::size_t             count,
                            const char*             format,
                            va_list                 arg )
    {
        // return vsnprintf(buffer, count, format, arg);
        return _vsnprintf(buffer, count, format, arg);

    }

    static int          VSNPrintf(
                            wchar_t*                buffer,
                            std::size_t             count,
                            const wchar_t*          format,
                            va_list                 arg )
    {
        // return std::vswprintf(buffer, count, format, arg);
        return _vsnwprintf(buffer, count, format, arg);
    }

    static int          StrLen(const char* str)
    {
        return static_cast<int>(std::strlen(str));
    }

    static int          StrLen(const wchar_t* str)
    {
        return static_cast<int>(std::wcslen(str));
    }


private:
    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    static const int DEFAULT_FORMAT_BUFFER_SIZE = 256;


    /* ------------------------------------------------------------------------
            変数
       ------------------------------------------------------------------------ */
    float                   m_WidthLimit;           //!< 描画幅制限
    float                   mCharSpace;             // 文字間
    float                   mLineSpace;             // 行間
    int                     mTabWidth;              // タブ幅
    u32                     mDrawFlag;              // 描画フラグ
    TagProcessor*           mTagProcessor;          // タグ処理機
    bool                    m_IsDrawFromRightToLeftEnabled;  // 右側から描画するかどうか

    //---- sprintf バッファ
    static CharType*        mFormatBuffer;          // バッファへのポインタ
    static std::size_t      mFormatBufferSize;      // バッファサイズ

    //---- デフォルトタグプロセッサ
    static TagProcessor     mDefaultTagProcessor;



    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- 文字列処理
    float               CalcLineWidth(
                            StreamType              str,
                            int                     length );

    int                CalcLineRectImpl(
                            Rect*                   pRect,
                            StreamType*             pStr,
                            int                     length );

    void                CalcStringRectImpl(
                            Rect*                   pRect,
                            StreamType              str,
                            int                     length );

    float               PrintImpl(
                            StreamType              str,
                            int                     length,
                            int                     lineOffsetCount,
                            const float*            pLineOffset );

    float               AdjustCursor(
                            float*                  xOrigin,
                            float*                  yOrigin,
                            StreamType              str,
                            int                     length );

    bool                IsDrawFlagSet( u32 mask, u32 flag ) const
    {
        return (mDrawFlag & mask) == flag;
    }

    float               GetLineOffset(int lineOffsetCount, const float* pLineOffset, int line) const;
};




} /* UnManaged */
} /* namespace ut */
} /* namespace nw4r */

#endif //  NW4R_UT_TEXTWRITERBASE_H_
