﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "inlines.h"
#include "nw4r/misc.h"

#include "ByteOrderUtil.h"
#include "BinaryFileFormat.h"

#ifndef BYTESWAPUTIL_H_
#define BYTESWAPUTIL_H_

namespace NW4R
{
namespace Font
{
namespace UnManaged
{

class FontSwapUtil
{
private:
    bool    mSwapEnabled;

public:
    void SetByteSwapEnabled(bool byteSwapEnabled) { mSwapEnabled = byteSwapEnabled;}

//-------------------------------------------------------------------------
// BinaryFileHeader を バイトスワップします
void SwapBinaryFileHeader_( BinaryFileHeader* header )
{
    NW4R_POINTER_ASSERT( header );

    // signature, byteOrder に関しては、スワップする必要はない
    header->fileSize  = mSwapEnabled ? Swap4Byte( header->fileSize ) : header->fileSize;
    header->headerSize   = mSwapEnabled ? Swap2Byte( header->headerSize ) : header->headerSize;
    header->version      = mSwapEnabled ? Swap4Byte( header->version ) : header->version;
    header->fileSize     = mSwapEnabled ? Swap4Byte( header->fileSize ) : header->fileSize;
    header->dataBlocks   = mSwapEnabled ? Swap2Byte( header->dataBlocks ) : header->dataBlocks;
}

//-------------------------------------------------------------------------
// BinaryBlockHeader を バイトスワップします
void SwapBinaryBlockHeader_( BinaryBlockHeader* header )
{
    NW4R_POINTER_ASSERT( header );

    // kind については、スワップする必要はない
    header->size  = mSwapEnabled ? Swap4Byte( header->size ) : header->size;
}


//-------------------------------------------------------------------------
// FontGlyphGroups を バイトスワップします
void SwapFontGlyphGroups_( FontGlyphGroups* pGroup )
{
    pGroup->sheetSize = mSwapEnabled ? Swap4Byte( pGroup->sheetSize ) : pGroup->sheetSize;

    pGroup->glyphsPerSheet = mSwapEnabled ? Swap2Byte( pGroup->glyphsPerSheet ) : pGroup->glyphsPerSheet;
    pGroup->numSet = mSwapEnabled ? Swap2Byte( pGroup->numSet ) : pGroup->numSet;
    pGroup->numSheet = mSwapEnabled ? Swap2Byte( pGroup->numSheet ) : pGroup->numSheet;
    pGroup->numCWDH = mSwapEnabled ? Swap2Byte( pGroup->numCWDH ) : pGroup->numCWDH;
    pGroup->numCMAP = mSwapEnabled ? Swap2Byte( pGroup->numCMAP ) : pGroup->numCMAP;

    u16* pNameOffsets = reinterpret_cast<u16*>(&pGroup->nameOffsets);
    for( int i = 0; i < pGroup->numSet; i++ )
    {
        pNameOffsets[i] = mSwapEnabled ? Swap2Byte( pNameOffsets[i] ) : pNameOffsets[i];
    }
}

//-------------------------------------------------------------------------
// FontInformation を バイトスワップします
void SwapFontInformation_( FontInformation* pInfo )
{
    NW4R_POINTER_ASSERT( pInfo );

    pInfo->linefeed   = mSwapEnabled ? Swap2Byte( pInfo->linefeed ) : pInfo->linefeed;
    pInfo->pGlyph = mSwapEnabled ? Swap4Byte( pInfo->pGlyph ) : pInfo->pGlyph;
    pInfo->pWidth = mSwapEnabled ? Swap4Byte( pInfo->pWidth ) : pInfo->pWidth;
    pInfo->pMap   = mSwapEnabled ? Swap4Byte( pInfo->pMap ) : pInfo->pMap;

    pInfo->alterCharIndex   = mSwapEnabled ? Swap2Byte( pInfo->alterCharIndex ) : pInfo->alterCharIndex;
}


//-------------------------------------------------------------------------
// FontTextureGlyph を バイトスワップします
void SwapFontTextureGlyph_( FontTextureGlyph* pGlyph )
{
    NW4R_POINTER_ASSERT( pGlyph );

    pGlyph->sheetSize    = mSwapEnabled ? Swap4Byte( pGlyph->sheetSize ) : pGlyph->sheetSize;
    pGlyph->baselinePos  = mSwapEnabled ? Swap2Byte( pGlyph->baselinePos ) : pGlyph->baselinePos;
    pGlyph->sheetFormat  = mSwapEnabled ? Swap2Byte( pGlyph->sheetFormat ) : pGlyph->sheetFormat;
    pGlyph->sheetRow     = mSwapEnabled ? Swap2Byte( pGlyph->sheetRow ) : pGlyph->sheetRow;
    pGlyph->sheetLine    = mSwapEnabled ? Swap2Byte( pGlyph->sheetLine ) : pGlyph->sheetLine;
    pGlyph->sheetWidth   = mSwapEnabled ? Swap2Byte( pGlyph->sheetWidth ) : pGlyph->sheetWidth;
    pGlyph->sheetHeight  = mSwapEnabled ? Swap2Byte( pGlyph->sheetHeight ) : pGlyph->sheetHeight;

    pGlyph->sheetImage   = mSwapEnabled ? Swap4Byte( pGlyph->sheetImage ) : pGlyph->sheetImage;
}

//-------------------------------------------------------------------------
// FontWidth を バイトスワップします
void SwapFontWidth_( FontWidth* pFontWidth )
{
    NW4R_POINTER_ASSERT( pFontWidth );

    pFontWidth->indexBegin   = mSwapEnabled ? Swap2Byte( pFontWidth->indexBegin ) : pFontWidth->indexBegin;
    pFontWidth->indexEnd     = mSwapEnabled ? Swap2Byte( pFontWidth->indexEnd ) : pFontWidth->indexEnd;

    pFontWidth->pNext        = mSwapEnabled ? Swap4Byte( pFontWidth->pNext ) : pFontWidth->pNext;
}

//-------------------------------------------------------------------------
// FontCodeMap を バイトスワップします
void SwapFontCodeMap_( FontCodeMap* pFontCodeMap )
{
    NW4R_POINTER_ASSERT( pFontCodeMap );

    pFontCodeMap->ccodeBegin        = mSwapEnabled ? Swap4Byte( pFontCodeMap->ccodeBegin ) : pFontCodeMap->ccodeBegin;
    pFontCodeMap->ccodeEnd          = mSwapEnabled ? Swap4Byte( pFontCodeMap->ccodeEnd ) : pFontCodeMap->ccodeEnd;

    pFontCodeMap->mappingMethod     = mSwapEnabled ? Swap2Byte( pFontCodeMap->mappingMethod ) : pFontCodeMap->mappingMethod;
    pFontCodeMap->reserved          = mSwapEnabled ? Swap2Byte( pFontCodeMap->reserved ) : pFontCodeMap->reserved;

    pFontCodeMap->pNext             = mSwapEnabled ? Swap4Byte( pFontCodeMap->pNext ) : pFontCodeMap->pNext;


    // マッピング方式に従って、マップ情報をバイトスワップします
    FontMapMethod mapMethod = (FontMapMethod)pFontCodeMap->mappingMethod;
    switch( mapMethod )
    {
    //-------------------------------------------------------------
    case FONT_MAPMETHOD_DIRECT:
       // 要素はひとつだけ存在します
        pFontCodeMap->mapInfo = mSwapEnabled ? Swap2Byte(pFontCodeMap->mapInfo) : pFontCodeMap->mapInfo;
    break;
    //-------------------------------------------------------------
    case FONT_MAPMETHOD_TABLE:
       {
           // 要素は（ccodeEnd - ccodeBegin + 1）分存在します
           const int mapLength = pFontCodeMap->ccodeEnd - pFontCodeMap->ccodeBegin + 1;
           for( int i = 0; i < mapLength; i++ )
           {
              reinterpret_cast<u16*>(&pFontCodeMap->mapInfo)[i] =
                  mSwapEnabled ? Swap2Byte( reinterpret_cast<u16*>(&pFontCodeMap->mapInfo)[i] ) : reinterpret_cast<u16*>(&pFontCodeMap->mapInfo)[i];
           }
       }
    break;
    //-------------------------------------------------------------
    case FONT_MAPMETHOD_SCAN:
       {
           // 要素は、mapInfoをキャストして参照する CMapInfoScan に記述されています
           CMapInfoScan*  pMapInfoScan
              = reinterpret_cast<CMapInfoScan*>( &pFontCodeMap->mapInfo );

           pMapInfoScan->num = mSwapEnabled ? Swap2Byte( pMapInfoScan->num ) : pMapInfoScan->num;

           CMapScanEntry*    pMapEntry = NULL;
           for( int i = 0; i < pMapInfoScan->num; i++ )
           {
              pMapEntry = &pMapInfoScan->entries[i];

              pMapEntry->ccode = mSwapEnabled ? Swap4Byte( pMapEntry->ccode ) : pMapEntry->ccode;
              pMapEntry->index = mSwapEnabled ? Swap2Byte( pMapEntry->index ) : pMapEntry->index;
           }
       }
    break;
    //-------------------------------------------------------------
    default:
       NW4R_ASSERTMSG(false, "The FontCodeMap has unknown FontMapMethod.");
       break;
    }

}

//-------------------------------------------------------------------------
// BinaryBlockHeader を バイトスワップします
void SwapFontKerningMap_( FontKerningTable* pKerningTable )
{
    NW4R_POINTER_ASSERT( pKerningTable );

    // 最初のテーブル
    pKerningTable->firstWordNum = mSwapEnabled ? Swap2Byte( pKerningTable->firstWordNum ) : pKerningTable->firstWordNum;
    KerningFirstTableElem* firstTblElem = reinterpret_cast<KerningFirstTableElem*>(&pKerningTable->firstTable);
    for( int i = 0; i < pKerningTable->firstWordNum; i++ )
    {
        firstTblElem[i].firstWord = mSwapEnabled ? Swap4Byte( firstTblElem[i].firstWord ) : firstTblElem[i].firstWord;
        firstTblElem[i].offset    = mSwapEnabled ? Swap4Byte( firstTblElem[i].offset ) : firstTblElem[i].offset;

        // 2番目のテーブル
        KerningSecondTable* pSecondTable = reinterpret_cast<KerningSecondTable*>(&reinterpret_cast<u8*>(pKerningTable)[firstTblElem[i].offset]);
        pSecondTable->secondWordNum = mSwapEnabled ? Swap2Byte( pSecondTable->secondWordNum ) : pSecondTable->secondWordNum;
        KerningSecondTableElem* secondTblElem = reinterpret_cast<KerningSecondTableElem*>(&pSecondTable->elems);
        for(int j = 0; j < pSecondTable->secondWordNum; j++)
        {
            secondTblElem[j].secondWord     = mSwapEnabled ? Swap4Byte( secondTblElem[j].secondWord ) : secondTblElem[j].secondWord;
            secondTblElem[j].kerningValue   = mSwapEnabled ? Swap2Byte( secondTblElem[j].kerningValue ) : secondTblElem[j].kerningValue;
        }
    }
}
};

}// NW4R

}// Font

}// UnManaged

#endif // BYTESWAPUTIL_H_
