﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;

namespace NW4R.ColorPicker
{
    /// <summary>
    /// テキストボックススライダーコントロールクラス。
    /// </summary>
    internal sealed class TextSliderControl : UserControl
    {
        private System.Windows.Forms.TextBox m_txtValue;
        private NW4R.ColorPicker.RGBSlider m_bar;
        private System.Windows.Forms.Label m_label;
        //	装飾用
        private Rectangle m_BarWakuOut	= new Rectangle(32, 8, 256, 24);
        private Rectangle m_BarWakuIn	= new Rectangle(32, 8, 256, 24);

        private ValueMode	m_mode	= ValueMode.Uchar;
        private bool		m_editRight	= false;

        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;

        public TextSliderControl()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();

            // イベント、プロパティ設定
            this.DoubleBuffered = true;
            this.m_bar.ColorSelected += new ColorSlider.ColorSelectedHandler(ColorSlider_ColorSelected);

            InitializeLine();
            Color color = this.m_bar.GetSelectColor();
            this.m_bar.UpdateColor(color, HSV.FromRGB(new RGB(color)), true, true);
            this.Invalidate(true);
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.m_txtValue = new System.Windows.Forms.TextBox();
            this.m_label = new System.Windows.Forms.Label();
            this.m_bar = new NW4R.ColorPicker.RGBSlider();
            this.SuspendLayout();
            //
            // m_txtValue
            //
            this.m_txtValue.AcceptsReturn = true;
            this.m_txtValue.ImeMode = System.Windows.Forms.ImeMode.Disable;
            this.m_txtValue.Location = new System.Drawing.Point(16, 4);
            this.m_txtValue.Multiline = true;
            this.m_txtValue.Name = "m_txtValue";
            this.m_txtValue.Size = new System.Drawing.Size(48, 19);
            this.m_txtValue.TabIndex = 1;
            this.m_txtValue.Text = "0";
            this.m_txtValue.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this.m_txtValue.Leave += new System.EventHandler(this.TextBox_Leave);
            this.m_txtValue.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.TextBox_KeyPress);
            this.m_txtValue.KeyDown += new System.Windows.Forms.KeyEventHandler(this.TextBox_KeyDown);
            //
            // m_label
            //
            this.m_label.Location = new System.Drawing.Point(0, 4);
            this.m_label.Name = "m_label";
            this.m_label.Size = new System.Drawing.Size(16, 19);
            this.m_label.TabIndex = 0;
            this.m_label.Text = "R";
            this.m_label.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            //
            // m_bar
            //
            this.m_bar.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                        | System.Windows.Forms.AnchorStyles.Right)));
            this.m_bar.ColorValue = 0;
            this.m_bar.ComponentType = NW4R.ColorPicker.ColorType.R;
            this.m_bar.Location = new System.Drawing.Point(72, 6);
            this.m_bar.MaxValue = 255;
            this.m_bar.Name = "m_bar";
            this.m_bar.Size = new System.Drawing.Size(256, 15);
            this.m_bar.TabIndex = 2;
            this.m_bar.TabStop = false;
            //
            // TextSliderControl
            //
            this.Controls.Add(this.m_label);
            this.Controls.Add(this.m_bar);
            this.Controls.Add(this.m_txtValue);
            this.Name = "TextSliderControl";
            this.Size = new System.Drawing.Size(336, 24);
            this.ResumeLayout(false);
            this.PerformLayout();

        }
        #endregion

        #region イベント

        /// <summary>
        /// カラーが変更されたときに発生します。
        /// </summary>
        internal delegate void ColorSelectedHandler( object sender, ColorChangedEventArgs e );
        internal event ColorSelectedHandler ColorSelected;

        #endregion

        #region テキストボックスのキーやフォーカスのイベント処理
        /// <summary>
        /// 数字のみ入力許可
        /// </summary>
        private void TextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            TextBox obj = (TextBox)sender;
            if ((e.KeyChar < '0' || e.KeyChar > '9') && e.KeyChar != '\b')
            {
                if (m_mode != ValueMode.Float || e.KeyChar != '.')
                {
                    e.Handled = true;
                }
                else if (obj.Text.IndexOf('.') >= 0)
                {
                    e.Handled = true;
                }
            }
        }

        /// <summary>
        /// リターンキーなら入力確定
        /// </summary>
        private void TextBox_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter || e.KeyCode == Keys.Return)
            {
                TextBox_Leave(sender, EventArgs.Empty);
                m_txtValue.SelectAll();
            }
            else if (e.KeyData == (Keys.Control | Keys.C))
            {
                m_txtValue.Copy();
            }
            else if (e.KeyData == (Keys.Control | Keys.V))
            {
                m_txtValue.Paste();
            }
        }

        /// <summary>
        /// 数値の確定処理
        /// </summary>
        private void TextBox_Leave(object sender, EventArgs e)
        {
            bool rewrite = false;
            TextBox obj = (TextBox)sender;
            if (obj.Text.Length < 1)
            {
                rewrite = true;
            }
            else if (obj.Modified)
            {
                int colValue;
                if (m_mode == ValueMode.Uchar)
                {
                    try
                    {
                        colValue = int.Parse(obj.Text);
                    }
                    catch
                    {
                        rewrite = true;
                        goto label_end;
                    }
                    if (colValue < 0)
                    {
                        rewrite = true;
                        colValue = 0;
                    }
                    else if (colValue > m_bar.MaxValue)
                    {
                        rewrite = true;
                        colValue = m_bar.MaxValue;
                    }
                }
                else
                {
                    float ff;
                    try
                    {
                        ff = float.Parse(obj.Text);
                    }
                    catch
                    {
                        rewrite = true;
                        goto label_end;
                    }
                    if (ff < 0)
                    {
                        rewrite = true;
                        ff = 0;
                    }
                    else if (ff > 1)
                    {
                        rewrite = true;
                        ff = 1;
                    }
                    int index = obj.Text.IndexOf('.');
                    if (index >= 0 && (obj.Text.Length - index) >= 2)
                    {
                        rewrite = true;
                    }
                    colValue = (int)Math.Round(ff * m_bar.MaxValue);
                }
                if (this.m_bar.ColorValue != colValue)
                {
                    this.m_bar.ColorValue = colValue;
                    Color color = m_bar.GetSelectColor();
                    HSV hsv = HSV.FromRGB(new RGB(color));
                    switch (m_bar.ComponentType)
                    {
                        case ColorType.R:
                            color = Color.FromArgb(color.A, colValue, color.G, color.B);
                            break;
                        case ColorType.G:
                            color = Color.FromArgb(color.A, color.R, colValue, color.B);
                            break;
                        case ColorType.B:
                            color = Color.FromArgb(color.A, color.R, color.G, colValue);
                            break;
                        case ColorType.A:
                            color = Color.FromArgb(colValue, color.R, color.G, color.B);
                            break;
                        case ColorType.H:
                            color = RGB.FromHSV(new HSV(colValue, hsv.S, hsv.V)).GetColor();
                            break;
                        case ColorType.S:
                            color = RGB.FromHSV(new HSV(hsv.H, colValue, hsv.V)).GetColor();
                            break;
                        default:
                            color = RGB.FromHSV(new HSV(hsv.H, hsv.S, colValue)).GetColor();
                            break;
                    }
                    hsv = HSV.FromRGB(new RGB(color));
                    this.m_bar.UpdateColor(color, hsv, true, false);
                    ColorSlider_ColorSelected(this, new ColorChangedEventArgs(color, true));
                }
            }
label_end:
            if (rewrite)
            {
                this.WriteText();
            }
            obj.Invalidate();
        }
        #endregion

        #region プロパティ
        /// <summary>
        /// 選択カラー。
        /// </summary>
        public int ColorValue
        {
            get { return m_bar.ColorValue; }
        }

        /// <summary>
        /// 選択カラー。
        /// </summary>
        [Category("カスタム")]
        [Description("選択カラー。")]
        [DefaultValue(typeof(Color), "Black")]
        public Color Color
        {
            get { return m_bar.GetSelectColor(); }
            set
            {
                HSV hsv = HSV.FromRGB(new RGB(value));
                WriteText();
                m_bar.UpdateColor(value, hsv, true, false);
            }
        }

        /// <summary>
        /// RGBAのいずれか？
        /// </summary>
        [Category("カスタム")]
        [Description("RGBHSVのいずれか。")]
        [DefaultValue(ColorType.R)]
        public ColorType ComponentType
        {
            get { return m_bar.ComponentType; }
            set
            {
                m_bar.ComponentType = value;
                switch (value)
                {
                    case ColorType.R:
                        m_label.Text = "R";
                        break;
                    case ColorType.G:
                        m_label.Text = "G";
                        break;
                    case ColorType.B:
                        m_label.Text = "B";
                        break;
                    case ColorType.A:
                        m_label.Text = "A";
                        break;
                    case ColorType.H:
                        m_label.Text = "H";
                        break;
                    case ColorType.S:
                        m_label.Text = "S";
                        break;
                    case ColorType.V:
                        m_label.Text = "B"; // Photoshopの表記に合わせる
                        break;
                    default:
                        m_label.Text = "?";
                        break;
                }
            }
        }

        /// <summary>
        /// カラー値の表示方式。
        /// </summary>
        [Category("カスタム")]
        [Description("カラー値の表示方式。")]
        [DefaultValue(ValueMode.Uchar)]
        public ValueMode ValueMode
        {
            get { return m_mode; }
            set
            {
                m_mode = value;
                if (this.Created)
                {
                    WriteText();
                }
            }
        }

        /// <summary>
        /// テキストボックスの表示位置を右側に設定する。
        /// </summary>
        [Category("カスタム")]
        [Description("テキストボックスの表示位置を右側に設定する。")]
        [DefaultValue(false)]
        public bool TextBoxIsRight
        {
            get { return m_editRight; }
            set
            {
                m_editRight = value;
                this.SuspendLayout();
                if (value)
                {
                    this.m_txtValue.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
                    this.m_bar.Location = new System.Drawing.Point(20, 6);
                    this.m_txtValue.Location = new System.Drawing.Point(this.Size.Width - 52, 4);
                    InitializeLine();
                }
                else
                {
                    this.m_txtValue.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)));
                    this.m_bar.Location = new System.Drawing.Point(72, 6);
                    this.m_txtValue.Location = new System.Drawing.Point(16, 4);
                    InitializeLine();
                }
                this.ResumeLayout();
                this.Invalidate();
            }
        }

        /// <summary>
        /// テキストボックスの前景色
        /// </summary>
        [Category("カスタム")]
        [Description("テキストボックスの前景色を取得または設定します。")]
        public Color TextBoxForeColor
        {
            get { return m_txtValue.ForeColor; }
            set { m_txtValue.ForeColor = value; }
        }

        /// <summary>
        /// テキストボックスの背景色
        /// </summary>
        [Category("カスタム")]
        [Description("テキストボックスの背景色を取得または設定します。")]
        public Color TextBoxBackColor
        {
            get { return m_txtValue.BackColor; }
            set { m_txtValue.BackColor = value; }
        }

        /// <summary>
        /// テキストボックスのフォント
        /// </summary>
        [Category("カスタム")]
        [Description("テキストボックスのフォントを取得または設定します。")]
        public Font TextBoxFont
        {
            get { return m_txtValue.Font; }
            set { m_txtValue.Font = value; }
        }
        #endregion

        #region オーバーライド
        /// <summary>
        /// OnPaintのオーバーライド（ボーダーの装飾）。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            if (this.DesignMode)
            {
                InitializeLine();
            }

            Pen pen = this.Enabled ? Pens.Black : Pens.DarkGray;
            Graphics g = e.Graphics;
            ControlPaint.DrawBorder3D(g, m_BarWakuOut);
            g.DrawRectangle(pen, m_BarWakuIn);
        }

        /// <summary>
        /// デザインモード時は親を再描画
        /// </summary>
        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);
            InitializeLine();
            if (this.DesignMode)
            {
                Color color = this.m_bar.GetSelectColor();
                this.m_bar.UpdateColor(color, HSV.FromRGB(new RGB(color)), true, false);
                this.Invalidate(true);
            }
        }
        #endregion

        /// <summary>
        /// RGBスライダーでカラー選択されたときに実行される
        /// </summary>
        private void ColorSlider_ColorSelected(object sender, ColorChangedEventArgs e)
        {
            this.Invalidate(true);
            if (sender == m_bar)
            {
                WriteText();
            }
            // 更新
            if (ColorSelected != null)
            {
                ColorSelected(this, e);
            }
        }

        /// <summary>
        /// テキストボックスの変更
        /// </summary>
        private void WriteText()
        {
            if (this.ValueMode == ValueMode.Uchar)
            {
                this.m_txtValue.Text = m_bar.ColorValue.ToString();
            }
            else
            {
                this.m_txtValue.Text = ((float)m_bar.ColorValue / m_bar.MaxValue).ToString("F");
            }
        }

        /// <summary>
        /// ラインの再設定
        /// </summary>
        private void InitializeLine()
        {
            int x = this.m_bar.Location.X;
            int y = this.m_bar.Location.Y;
            int w = this.m_bar.Size.Width;
            int h = this.m_bar.Size.Height;
            m_BarWakuOut = new Rectangle(x - 3, y - 3, w + 7, h + 7);
            m_BarWakuIn = new Rectangle(x - 1, y - 1, w + 1, h + 1);
        }

        /// <summary>
        /// 指定されたカラーで描画するビットマップとカラーを更新する
        /// </summary>
        public void UpdateColor(Color color, HSV hsv, bool resetPoint, bool flag)
        {
            this.m_bar.UpdateColor(color, hsv, resetPoint, flag);
            WriteText();
            this.Invalidate();
        }
    }
}
