﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;

namespace NW4R.ColorPicker
{
    /// <summary>
    /// RGBA、HSVA直接編集パネルコントロールクラス。
    /// </summary>
    public sealed class ComponentPanel : UserControl
    {
        // 現在のカラー
        private Color _color = Color.Black;
        // ＨＳＶカラー
        private HSV _hsv;
        // 値モード
        private ValueMode _valueMode = ValueMode.Uchar;
        // カラーモード
        private ColorMode _colorMode = ColorMode.RGBA;

        private NW4R.ColorPicker.TextSliderControl m_rh;
        private NW4R.ColorPicker.TextSliderControl m_gs;
        private NW4R.ColorPicker.TextSliderControl m_bv;
        private NW4R.ColorPicker.TextSliderControl m_alpha;
        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ComponentPanel()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();

            // イベント、プロパティ設定
            this.m_rh.ColorSelected += new TextSliderControl.ColorSelectedHandler(ColorSlider_ColorSelected);
            this.m_gs.ColorSelected += new TextSliderControl.ColorSelectedHandler(ColorSlider_ColorSelected);
            this.m_bv.ColorSelected += new TextSliderControl.ColorSelectedHandler(ColorSlider_ColorSelected);
            this.m_alpha.ColorSelected += new TextSliderControl.ColorSelectedHandler(AlphaSlider_ColorSelected);
            _hsv = HSV.FromRGB(new RGB(_color));

            this.DoubleBuffered = true;
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.m_rh = new NW4R.ColorPicker.TextSliderControl();
            this.m_gs = new NW4R.ColorPicker.TextSliderControl();
            this.m_bv = new NW4R.ColorPicker.TextSliderControl();
            this.m_alpha = new NW4R.ColorPicker.TextSliderControl();
            this.SuspendLayout();
            //
            // m_rh
            //
            this.m_rh.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                | System.Windows.Forms.AnchorStyles.Right)));
            this.m_rh.Color = System.Drawing.Color.FromArgb(((System.Byte)(0)), ((System.Byte)(0)), ((System.Byte)(0)));
            this.m_rh.Location = new System.Drawing.Point(0, 0);
            this.m_rh.Name = "m_rh";
            this.m_rh.Size = new System.Drawing.Size(336, 32);
            this.m_rh.TabIndex = 0;
            //
            // m_gs
            //
            this.m_gs.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                | System.Windows.Forms.AnchorStyles.Right)));
            this.m_gs.Color = System.Drawing.Color.FromArgb(((System.Byte)(0)), ((System.Byte)(0)), ((System.Byte)(0)));
            this.m_gs.ComponentType = NW4R.ColorPicker.ColorType.G;
            this.m_gs.Location = new System.Drawing.Point(0, 24);
            this.m_gs.Name = "m_gs";
            this.m_gs.Size = new System.Drawing.Size(336, 32);
            this.m_gs.TabIndex = 1;
            //
            // m_bv
            //
            this.m_bv.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                | System.Windows.Forms.AnchorStyles.Right)));
            this.m_bv.Color = System.Drawing.Color.FromArgb(((System.Byte)(0)), ((System.Byte)(0)), ((System.Byte)(0)));
            this.m_bv.ComponentType = NW4R.ColorPicker.ColorType.B;
            this.m_bv.Location = new System.Drawing.Point(0, 48);
            this.m_bv.Name = "m_bv";
            this.m_bv.Size = new System.Drawing.Size(336, 32);
            this.m_bv.TabIndex = 2;
            //
            // m_alpha
            //
            this.m_alpha.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                | System.Windows.Forms.AnchorStyles.Right)));
            this.m_alpha.Color = System.Drawing.Color.FromArgb(((System.Byte)(0)), ((System.Byte)(0)), ((System.Byte)(0)));
            this.m_alpha.ComponentType = NW4R.ColorPicker.ColorType.A;
            this.m_alpha.Location = new System.Drawing.Point(0, 72);
            this.m_alpha.Name = "m_alpha";
            this.m_alpha.Size = new System.Drawing.Size(336, 24);
            this.m_alpha.TabIndex = 3;
            //
            // ComponentPanel
            //
            this.Controls.Add(this.m_alpha);
            this.Controls.Add(this.m_bv);
            this.Controls.Add(this.m_gs);
            this.Controls.Add(this.m_rh);
            this.Name = "ComponentPanel";
            this.Size = new System.Drawing.Size(336, 104);
            this.ResumeLayout(false);

        }
        #endregion

        #region イベント

        /// <summary>
        /// カラーが変更されたときに発生します。
        /// </summary>
        [Category("カスタム")]
        [Description("カラーが変更されたときに発生します。")]
        public event ColorChangedEventHandler ColorChanged;

        #endregion

        #region プロパティ

        /// <summary>
        /// 選択カラー。
        /// </summary>
        [Description("選択カラー。")]
        [DefaultValue(typeof(Color), "Black")]
        public Color Color
        {
            get { return _color; }
            set
            {
                _color = value;
                _hsv = HSV.FromRGB(new RGB(_color));
                if (this.Created)
                {
                    ColorChangedEventArgs arg = new ColorChangedEventArgs(_color, true);
                    ColorSlider_ColorSelected(this, arg);
                    AlphaSlider_ColorSelected(this, arg);
                }
            }
        }

        /// <summary>
        /// 選択カラー。
        /// </summary>
        [Description("選択カラー。")]
        [DefaultValue(typeof(Color), "Black")]
        public HSV HSV
        {
            get
            {
                return new HSV(this.m_rh.ColorValue, this.m_gs.ColorValue, this.m_bv.ColorValue);
            }
        }

        /// <summary>
        /// カラー値の表示方式。
        /// </summary>
        [Description("カラー値の表示方式。")]
        [DefaultValue(ValueMode.Uchar)]
        public ValueMode ValueMode
        {
            get { return _valueMode; }
            set
            {
                _valueMode = value;
                this.m_rh.ValueMode = value;
                this.m_gs.ValueMode = value;
                this.m_bv.ValueMode = value;
                this.m_alpha.ValueMode = value;
                if (this.Created)
                {
                    this.m_rh.UpdateColor(_color, _hsv, true, true);
                    this.m_gs.UpdateColor(_color, _hsv, true, true);
                    this.m_bv.UpdateColor(_color, _hsv, true, true);
                    this.m_alpha.UpdateColor(_color, _hsv, true, true);
                }
            }
        }

        /// <summary>
        /// カラー成分のモード。
        /// </summary>
        [Description("カラー成分のモード。")]
        [DefaultValue(ColorMode.RGBA)]
        public ColorMode ColorMode
        {
            get { return _colorMode; }
            set
            {
                _colorMode = value;
                this.SetColor(_color);
                SetControlVisibility();
                if (this.Created)
                {
                    InitializePosition(this.Size);
                    this.m_rh.UpdateColor(_color, _hsv, true, true);
                    this.m_gs.UpdateColor(_color, _hsv, true, true);
                    this.m_bv.UpdateColor(_color, _hsv, true, true);
                    this.m_alpha.UpdateColor(_color, _hsv, true, true);
                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// テキストボックスを右側に表示するかどうかのフラグ。
        /// </summary>
        [Description("テキストボックスを右側に表示するかどうかを示します。")]
        [DefaultValue(false)]
        public bool TextBoxIsRight
        {
            get { return this.m_rh.TextBoxIsRight; }
            set
            {
                this.m_rh.TextBoxIsRight = value;
                this.m_gs.TextBoxIsRight = value;
                this.m_bv.TextBoxIsRight = value;
                this.m_alpha.TextBoxIsRight = value;
            }
        }

        #endregion

        #region オーバーライド
        /// <summary>
        /// OnHandleCreatedのオーバーライド（コントロールの初期化）。
        /// </summary>
        /// <param name="e"></param>
        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated (e);

            InitializeControl();
        }

        /// <summary>
        /// デザインモード時は親を再描画
        /// </summary>
        /// <param name="e"></param>
        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged (e);
            InitializePosition( this.Size );
            if( this.DesignMode )
            {
                this.m_rh.UpdateColor( this._color, _hsv, true, false );
                this.m_gs.UpdateColor( this._color, _hsv, true, false );
                this.m_bv.UpdateColor( this._color, _hsv, true, false );
                this.Invalidate(true);
            }
        }

        #endregion

        /// <summary>
        /// イベント発行せずに、現在のカラーを変更する。
        /// </summary>
        public void SetColor(Color color)
        {
            _color = color;
            _hsv = HSV.FromRGB(new RGB(color));
            UpdateColor(_color, true);
            this.Invalidate();
        }

        /// <summary>
        /// RGBスライダーでカラー選択されたときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ColorSlider_ColorSelected(object sender, ColorChangedEventArgs e)
        {
            Color select = Color.FromArgb(255, e.Color);
            if (!m_rh.Color.Equals(select) ||
                !m_gs.Color.Equals(select) ||
                !m_bv.Color.Equals(select))
            {
                UpdateColor(sender);
            }

            this.Invalidate(true);
            // 更新
            if (sender != this)
            {
                ChangeColor(sender, new ColorChangedEventArgs(Color.FromArgb(_color.A, e.Color), e.EditFixed));
            }
        }

        /// <summary>
        /// Alphaスライダーでカラー選択されたときに実行される
        /// </summary>
        private void AlphaSlider_ColorSelected(object sender, ColorChangedEventArgs e)
        {
            if (_color.A != e.Color.A)
            {
                m_alpha.Invalidate();
            }
            if (sender != this)
            {
                ChangeColor(new ColorChangedEventArgs(Color.FromArgb(e.Color.A, _color), e.EditFixed));
            }
        }

        /// <summary>
        /// RGBカラーが変更されたときに呼ばれる
        /// </summary>
        private void ChangeColor(object sender, ColorChangedEventArgs e)
        {
            _color = e.Color;
            _hsv = HSV.FromRGB(new RGB(e.Color));
            UpdateColor(sender);
            if (ColorChanged != null)
            {
                ColorChanged(this, e);
            }
        }

        /// <summary>
        /// Alphaカラーが変更されたときに呼ばれる
        /// </summary>
        private void ChangeColor(ColorChangedEventArgs e)
        {
            _color = e.Color;
            _hsv = HSV.FromRGB(new RGB(e.Color));
            UpdateColor(this.m_alpha);
            if (ColorChanged != null)
            {
                ColorChanged(this, e);
            }
        }

        /// <summary>
        /// プロパティ値からコントロールの設定を初期化
        /// </summary>
        private void InitializeControl()
        {
            InitializePosition(this.Size);
            this.m_rh.UpdateColor(_color, _hsv, true, true);
            this.m_gs.UpdateColor(_color, _hsv, true, true);
            this.m_bv.UpdateColor(_color, _hsv, true, true);
            this.m_alpha.UpdateColor(_color, _hsv, true, true);

            SetControlVisibility();
            this.Invalidate();
        }

        /// <summary>
        /// ラインの再設定
        /// </summary>
        private void InitializePosition(Size size)
        {
            //	小さくしてるので無視
            if (size.Height < 104)
                return;
            int add = (size.Height - 104) / 3;
            if (add < 0)
                return;
            add += 24;

            int y = 0;
            if (_colorMode == ColorMode.A)
            {
                y = add * 3;
                add = -add;
            }
            this.m_rh.Location = new Point(this.m_rh.Location.X, y);
            y += add;
            this.m_gs.Location = new Point(this.m_gs.Location.X, y);
            y += add;
            this.m_bv.Location = new Point(this.m_bv.Location.X, y);
            y += add;
            this.m_alpha.Location = new Point(this.m_bv.Location.X, y);
        }

        /// <summary>
        /// コントロールに設定を反映。
        /// </summary>
        private void SetControlVisibility()
        {
            foreach (Control ctrl in this.Controls)
            {
                ctrl.Enabled = true;
            }
            if (_colorMode == ColorMode.HSVA || _colorMode == ColorMode.HSV)
            {
                this.m_rh.ComponentType = ColorType.H;
                this.m_gs.ComponentType = ColorType.S;
                this.m_bv.ComponentType = ColorType.V;
            }
            else
            {
                this.m_rh.ComponentType = ColorType.R;
                this.m_gs.ComponentType = ColorType.G;
                this.m_bv.ComponentType = ColorType.B;
            }
            switch (_colorMode)
            {
                case ColorMode.HSVA:
                case ColorMode.RGBA:
                    this.m_rh.Visible = true;
                    this.m_gs.Visible = true;
                    this.m_bv.Visible = true;
                    this.m_alpha.Visible = true;
                    break;
                case ColorMode.HSV:
                case ColorMode.RGB:
                    this.m_rh.Visible = true;
                    this.m_gs.Visible = true;
                    this.m_bv.Visible = true;
                    this.m_alpha.Visible = false;
                    this.m_alpha.Enabled = false;
                    break;
                case ColorMode.A:
                    this.m_rh.Visible = false;
                    this.m_gs.Visible = false;
                    this.m_bv.Visible = false;
                    this.m_rh.Enabled = false;
                    this.m_gs.Enabled = false;
                    this.m_bv.Enabled = false;
                    this.m_alpha.Visible = true;
                    break;
            }
        }

        /// <summary>
        /// 指定されたカラーで描画するビットマップとカラーを更新する
        /// </summary>
        private void UpdateColor(object sender)
        {
            HSV hsv = new HSV(this.m_rh.ColorValue, this.m_gs.ColorValue, this.m_bv.ColorValue);
            if (!sender.Equals(this.m_rh))
                this.m_rh.UpdateColor(_color, hsv, true, false);
            if (!sender.Equals(this.m_gs))
                this.m_gs.UpdateColor(_color, hsv, true, false);
            if (!sender.Equals(this.m_bv))
                this.m_bv.UpdateColor(_color, hsv, true, false);
            if (!sender.Equals(this.m_alpha))
                this.m_alpha.UpdateColor(_color, hsv, true, false);
            this.Invalidate();
        }

        /// <summary>
        /// 指定されたカラーで描画するビットマップとカラーを更新する
        /// </summary>
        public void UpdateColor(Color color, bool resetPoint)
        {
            HSV hsv = HSV.FromRGB(new RGB(color));
            _color = color;
            this.m_rh.UpdateColor(_color, hsv, resetPoint, false);
            this.m_gs.UpdateColor(_color, hsv, resetPoint, false);
            this.m_bv.UpdateColor(_color, hsv, resetPoint, false);
            this.m_alpha.UpdateColor(_color, hsv, resetPoint, false);
            this.Invalidate();
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス前景色を設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="color">前景色</param>
        public void SetTextBoxForeColor(ColorChannel ch, Color color)
        {
            if (ch == ColorChannel.R)
                this.m_rh.TextBoxForeColor = color;
            else if (ch == ColorChannel.G)
                this.m_gs.TextBoxForeColor = color;
            else if (ch == ColorChannel.B)
                this.m_bv.TextBoxForeColor = color;
            else
                this.m_alpha.TextBoxForeColor = color;
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス前景色を取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>前景色</returns>
        public Color GetTextBoxForeColor(ColorChannel ch)
        {
            if (ch == ColorChannel.R)
                return this.m_rh.TextBoxForeColor;
            else if (ch == ColorChannel.G)
                return this.m_gs.TextBoxForeColor;
            else if (ch == ColorChannel.B)
                return this.m_bv.TextBoxForeColor;
            else
                return this.m_alpha.TextBoxForeColor;
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス背景色を設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="color">背景色</param>
        public void SetTextBoxBackColor(ColorChannel ch, Color color)
        {
            if (ch == ColorChannel.R)
                this.m_rh.TextBoxBackColor = color;
            else if (ch == ColorChannel.G)
                this.m_gs.TextBoxBackColor = color;
            else if (ch == ColorChannel.B)
                this.m_bv.TextBoxBackColor = color;
            else
                this.m_alpha.TextBoxBackColor = color;
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス背景色を取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>背景色</returns>
        public Color GetTextBoxBackColor(ColorChannel ch)
        {
            if (ch == ColorChannel.R)
                return this.m_rh.TextBoxBackColor;
            else if (ch == ColorChannel.G)
                return this.m_gs.TextBoxBackColor;
            else if (ch == ColorChannel.B)
                return this.m_bv.TextBoxBackColor;
            else
                return this.m_alpha.TextBoxBackColor;
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックスフォントを設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="font">フォント</param>
        public void SetTextBoxFont(ColorChannel ch, Font font)
        {
            if (ch == ColorChannel.R)
                this.m_rh.TextBoxFont = font;
            else if (ch == ColorChannel.G)
                this.m_gs.TextBoxFont = font;
            else if (ch == ColorChannel.B)
                this.m_bv.TextBoxFont = font;
            else
                this.m_alpha.TextBoxFont = font;
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックスフォントを取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>フォント</returns>
        public Font GetTextBoxFont(ColorChannel ch)
        {
            if (ch == ColorChannel.R)
                return this.m_rh.TextBoxFont;
            else if (ch == ColorChannel.G)
                return this.m_gs.TextBoxFont;
            else if (ch == ColorChannel.B)
                return this.m_bv.TextBoxFont;
            else
                return this.m_alpha.TextBoxFont;
        }
    }
}
