﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using NintendoWare.Font;
using System.IO;
using System.Xml.Serialization;
using System.Reflection;
using System.Diagnostics;
namespace FontConverterNW4FProxy
{
    class ProxyProgram
    {
        static int Main(string[] args)
        {
            ConvertImageFontCache cache = null;
            try
            {
                CmdLine cmd = new CmdLine();

                bool readFfnt = false;
                cmd.AnalizedAnOption += (o, e) =>
                {
                    switch (e.Option)
                    {
                        case Program.OptionInput:
                        case Program.OptionInputFile:
                            // イメージ変換設定の場合
                            if (!readFfnt &&
                                cmd.Exists(Program.OptionInput) && Program._tcscmp(cmd[Program.OptionInput], Program.InputImageFontConvertSettings) == 0 &&
                                cmd.Exists(Program.OptionInputFile))
                            {
                                readFfnt = true;

                                // オプションの追加
                                Program.AddImageFontConvertOptions(cmd);
                            }
                            break;
                    }
                };

                cmd.Analyze(Environment.CommandLine, Program.Options, Program.Options.Length);

                // イメージ変換設定の場合
                if (readFfnt)
                {
                    // キャッシュを有効にする
                    if (!cmd.Exists(Program.OptionNoCache) && Program._tcscmp(cmd[Program.OptionOutput], Program.OutputFont) == 0)
                    {
                        cache = new ConvertImageFontCache();
                        cache.AddCacheKey("NW4FProxy version 1");
                    }
                }

                // キャッシュ設定読み込みのためにプラットフォームを設定する
                // Proxy のプラットフォームはNW4F固定
                bool platformInitResult = ConverterEnvironment.InitializeTargetPlatform("Generic", "linear", "Win", cache);
                if (!platformInitResult)
                {
                    return 1;
                }

                var newArgs = new List<string>();

                foreach (var optionDef in NW4FOptions.Options)
                {
                    if (optionDef.Name == Program.OptionOutputFile)
                    {
                        continue;
                    }

                    // キャッシュキーに追加
                    if (cache != null)
                    {
                        if (!Program.skipCacheSourceOptions.Contains(optionDef.Name))
                        {
                            cache.AddCacheKey(optionDef.Name);
                        }
                    }
                    StringList optionValue = cmd.FindOptionValueOf(optionDef.Name);
                    if (optionValue != null)
                    {
                        if (cache != null)
                        {
                            if (!Program.skipCacheSourceOptions.Contains(optionDef.Name))
                            {
                                cache.AddCacheKey(optionValue.Count);

                                foreach (var value in optionValue)
                                {
                                    cache.AddCacheKey(value);
                                }
                            }
                        }

                        newArgs.Add("-" + optionDef.Name);
                        if (optionValue.Any())
                        {
                            var last = optionValue.Last();
                            newArgs.Add(last);
                        }
                    }
                    else
                    {
                        // キャッシュキーに追加
                        if (cache != null)
                        {
                            if (!Program.skipCacheSourceOptions.Contains(optionDef.Name))
                            {
                                cache.AddCacheKey(0);
                            }
                        }
                    }
                }

                // キャッシュにキー追加
                if (cache != null)
                {
                    cache.AddCacheKeyFromFile(cmd[Program.OptionInputFile]);
                    cache.AddCacheKeyFromFile(cmd[Program.OptionInputOrder]);
                    cache.AddCacheKeyFromFile(Path.ChangeExtension(cmd[Program.OptionInputFile], ".bffkn"));

                    // キャッシュキーに追加
                    if (cmd.Exists(Program.OptionFilter))
                    {
                        cache.AddCacheKeyFromFile(cmd[Program.OptionFilter]);
                    }
                    else
                    {
                        cache.AddCacheKey(0);
                    }
                }


                var cacheFilePath = cache != null ? cache.GetCacheFilePath() : null;
                var outputPath = cmd[Program.OptionOutputFile];
                if (cacheFilePath != null)
                {
                    FontIO.ValidateOutputPath(outputPath);

                    // キャッシュからコピー
                    cache.WaitCacheFileMutex();
                    if (cache.CopyToOutput(outputPath))
                    {
                        cache.ClearCache();
                        Console.Write(string.Format("------- copied font from cache {0}\n", cacheFilePath));
                        return 0;
                    }

                    // オプションの差し替え
                    newArgs.Add("-" + NW4FOptions.OptionOutputFile);
                    newArgs.Add(Path.ChangeExtension(cacheFilePath, LibFormat.ExtensionFont));
                }
                else
                {
                    newArgs.Add("-" + NW4FOptions.OptionOutputFile);
                    newArgs.Add(outputPath);
                }

                Console.Error.WriteLine(ConverterEnvironment.PlatformPreference.ConverterPlatformName);

                var originalExe = Path.Combine(ConverterEnvironment.PlatformDetails.NwToolsRootPath, ConverterEnvironment.PlatformPreference.OriginalFontConverterConsolePath);
                Console.Error.WriteLine(originalExe);
                var arguments = string.Join(" ", newArgs.Select(x => x.Contains(" ") ? "\"" + x + "\"" : x));
                Console.Error.WriteLine(arguments);
                var info = new ProcessStartInfo(originalExe, arguments);
                info.CreateNoWindow = true;
                info.UseShellExecute = false;
                info.RedirectStandardError = true;
                info.StandardErrorEncoding = Encoding.Default;
                info.RedirectStandardOutput = true;
                info.StandardOutputEncoding = Encoding.Default;

                Process process = new Process
                {
                    StartInfo = info
                };

                process.OutputDataReceived += (s, e) => { Console.Out.Write(e.Data); };
                process.ErrorDataReceived += (s, e) => { Console.Error.Write(e.Data); };

                process.Start();

                process.BeginOutputReadLine();
                process.BeginErrorReadLine();

                process.WaitForExit();

                var exitCode = process.ExitCode;

                if (exitCode == 0 && cacheFilePath != null)
                {
                    // 拡張子をキャッシュ用ファイルのものに変更
                    File.Move(Path.ChangeExtension(cacheFilePath, LibFormat.ExtensionFont), cacheFilePath);

                    // ファイルはキャッシュ側に生成されるので、キャッシュからコピー
                    if (!cache.CopyToOutput(outputPath))
                    {
                        Console.Error.Write("Error: failed to copy font from cache {0}\n", cacheFilePath);
                        return 1;
                    }
                    Console.Write(string.Format("------- copied font from cache {0}\n", cacheFilePath));

                    // キャッシュの整理
                    cache.ClearCache();
                }

                return process.ExitCode;

            }
            catch (Exception e)
            {
                Console.Error.WriteLine(Strings.IDS_ERROR_MSG, e.ToString());
                return 2;
            }
            finally
            {
                if (cache != null)
                {
                    cache.Dispose();
                }
            }
        }
    }
}
