﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NintendoWare.Font.Runtime;
using System.Reflection;
using System.IO;

namespace NintendoWare.Font
{
    public class PlatformDetailsGeneric : IPlatformDetails
    {
        /// <summary>
        /// UI に表示する名前です。
        /// </summary>
        public string UIPlatformName { get { return "Win"; } }

        /// <summary>
        /// 名前です。
        /// </summary>
        public string PlatformName { get { return "Generic"; } }

        /// <summary>
        /// CAFE内蔵フォントで利用される、bfttf 形式の読み込みをサポートするかどうか
        /// </summary>
        public bool BfttfEnabled { get { return true; } }

        /// <summary>
        /// スケーラブルフォントが利用できるか？
        /// </summary>
        public bool ScfontEnabled { get { return true; } }

        /// <summary>
        /// 4bit フォーマット(A4 LA4)が利用できるか？
        /// </summary>
        public bool CanHWSupports4BitFormat { get { return false; } }

        /// <summary>
        /// 入力フォントのテクスチャが天地反転方向で格納されるか
        /// </summary>
        public bool IsInputFontReversed { get { return true; } }

        /// <summary>
        /// 入力フォントバージョン
        /// </summary>
        public uint InputFontVersion { get { return RtConsts.MakeVersion(4, 1, 0, 0); } }

        /// <summary>
        /// 出力フォントバージョン
        /// </summary>
        public uint OutputFontVersion { get { return RtConsts.MakeVersion(4, 1, 0, 0); } }

        /// <summary>
        ///
        /// </summary>
        public IEnumerable<WindowsFontInputFormatDesc> GetWindowsFontInputFormatList()
        {
            yield return new WindowsFontInputFormatDesc(4, false, "A4", string.Format(Strings.IDS_INWIN_FORMAT_I4, "A4( --> BC4 )") + string.Format("[{0}]", Strings.NotRecommended), GlyphImageFormat.A4);
            yield return new WindowsFontInputFormatDesc(6, false, "A6", string.Format(Strings.IDS_INWIN_FORMAT_I8, "A8( --> BC4 )"), GlyphImageFormat.A8);
            yield return new WindowsFontInputFormatDesc(8, false, "A8", string.Format(Strings.IDS_INWIN_FORMAT_I8, "A8( --> BC4 )"), GlyphImageFormat.A8);
            yield return new WindowsFontInputFormatDesc(4, true, "LA4", string.Format(Strings.IDS_INWIN_FORMAT_I4, "LA4( --> BC5 )") + string.Format("[{0}]", Strings.NotRecommended), GlyphImageFormat.LA4);
            yield return new WindowsFontInputFormatDesc(6, true, "LA6", string.Format(Strings.IDS_INWIN_FORMAT_I8, "LA8( --> BC5 )"), GlyphImageFormat.LA8);
            yield return new WindowsFontInputFormatDesc(8, true, "LA8", string.Format(Strings.IDS_INWIN_FORMAT_I8, "LA8( --> BC5 )"), GlyphImageFormat.LA8);
        }

        /// <summary>
        ///
        /// </summary>
        public IEnumerable<ImageInputFormatDesc> GetImageInputFormatFormatList()
        {
            yield return new ImageInputFormatDesc(8, 0, "A4( --> BC4 )", GlyphImageFormat.A4); // BC4として扱う ので 8bit
            yield return new ImageInputFormatDesc(8, 0, "A8( --> BC4 )", GlyphImageFormat.A8);
            yield return new ImageInputFormatDesc(16, 8, "LA4( --> BC5 )", GlyphImageFormat.LA4); // BC5として扱う ので 16-8bit
            yield return new ImageInputFormatDesc(16, 8, "LA8( --> BC5 )", GlyphImageFormat.LA8);

            yield return new ImageInputFormatDesc(8, 8, "LA4Packed ( --> BC4-Packed)", GlyphImageFormat.LA4Packed);
            yield return new ImageInputFormatDesc(8, 8, "LA4Packed ( --> BC4-Packed)", GlyphImageFormat.LA4Packed, GlyphImageFormat.LA4PackedPassThrough);
            yield return new ImageInputFormatDesc(8, 8, "LA4PackedNoCompress  ( --> A8-Packed)", GlyphImageFormat.LA4PackedNoCompress);
            yield return new ImageInputFormatDesc(8, 8, "LA4PackedNoCompress  ( --> A8-Packed)", GlyphImageFormat.LA4PackedNoCompress, GlyphImageFormat.LA4PackedNoCompressPassThrough);
            yield return new ImageInputFormatDesc(16, 0, "RGB565( --> BC1)", GlyphImageFormat.RGB565);

            yield return new ImageInputFormatDesc(16, 3, "RGB5A3( --> BC3)", GlyphImageFormat.RGB5A3);

            yield return new ImageInputFormatDesc(32, 8, "RGBA8( --> RGBA8)", GlyphImageFormat.RGBA8);

            yield return new ImageInputFormatDesc(32, 8, "RGBA8_SRGB( --> RGBA8)", GlyphImageFormat.RGBA8_SRGB);
            yield return new ImageInputFormatDesc(16, 0, "RGB565_SRGB( --> BC1)", GlyphImageFormat.RGB565_SRGB);
            yield return new ImageInputFormatDesc(16, 3, "RGB5A3_SRGB( --> BC3)", GlyphImageFormat.RGB5A3_SRGB);

            yield return new ImageInputFormatDesc(32, 8, "RGBA8_BC7( --> BC7)", GlyphImageFormat.RGBA8_BC7);
            yield return new ImageInputFormatDesc(32, 8, "RGBA8_BC7_SRGB( --> BC7_SRGB)", GlyphImageFormat.RGBA8_BC7_SRGB);
        }

        public bool IsInputFontLittleEndian
        {
            get { return true; }
        }

        public bool IsOutputFontLittleEndian
        {
            get { return true; }
        }

        public void CUISetup(string[] args)
        {
        }

        public NitroFontReader GetFontReader(string inputFilePath)
        {
            return new GenericFontReader(inputFilePath);
        }

        public NitroFontWriter GetFontWriter(
            string file,
            string groupsFile,
            GlyphDataType type,
            uint alter,
            int linefeed,
            int width,
            int left,
            int right,
            CharEncoding encoding,
            int sheetWidth,
            int sheetHeight,
            int sheetPixels,
            bool isOutputTargetLittleEndian,
            bool isOutKerningData)
        {
            return new GenericFontWriter(
                file, groupsFile, type, alter, linefeed, width, left, right,
                encoding, sheetWidth, sheetHeight, sheetPixels, isOutputTargetLittleEndian, isOutKerningData);
        }

        static private readonly string _NwToolsRootPath;
        public string NwToolsRootPath { get { return _NwToolsRootPath; } }


        static PlatformDetailsGeneric()
        {
            // テスト：特定のフォルダ構成が検出された場合は処理を切り替えます。
            string executingFileName = Assembly.GetExecutingAssembly().Location;
            string executingFolder = Path.GetDirectoryName(executingFileName);
            string fontConverterFolderParentPath = Path.GetDirectoryName(executingFolder);
            if (fontConverterFolderParentPath.Contains("Tools\\Graphics"))
            {
                if (fontConverterFolderParentPath.EndsWith("Tools\\Graphics"))
                {
                    // パッケージフォルダ構成での動作
                    _NwToolsRootPath = fontConverterFolderParentPath;
                }
                else
                {
                    // リポジトリフォルダ構成での動作
                    // 親フォルダをたどって基点を見つけ、Toolフォルダを決定する。
                    string parentDir = fontConverterFolderParentPath;
                    try
                    {
                        while (!parentDir.EndsWith("Programs"))
                        {
                            parentDir = Path.GetDirectoryName(parentDir);
                        }
                        parentDir = Path.GetDirectoryName(parentDir);
                    }
                    catch
                    {
                        // 想定外のフォルダ構成だった。
                        // 適当な値を設定しておく。
                        parentDir = fontConverterFolderParentPath;
                    }

                    _NwToolsRootPath = Path.Combine(parentDir, "Tools\\Graphics");
                }
            }
        }

    }
}

