﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using NintendoWare.Font.Win32;

    public class WinFontReader : FontReader
    {
        private const int UnicodeEnumBegin = 0x0000;
        private const int UnicodeEnumEnd = 0xFFFD + 1;
        private readonly string fontPath;
        private readonly string fontName;
        private readonly int fontSize;
        private readonly int avgWidth;
        private readonly int weight; // 0 - 900(Normal は 400)
        private readonly int bpp;
        private readonly int fixedWidth;
        private readonly WidthType widthType;
        private readonly bool isSoftAA;
        private readonly bool hasAlpha;
        private readonly bool useScfontKerning;
        private readonly bool isTemporaryFontLoaded;
        private readonly string forcedFontFaceName;

        public WinFontReader(string path, string name, int size, int weight, int avgwidth, int bpp, bool hasAlpha, WidthType wt, int fixedWidth, bool isSoftAA, bool useScfontKerning, bool isTemporaryFontLoaded, string forcedFontFaceName)
        {
            this.fontPath = path;
            this.fontName = name;
            this.fontSize = size;
            this.avgWidth = avgwidth;
            this.weight = weight;
            this.bpp = bpp;
            this.hasAlpha = hasAlpha;
            this.widthType = wt;
            this.fixedWidth = fixedWidth;
            this.isSoftAA = isSoftAA;
            this.useScfontKerning = useScfontKerning;
            this.isTemporaryFontLoaded = isTemporaryFontLoaded;
            this.forcedFontFaceName = forcedFontFaceName;
        }

        public override void ReadFontData(FontData fontData, CharFilter filter)
        {
            const byte FALSE = 0;

            var isUnicode = true;

            var lf = new LOGFONT();
            var list = fontData.GetGlyphList();

            Letterer letterer = null;
            Letterer letterer_for_kerning = null;

            // プログレスバー設定
            ProgressControl.GetInstance().SetStatusString(Strings.IDS_STATUS_READ_WIN);
            ProgressControl.GetInstance().ResetProgressBarPos();
            ProgressControl.GetInstance().SetProgressBarMax(0x10000);

            // LOGFONT構成
            lf.lfHeight = -this.fontSize;
            lf.lfWidth = this.avgWidth;
            lf.lfEscapement = 0;
            lf.lfOrientation = 0;
            lf.lfWeight = this.weight;
            lf.lfItalic = FALSE;
            lf.lfUnderline = FALSE;
            lf.lfStrikeOut = FALSE;
            lf.lfCharSet = GdiDef.DEFAULT_CHARSET;
            lf.lfOutPrecision = GdiDef.OUT_DEFAULT_PRECIS;
            lf.lfClipPrecision = GdiDef.CLIP_DEFAULT_PRECIS;
            lf.lfQuality = GdiDef.ANTIALIASED_QUALITY;
            lf.lfPitchAndFamily = GdiDef.DEFAULT_PITCH | GdiDef.FF_DONTCARE;
            lf.lfFaceName = this.fontName;

            // グリフ作成
            try
            {
                // Letterer作成
                if (this.bpp <= 1)
                {
                    letterer = new RasterLetterer();
                    letterer_for_kerning = new RasterLetterer();
                }
                else
                {
                    VectorLetterer pvl = new VectorLetterer();
                    pvl.SetVectorOption(this.isSoftAA);
                    letterer = pvl;
                    letterer_for_kerning = new VectorLetterer();
                }

                letterer.SetOption(isUnicode, this.bpp, this.hasAlpha, this.widthType, this.fixedWidth, this.useScfontKerning);
                letterer.Create(ref lf, fontPath, forcedFontFaceName);
                letterer_for_kerning.SetOption(isUnicode, this.bpp, this.hasAlpha, this.widthType, this.fixedWidth, this.useScfontKerning);
                letterer_for_kerning.Create(ref lf, fontPath, forcedFontFaceName);
                /*
                        {
                            const int fontHeight = pLetterer->GetFontHeight();
                            const int ascent     = pLetterer->GetAscent();
                            const int descent    = fontHeight - ascent;

                            pFont->SetHeight(ascent, descent);
                        }
                */

                // 描画
                for (int i = UnicodeEnumBegin; i < UnicodeEnumEnd; ++i)
                {
                    uint c = (uint)i;

                    ProgressControl.GetInstance().StepProgressBar();

                    if (!filter.IsFiltered(c))
                    {
                        var glyph = new Glyph();

                        if (letterer.LetterChar(glyph, c))
                        {
                            list.AddGlyph(glyph, c);
                        }
                        else
                        {
                            glyph = null;
                        }
                    }
                }

                // カーニング情報を取得、フィルタリングも行う
                fontData.KerningPairs = FilterKerningPairs(letterer_for_kerning.GetKerningPairs(), filter);
            }
            finally
            {
                if (letterer != null)
                {
                    letterer.Dispose();
                }
            }

            if (this.hasAlpha)
            {
                if (this.bpp <= 4)
                {
                    fontData.OutputFormat = GlyphImageFormat.LA4;
                }
                else
                {
                    fontData.OutputFormat = GlyphImageFormat.LA8;
                }
            }
            else
            {
                if (this.bpp <= 4)
                {
                    fontData.OutputFormat = GlyphImageFormat.A4;
                }
                else
                {
                    fontData.OutputFormat = GlyphImageFormat.A8;
                }
            }
        }

        public override void ValidateInput()
        {
            // Font Name
            {
                if (this.fontName.Length == 0)
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_FONT_NOT_SPECIFIED);
                }

                // プロセス内に一時的にフォントが読み込まれている場合は、ファイルは存在しませんし、インストールフォントとして列挙もされないのでチェックを行いません。
                if (!this.isTemporaryFontLoaded)
                {
                    if (!System.IO.File.Exists(this.fontPath) && !GlCm.GetInstalledFontNames().ContainsKey(this.fontName))
                    {
                        throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_FONT_NOT_EXISTS, this.fontName);
                    }
                }
            }

            // Font Size
            if (this.fontSize <= 0)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_UNDERMIN_FONTSIZE, this.fontSize);
            }

            // Levels of Gray
            if (this.bpp < 1 || 8 < this.bpp)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_OUTOFRANGE_COLORBPP, this.bpp);
            }

            // Glyph Width
            if ((int)this.widthType < 0 || 3 < (int)this.widthType)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_INVALID_WIDTH_TYPE, this.widthType);
            }

            // Fixed Width
            if (this.widthType == WidthType.Fixed)
            {
                if (this.fixedWidth <= 0)
                {
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_UNDERMIN_FIXEDWIDTH, this.fixedWidth);
                }
            }
        }
    }
}
