﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using System.Runtime.InteropServices;
    using NintendoWare.Font.Win32;
    using HBITMAP = System.IntPtr;
    using HBRUSH = System.IntPtr;
    using HDC = System.IntPtr;

    public class RasterLetterer : Letterer
    {
        private const uint ColorBlack = 0x00000000;
        private const uint ColorWhite = 0x00FFFFFF;
        private const uint ColorBg = 0x00FF0000;

        private HDC masterDCHandle;
        private HBITMAP oldBmpHandle;
        private HBITMAP bmpHandle;

        public RasterLetterer()
        {
            this.masterDCHandle = IntPtr.Zero;
            this.oldBmpHandle = IntPtr.Zero;
            this.bmpHandle = IntPtr.Zero;
        }

        //---------------------------------------------------------
        // LetterChar
        //---------------------------------------------------------
        public override bool LetterChar(Glyph glyph, uint c)
        {
            if (!IsGlyphExists(c))
            {
                return false;
            }

            // clear dc image
            {
                RECT r = new RECT(0, 0, CellSize, CellSize);
                LOGBRUSH lb = new LOGBRUSH(BS.SOLID, Gdi.RGB(255, 0, 0), UIntPtr.Zero);

                var brush = Gdi.CreateBrushIndirect(lb);
                Gdi.FillRect(DeviceContext, ref r, brush);
                Gdi.DeleteObject(brush);
            }

            // draw glyph
            {
                var isSuccess = TextOutC(DeviceContext, 0, 0, c, IsUnicode);

                if (!isSuccess)
                {
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_TEXTOUT);
                }
            }

            try
            {
                int dwidth = GlCm.ROUND_UP(CellSize, 4);
                var size = CellSize * dwidth;
                var srcBmp = new byte[size];
                var dstBmp = new byte[size];
                IndexImage image = new IndexImage();
                GlyphInfo info = new GlyphInfo();

                // get raw bmp from GDI BMP
                // srcBmp: bottomup
                this.GetGlyphBmp(srcBmp, size, info);

                // 256 system color palette => (1 << m_bpp) monochrome palette
                // srcBmp: bottomup
                // dstBmp: topdown
                this.AdjustBmp(dstBmp, srcBmp, CellSize, CellSize, info);

                if (info.BmpWidth <= 0)
                {
                    return false;
                }

                info.GlyphLeft = 0;
                info.CharWidth = info.BmpWidth;
                info.GlyphAscent = Baseline;

                image.Create(info.BmpWidth, info.BmpHeight, 8);
                image.Paste(dstBmp);
                image.SetGrayScaleTable(Bpp);
                MakeAlpha(image, Bpp);

                // transfer bmp
                SetGlyph(glyph, image, info);
            }
            catch (GeneralException ge)
            {
                throw ge;
            }

            return true;
        }

        //---------------------------------------------------------
        // Create
        //---------------------------------------------------------
        public override void Create(ref LOGFONT logfont, string fontPath, string inputFontFaceName)
        {
            // create DC
            {
                this.masterDCHandle = Gdi.CreateDC("DISPLAY", null, null, IntPtr.Zero);
                if (this.masterDCHandle == IntPtr.Zero)
                {
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_CREATEDC);
                }

                DeviceContext = Gdi.CreateCompatibleDC(this.masterDCHandle);
                if (DeviceContext == IntPtr.Zero)
                {
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_CREATECOMPATIBLEDC);
                }
            }

            SetupFont(ref logfont, fontPath, inputFontFaceName);
            CalcurateParameter();

            // set bmp
            this.bmpHandle = Gdi.CreateCompatibleBitmap(this.masterDCHandle, CellSize, CellSize);
            if (this.bmpHandle == IntPtr.Zero)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_CREATECOMPATIBLEBITMAP);
            }

            this.oldBmpHandle = Gdi.SelectBitmap(DeviceContext, this.bmpHandle);
            if (this.oldBmpHandle == IntPtr.Zero)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_SELECTOBJECT_NEWBMP);
            }
        }

        protected override void Dispose(bool disposing)
        {
            CleanupFont();

            /* delete bmp */

            if (this.oldBmpHandle != IntPtr.Zero)
            {
                var bmp = Gdi.SelectBitmap(DeviceContext, this.oldBmpHandle);
                if (bmp == IntPtr.Zero)
                {
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_SELECTOBJECT_OLDBMP);
                }
            }

            if (this.bmpHandle != IntPtr.Zero)
            {
                Gdi.DeleteBitmap(this.bmpHandle);
            }

            /* delete dc */

            if (DeviceContext != IntPtr.Zero)
            {
                Gdi.DeleteDC(DeviceContext);
            }

            if (this.masterDCHandle != IntPtr.Zero)
            {
                Gdi.DeleteDC(this.masterDCHandle);
            }
        }

        private static bool TextOutC(HDC hDC, int x, int y, uint c, bool isUnicde)
        {
            if (isUnicde)
            {
                ushort[] str = { (ushort)c };
                return Gdi.TextOutW(hDC, x, y, str, 1);
            }
            else
            {
                int charNum;
                byte[] str = new byte[2];

                if (c > 0xFF)
                {
                    str[0] = (byte)(c >> 8);
                    str[1] = (byte)(c & 0xFF);
                    charNum = 2;
                }
                else
                {
                    str[0] = (byte)(c & 0xFF);
                    charNum = 1;
                }

                return Gdi.TextOutA(hDC, x, y, str, charNum);
            }
        }

        private static uint TreatUINT(RGBQUAD rgb)
        {
            return GlCm.BMP_RGB(rgb.rgbRed, rgb.rgbGreen, rgb.rgbBlue);
        }

        //---------------------------------------------------------
        // GetGlyphBmp
        //---------------------------------------------------------
        // DC から BMP を取得
        private void GetGlyphBmp(byte[] bmp, int size, GlyphInfo info)
        {
            int ret;
            BITMAPINFO256 pbi = new BITMAPINFO256();

            // UNNES memset(pbi, 0, sizeof(*pbi));
            pbi.bmiHeader.biSize = (uint)Marshal.SizeOf(pbi.bmiHeader.GetType());
            pbi.bmiHeader.biWidth = CellSize;
            pbi.bmiHeader.biHeight = CellSize;
            pbi.bmiHeader.biPlanes = 1;
            pbi.bmiHeader.biBitCount = 8;
            pbi.bmiHeader.biCompression = GdiDef.BI_RGB;
            pbi.bmiHeader.biSizeImage = (uint)size;

            Gdi.SelectBitmap(DeviceContext, this.oldBmpHandle);
            ret = Gdi.GetDIBits(DeviceContext, this.bmpHandle, 0, (uint)CellSize, bmp, pbi, DIB.RGB_COLORS);
            Gdi.SelectBitmap(DeviceContext, this.bmpHandle);

            if (ret != CellSize)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_GETDIBITS);
            }

            info.Black = 0;
            ////    pInfo->bg = 0;
            info.White = 0;

            for (int i = 0; i < 256; i++)
            {
                uint val = TreatUINT(pbi.bmiColors[i]);
                switch (val)
                {
                case ColorBlack:
                    info.Black = (byte)i;
                    break;
                //// case ColorBg:      pInfo->bg = static_cast<byte>(i);       break;
                case ColorWhite:
                    info.White = (byte)i;
                    break;
                default:
                    break;
                }
            }
        }

        //---------------------------------------------------------
        // AdjustBmp
        //---------------------------------------------------------
        // ボトムアップ→トップダウン、色番号変換
        // src: bottomup
        // dst: topdown
        // cut bg
        private void AdjustBmp(byte[] dst, byte[] src, int width, int height, GlyphInfo info)
        {
            int dwidth = GlCm.ROUND_UP(width, 4);
            byte white = 0;
            byte black = (byte)((1 << Bpp) - 1);
            int srcBaseIdx = (height - 1) * dwidth;
            int dstIdx = 0;
            info.BmpHeight = height;
            info.BmpWidth = width;

            for (int y = 0; y < height; ++y)
            {
                int srcIdx = srcBaseIdx;

                for (int x = 0; x < width; x++)
                {
                    if (src[srcIdx] == info.Black)
                    {
                        dst[dstIdx] = black;
                    }
                    else if (src[srcIdx] == info.White)
                    {
                        dst[dstIdx] = white;
                    }
                    else
                    {
                        if (x == 0)
                        {
                            info.BmpHeight = y;
                            if (y == 0)
                            {
                                info.BmpWidth = 0;
                            }

                            goto ret;
                        }
                        else
                        {
                            Debug.Assert(y == 0 || info.BmpWidth == x);
                            info.BmpWidth = x;
                        }

                        break;
                    }

                    dstIdx++;
                    srcIdx++;
                }

                srcBaseIdx -= dwidth;
            }

        ret:
            info.White = white;
            info.Black = black;
        }
    }
}
