﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using NintendoWare.Font.Runtime;
using System.Reflection;
using System.IO;
using System;
namespace NintendoWare.Font
{
    /// <summary>
    /// テクスチャ情報
    /// </summary>
    public class WindowsFontInputFormatDesc
    {
        /// <summary>
        /// ピクセルあたりのビット数
        /// </summary>
        public int Bpp { get; private set; }

        /// <summary>
        /// 説明文字列
        /// </summary>
        public bool HasAlpha { get; private set; }

        /// <summary>
        /// 説明文字列
        /// </summary>
        public string Name { get; private set; }

        /// <summary>
        /// 説明文字列
        /// </summary>
        public string Description { get; private set; }

        /// <summary>
        /// グリフ画像フォーマット
        /// </summary>
        public GlyphImageFormat GlyphImageFormat { get; private set; }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public WindowsFontInputFormatDesc(int bpp, bool hasAlpha, string name, string desc, GlyphImageFormat fmt)
        {
            this.Bpp = bpp;
            this.HasAlpha = hasAlpha;
            this.Name = name;
            this.Description = desc;
            this.GlyphImageFormat = fmt;
        }
    }

    /// <summary>
    /// テクスチャ情報
    /// </summary>
    public class ImageInputFormatDesc
    {
        /// <summary>
        /// ピクセルあたりのビット数
        /// </summary>
        public int Bpp { get; private set; }

        /// <summary>
        /// ピクセルあたりのビット数
        /// </summary>
        public int BppAlpha { get; private set; }

        /// <summary>
        /// 明度フォーマットに変換するか？
        /// </summary>
        public bool IsToIntencity
        {
            get
            {
                return
                    GlyphImageFormat == GlyphImageFormat.A4 ||
                    GlyphImageFormat == GlyphImageFormat.A8 ||
                    GlyphImageFormat == GlyphImageFormat.LA4 ||
                    GlyphImageFormat == GlyphImageFormat.LA8 ||
                    GlyphImageFormat == GlyphImageFormat.LA4Packed ||
                    GlyphImageFormat == GlyphImageFormat.LA4PackedNoCompress;
            }
        }

        /// <summary>
        /// 説明文字列
        /// </summary>
        public bool HasAlpha { get { return this.BppAlpha > 0; } }

        /// <summary>
        /// 説明文字列
        /// </summary>
        public string Description { get; private set; }

        /// <summary>
        /// グリフ画像フォーマット
        /// </summary>
        public GlyphImageFormat GlyphImageFormat { get; private set; }


        /// <summary>
        /// グリフ画像フォーマット
        /// </summary>
        public GlyphImageFormat KeyGlyphImageFormat { get; private set; }

        /// <summary>
        /// コンストラクタです。
        /// bpp と bppAlpha には実機テクスチャに変換された際に有効となるビット数を設定する。
        /// </summary>
        public ImageInputFormatDesc(int bpp, int bppAlpha, string desc, GlyphImageFormat fmt)
            : this(bpp, bppAlpha, desc, fmt, fmt)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ImageInputFormatDesc(int bpp, int bppAlpha, string desc, GlyphImageFormat fmt, GlyphImageFormat keyFmt)
        {
            this.Bpp = bpp;
            this.BppAlpha = bppAlpha;
            this.Description = desc;
            this.GlyphImageFormat = fmt;
            this.KeyGlyphImageFormat = keyFmt;
        }
    }

    public interface IPlatformDetails
    {
        /// <summary>
        /// UIに表示する名前です。
        /// </summary>
        string UIPlatformName { get; }

        /// <summary>
        /// 名前です。
        /// </summary>
        string PlatformName { get; }

        /// <summary>
        /// CAFE内蔵フォントで利用される、bfttf 形式の読み込みをサポートするかどうか
        /// </summary>
        bool BfttfEnabled { get; }

        /// <summary>
        /// スケーラブルフォントが利用できるか？
        /// </summary>
        bool ScfontEnabled { get; }

        /// <summary>
        /// 4bit フォーマット(A4 LA4)が利用できるか？
        /// コンバータオプションとして指定できるかではなく、実際にハードウィアフォーマットとして利用可能かという意味です。
        /// </summary>
        bool CanHWSupports4BitFormat { get; }

        /// <summary>
        /// 入力フォントのテクスチャが天地反転方向で格納されるか
        /// </summary>
        bool IsInputFontReversed { get; }

        /// <summary>
        /// 入力フォントバージョン
        /// </summary>
        uint InputFontVersion { get; }

        /// <summary>
        /// 出力フォントバージョン
        /// </summary>
        uint OutputFontVersion { get; }

        /// <summary>
        ///
        /// </summary>
        IEnumerable<WindowsFontInputFormatDesc> GetWindowsFontInputFormatList();

        /// <summary>
        ///
        /// </summary>
        IEnumerable<ImageInputFormatDesc> GetImageInputFormatFormatList();

        bool IsInputFontLittleEndian { get; }
        bool IsOutputFontLittleEndian { get; }

        void CUISetup(string[] args);

        NitroFontReader GetFontReader(string inputFilePath);
        NitroFontWriter GetFontWriter(
            string file,
            string groupsFile,
            GlyphDataType type,
            uint alter,
            int linefeed,
            int width,
            int left,
            int right,
            CharEncoding encoding,
            int sheetWidth,
            int sheetHeight,
            int sheetPixels,
            bool isOutputTargetLittleEndian,
            bool isOutKerningData);

        string NwToolsRootPath { get; }
    }

    /// <summary>
    /// ダミーとして使われるプラットフォーム
    /// </summary>
    public class NullPlatformDetails : IPlatformDetails
    {
        /// <summary>
        /// UI に表示する名前です。
        /// </summary>
        public string UIPlatformName { get { return string.Empty; } }

        /// <summary>
        /// 名前です。
        /// </summary>
        public string PlatformName { get { return string.Empty; } }

        /// <summary>
        /// CAFE内蔵フォントで利用される、bfttf 形式の読み込みをサポートするかどうか
        /// </summary>
        public bool BfttfEnabled { get { return false; } }

        /// <summary>
        /// スケーラブルフォントが利用できるか？
        /// </summary>
        public bool ScfontEnabled { get { return false; } }

        /// <summary>
        /// 4bit フォーマット(A4 LA4)が利用できるか？
        /// コンバータオプションとして指定できるかではなく、実際にハードウィアフォーマットとして利用可能かという意味です。
        /// </summary>
        public bool CanHWSupports4BitFormat { get { return false; } }

        /// <summary>
        /// 入力フォントのテクスチャが天地反転方向で格納されるか
        /// </summary>
        public bool IsInputFontReversed { get { return false; } }

        /// <summary>
        /// 入力フォントバージョン
        /// </summary>
        public uint InputFontVersion { get { return RtConsts.MakeVersion(3, 0, 0, 0); } }

        /// <summary>
        /// 出力フォントバージョン
        /// </summary>
        public uint OutputFontVersion { get { return RtConsts.MakeVersion(3, 0, 0, 0); } }

        /// <summary>
        ///
        /// </summary>
        public IEnumerable<WindowsFontInputFormatDesc> GetWindowsFontInputFormatList()
        {
            yield break;
        }

        /// <summary>
        ///
        /// </summary>
        public IEnumerable<ImageInputFormatDesc> GetImageInputFormatFormatList()
        {
            yield break;
        }

        public bool IsInputFontLittleEndian { get { return true; } }
        public bool IsOutputFontLittleEndian { get { return true; } }

        public void CUISetup(string[] args) { }

        public NitroFontReader GetFontReader(string inputFilePath)
        {
            return null;
        }

        public NitroFontWriter GetFontWriter(
            string file,
            string groupsFile,
            GlyphDataType type,
            uint alter,
            int linefeed,
            int width,
            int left,
            int right,
            CharEncoding encoding,
            int sheetWidth,
            int sheetHeight,
            int sheetPixels,
            bool isOutputTargetLittleEndian,
            bool isOutKerningData)
        {
            return null;
        }

        public string NwToolsRootPath
        {
            get
            {
                throw new NotImplementedException();
            }
        }

    }
}
