﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using NintendoWare.Font.Runtime;

    /// <summary>
    /// （実機フォントを読んで）フォントフォーマット を書き出すモード
    /// </summary>
    public class FormatFontWriter : FontWriter
    {
        private readonly string _FilePath;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FormatFontWriter(string filePath)
        {
            _FilePath = filePath;
        }

        /// <summary>
        /// xlor ファイルを書き出します。
        /// </summary>
        public override void WriteFontData(FontData fontData, GlyphOrder order)
        {
            GlyphImageFormat outFmt = fontData.OutputFormat.HasValue ? fontData.OutputFormat.Value : GlyphImageFormat.A8;
            if (outFmt == GlyphImageFormat.LA4PackedPassThrough)
            {
                outFmt = GlyphImageFormat.LA4Packed;
            }
            else if (outFmt == GlyphImageFormat.LA4PackedNoCompressPassThrough)
            {
                outFmt = GlyphImageFormat.LA4PackedNoCompress;
            }

            File.WriteAllText(this._FilePath, outFmt.ToString(), System.Text.Encoding.UTF8);
        }

        public override void ValidateInput()
        {
            // 何も行いません。
        }

        public override void GetGlyphOrder(GlyphOrder order)
        {
            // 何も行いません。
        }
    }

    /// <summary>
    /// （実機フォントを読んで）XLOR を書き出すモード
    /// </summary>
    public class XlorFontWriter : FontWriter
    {
        private readonly string _FilePath;

        private readonly string _xlorDoc =
@"<?xml version=""1.0"" encoding=""UTF-8"" standalone=""no"" ?>

<letter-order version=""1.1"">
    <head>
        <create user=""sample"" date=""2005-02-18T10:51:13"" />
        <title>Auto generated xlor.</title>
        <comment>FontConverterによって 自動生成された xlor です。</comment>
    </head>

    <body>
        <area width=""16"" />

        <order>
@@content@@
        </order>
    </body>
</letter-order>";

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public XlorFontWriter(string filePath)
        {
            _FilePath = filePath;
        }

        /// <summary>
        /// xlor ファイルを書き出します。
        /// </summary>
        public override void WriteFontData(FontData fontData, GlyphOrder order)
        {
            string xlor = _xlorDoc;

            Glyph[] glyphSet = fontData.GetGlyphList().GetEnum().ToArray();
            string orderContents = string.Empty;

            int y = 0;
            int index = 0;
            while (index < glyphSet.Length)
            {
                string line = string.Empty;
                for (int x = 0; x < 16; x++)
                {
                    index = y * 16 + x;
                    if (index >= glyphSet.Length)
                    {
                        break;
                    }

                    uint code = glyphSet[index].GetCode();
                    // HTMLエンコードした文字列を書き込み
                    line += string.Format("&#x{0:X4}; ", code);
                }

                y++;
                orderContents += "\t\t\t" + line + Environment.NewLine;
            }

            xlor = xlor.Replace("@@content@@", orderContents);
            File.WriteAllText(this._FilePath, xlor, System.Text.Encoding.UTF8);
        }

        public override void ValidateInput()
        {
            // 何も行いません。
        }

        public override void GetGlyphOrder(GlyphOrder order)
        {
            // 何も行いません。
        }
    }

    public class NnsWriter
    {
        private readonly ByteOrderBinaryWriter bw;
        private BinaryFileHeader header = new BinaryFileHeader();
        private uint fileSize;
        private ushort dataBlocks;
        private bool isNnsWriting;

        public NnsWriter(BinaryFile binaryFile, bool isOutputTargetLittleEndian)
        {
            this.bw = new ByteOrderBinaryWriter(binaryFile, isOutputTargetLittleEndian);
            this.fileSize = 0;
            this.dataBlocks = 0;
            this.isNnsWriting = false;
        }

        public void WriteNnsBegin(Signature32 sig, uint version)
        {
            this.header.Signature = sig;
            this.header.Version = version;

            this.fileSize = (uint)Runtime.BinaryFileHeader.Length;
            this.dataBlocks = 0;
            this.isNnsWriting = true;

            this.bw.Seek((int)this.fileSize, SeekOrigin.Begin);
        }

        public void WriteNnsEnd()
        {
            Debug.Assert(this.bw.BaseStream.Position == this.fileSize);

            this.header.ByteOrder = ConverterEnvironment.ByteOrderMark;
            this.header.FileSize = this.fileSize;
            this.header.HeaderSize = (ushort)Runtime.BinaryFileHeader.Length;
            this.header.DataBlocks = this.dataBlocks;

            this.bw.Seek(0, SeekOrigin.Begin);
            this.bw.Write(this.header);

            this.isNnsWriting = false;
        }

        public bool WriteNnsBlock(NnsData nnsData, Signature32 id, int no)
        {
            NnsData.GeneralBinaryBlockInfo pbb;

            pbb = nnsData.GetBlock(id, no);

            if (pbb != null)
            {
                this.WriteNnsBlock(id, pbb.Body);
                return true;
            }
            else
            {
                return false;
            }
        }

        public bool WriteNnsBlock(NnsData nnsData, Signature32 id)
        {
            return this.WriteNnsBlock(nnsData, id, 0);
        }

        protected void WritePadding(uint size)
        {
            Debug.Assert(this.isNnsWriting);

            for (int i = 0; i < size; ++i)
            {
                byte value = 0;
                this.bw.Write(value);
            }
        }

        private void WriteNnsBlock(Signature32 kind, IBinarizable bina)
        {
            Debug.Assert(this.isNnsWriting);

            int contentsSize = bina.GetBlockContentsSize((int)this.fileSize + Runtime.BinaryBlockHeader.Length);
            int alignedContentsSize = GlCm.ROUND_UP(contentsSize, 4);

            var bbh = new Runtime.BinaryBlockHeader();
            bbh.Kind = kind;
            bbh.Size = (uint)(Runtime.BinaryBlockHeader.Length + alignedContentsSize);

            //---- 書き出し
            this.bw.Write(bbh);
            bina.Write(this.bw);
            this.WritePadding((uint)(alignedContentsSize - contentsSize));

            //---- パラメータ更新
            this.fileSize = (uint)this.bw.BaseStream.Position;
            this.dataBlocks++;
        }
    }
}
