﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System.Collections.Generic;
    using System.Xml;
    using NintendoWare.Font.Runtime;

    public class GlyphOrder : HandlerBase
    {
        public const uint NullGlyph = RtConsts.InvalidCharCode;
        private const int MaxChar = 0x110000; // UTF-8(4byte) が扱う領域の上限

        private const string TagCharacterOrder = "letter-order";
        private const string TagArea = "area";
        private const string TagBody = "body";
        private const string TagOrder = "order";
        private const string TagCode = "code";
        private const string TagNull = "null";
        private const string TagSp = "sp";

        private const string AttribWidth = "width";
        private const string AttribHeight = "height";
        private const string AttribVersion = "version";
        private const string XcoVersion = "1.1";
        private const string XcoVersionOld = "1.0";

        private readonly List<uint> OrderList = new List<uint>();

        private readonly bool[] CheckAry = new bool[MaxChar];

        private int width;
        private int height;
        private bool isHasHeight;
        private bool isHasOrder;
        private bool isOldVer;
        private string filePath;

        private int tagLevel;
        private bool isInOrder;
        private bool isInCode;
        private bool isInBody;

        public GlyphOrder()
        {
            this.width = 16;
            this.height = 0;
            this.tagLevel = 0;
            this.isInOrder = false;
            this.isInCode = false;
            this.isInBody = false;
            this.isHasHeight = false;
            this.isHasOrder = false;
            this.filePath = null;
            this.isOldVer = false;
        }

        public bool IsValid()
        {
            return this.isHasOrder;
        }

        public int GetHNum()
        {
            return this.width;
        }

        public int GetVNum()
        {
            return this.height;
        }

        // 順序定義ファイルを読み込む
        public void Load(string orderFilePath, bool useDtdValidation)
        {
            this.filePath = orderFilePath;

            try
            {
                using (var reader = CreateReader(orderFilePath, useDtdValidation))
                {
                    Parse(reader);
                }
            }
            catch (HandlerBaseException)
            {
                if (Error != null)
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_MSG, Error);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_INVALID_ORDER_FILE);
                }
            }
            catch (GeneralException)
            {
                throw;
            }
            catch (XmlException ex)
            {
                Rpt._RPT1("XML Error: {0}\n", ex.Message);
                throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_MSG, ex.Message);
            }

            // 縦横チェック
            if (this.width <= 0)
            {
                throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_INVALID_ORDER_WIDTH, this.width);
            }

            if (this.isHasHeight)
            {
                if (this.height <= 0)
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_INVALID_ORDER_HEIGHT, this.isHasHeight);
                }
            }
            else
            {
                this.height = (this.OrderList.Count + this.width - 1) / this.width;
            }

            this.isHasOrder = true;
        }

        // 位置(x,y)のグリフに対応する文字コードを返す
        public uint GetCharCode(int x, int y)
        {
            int n;

            if (x < 0 || this.width <= x)
            {
                return Runtime.RtConsts.InvalidCharCode;
            }

            if (y < 0 || this.height <= y)
            {
                return Runtime.RtConsts.InvalidCharCode;
            }

            n = (y * this.width) + x;

            if (n >= this.OrderList.Count)
            {
                return Runtime.RtConsts.InvalidCharCode;
            }

            return this.OrderList[n];
        }

        // ---------------------------------------------------------------------------
        //  PParseHandlers: Overrides of the SAX DocumentHandler interface
        // ---------------------------------------------------------------------------
        protected override void StartElement(string name, AttributeList attributes)
        {
            if (name == TagNull)
            {
                if (this.isInOrder)
                {
                    this.AddOrder(NullGlyph);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagNull, TagOrder);
                }
            }
            else if (name == TagSp)
            {
                if (this.isInOrder)
                {
                    this.AddOrder(0x0020);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagSp, TagOrder);
                }
            }
            else if (name == TagCode)
            {
                if (this.isInOrder)
                {
                    this.isInCode = true;
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagCode, TagOrder);
                }
            }
            else if (name == TagOrder)
            {
                if (this.isInBody)
                {
                    this.isInOrder = true;
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagOrder, TagBody);
                }
            }
            else if (name == TagArea)
            {
                if (this.isInBody)
                {
                    var width_string = attributes.GetValue(AttribWidth);
                    var height_string = attributes.GetValue(AttribHeight);

                    if (width_string != null)
                    {
                        int.TryParse(width_string, out this.width);
                    }

                    if (height_string != null)
                    {
                        int.TryParse(height_string, out this.height);
                        this.isHasHeight = true;
                    }

                    Rpt._RPT2("area: width={0} height={1}\n", this.width, this.height);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagArea, TagBody);
                }
            }
            else if (name == TagBody)
            {
                if (this.tagLevel == 1)
                {
                    this.isInBody = true;
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagBody, TagCharacterOrder);
                }
            }
            else if (name == TagCharacterOrder)
            {
                var version = attributes.GetValue(AttribVersion);
                var isErr = false;

                if (version == null)
                {
                    isErr = true;
                }
                else
                {
                    if (version != XcoVersion)
                    {
                        if (version != XcoVersionOld)
                        {
                            isErr = true;
                        }
                        else
                        {
                            this.isOldVer = true;
                        }
                    }
                }

                if (isErr && Error == null)
                {
                    string validVer = XcoVersion;
                    validVer += " or ";
                    validVer += XcoVersionOld;

                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_VERSION_MISMATCH, TagCharacterOrder, version, validVer);
                }
            }

            this.tagLevel++;
        }

        protected override void Characters(string chars, int length)
        {
            if (this.isInOrder)
            {
                if (this.isInCode)
                {
                    uint val = 0;

                    uint.TryParse(chars, out val);
                    this.AddOrder(val);

                    this.isInCode = false;
                }
                else
                {
                    for (var i = 0; i < length; ++i)
                    {
                        if (!IsWhitespace(chars[i]))
                        {
                            // サロゲートペアの場合は 2 文字処理する
                            if ((chars[i] & 0xf800) == 0xd800)
                            {
                                if (i + 1 >= length || (chars[i + 1] & 0xfc00) != 0xdc00)
                                {
                                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_UNKNOWN);
                                }
                                uint c = (uint)(0x10000 + (((ushort)chars[i] - 0xd800) << 10) + ((ushort)chars[i + 1] - 0xdc00));
                                i++;
                                this.AddOrder(c);
                            }
                            else
                            {
                                this.AddOrder(chars[i]);
                            }
                        }
                    }
                }
            }
        }

        protected override void EndElement(string name)
        {
            this.tagLevel--;

            if (name == TagCode)
            {
                this.isInCode = false;
            }
            else if (name == TagOrder)
            {
                this.isInOrder = false;
            }
            else if (name == TagBody)
            {
                this.isInBody = false;
            }
        }

        private void AddOrder(uint c)
        {
            if ((c != NullGlyph) && (!this.isOldVer) && this.CheckAry[c])
            {
                throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_MULTIPLE_ORDER_LETTER, c, (char)c);
            }

            this.CheckAry[c] = true;
            this.OrderList.Add(c);
        }
    }
}
