﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System.Collections.Generic;
    using System.Text.RegularExpressions;
    using System.Xml;

    public class GlyphGroups : HandlerBase
    {
        private const string TagGlyphGroups = "glyph-groups";
        private const string TagBody = "body";
        private const string TagGroup = "group";
        private const string TagSp = "sp";

        private const string AttribName = "name";
        private const string AttribIndex = "index";
        private const string AttribVersion = "version";
        private const string XggVersion = "1.0";

        private readonly GroupList Groups = new GroupList();
        private bool isValid;

        private int tagLevel;
        private bool isInBody;
        private Group currentGroup;

        public GlyphGroups()
        {
            this.isValid = false;
            this.tagLevel = 0;
            this.isInBody = false;
            this.currentGroup = null;
        }

        public bool IsValid()
        {
            return this.isValid;
        }

        public GroupList GetGroups()
        {
            return this.Groups;
        }

        // 順序定義ファイルを読み込む
        public void Load(string orderFilePath, bool useDtdValidation)
        {
            try
            {
                using (var reader = CreateReader(orderFilePath, useDtdValidation))
                {
                    Parse(reader);
                }
            }
            catch (HandlerBaseException)
            {
                if (Error == null)
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_MSG, Error);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_INVALID_GROUP_FILE);
                }
            }
            catch (GeneralException ex)
            {
                throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_MSG, ex.GetMsg());
            }
            catch (XmlException ex)
            {
                Rpt._RPT1("XML Error: {0}\n", ex.Message);
                throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_MSG, ex.Message);
            }

            // パラメータチェック
            IntIntMap indexCheck = new IntIntMap();
            int index = -1;

            GroupList_Normalize(this.Groups, ref index, indexCheck);

            this.isValid = true;
        }

        protected string ConvertGroupName(string str)
        {
            if (str == string.Empty)
            {
                throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_GROUP_REQUIRE_NAME);
            }

            // 文字要素をチェック
            // OK = [0-9a-zA-Z_]
            var match = Regex.Match(str, "([^0-9a-zA-Z_])");
            if (match.Success)
            {
                throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_GROUP_NAME_INVALID_CHAR, str, match.Value);
            }

            return str;
        }

        // ---------------------------------------------------------------------------
        //  PParseHandlers: Overrides of the SAX DocumentHandler interface
        // ---------------------------------------------------------------------------
        protected override void StartElement(string name, AttributeList attributes)
        {
            if (name == TagSp)
            {
                if (this.currentGroup != null)
                {
                    this.currentGroup.Chars.Add(0x0020);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagSp, TagGroup);
                }
            }
            else if (name == TagGroup)
            {
                if (this.isInBody)
                {
                    var name_string = attributes.GetValue(AttribName);
                    var index_string = attributes.GetValue(AttribIndex);

                    var group = new Group();

                    if (name_string == null)
                    {
                        throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_GROUP_REQUIRE_NAME);
                    }

                    group.Name = this.ConvertGroupName(name_string);
                    Rpt._RPT1("group name `{0}`\n", group.Name);

                    if (index_string != null)
                    {
                        int index;
                        if (int.TryParse(index_string, out index))
                        {
                            group.Index = index;
                        }
                    }

                    if (this.currentGroup != null)
                    {
                        group.Parent = this.currentGroup;
                        group.Parent.Groups.Add(group);
                    }
                    else
                    {
                        group.Parent = null;
                        this.Groups.Add(group);
                    }

                    this.currentGroup = group;
                    Rpt._RPT2("group: wname={0} aname={1}\n", name_string, this.currentGroup.Name);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagGroup, TagBody);
                }
            }
            else if (name == TagBody)
            {
                if (this.tagLevel == 1)
                {
                    this.isInBody = true;
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagBody, TagGlyphGroups);
                }
            }
            else if (name == TagGlyphGroups)
            {
                var version = attributes.GetValue(AttribVersion);

                if (version == null || version != XggVersion)
                {
                    if (Error == null)
                    {
                        Error = string.Format(
                            Strings.IDS_ERR_VERSION_MISMATCH,
                            TagGlyphGroups,
                            version,
                            XggVersion);
                    }
                }
            }

            this.tagLevel++;
        }

        protected override void Characters(string chars, int length)
        {
            if (this.currentGroup != null)
            {
                for (int i = 0; i < length; ++i)
                {
                    if (!IsWhitespace(chars[i]))
                    {
                        this.currentGroup.Chars.Add(chars[i]);
                    }
                }
            }
        }

        protected override void EndElement(string name)
        {
            this.tagLevel--;

            if (name == TagGroup)
            {
                if (this.currentGroup != null)
                {
                    this.currentGroup = this.currentGroup.Parent;
                }
            }
            else if (name == TagBody)
            {
                this.isInBody = false;
            }
        }

        private static void GroupList_Normalize(GlyphGroups.GroupList list, ref int index, IntIntMap map)
        {
            foreach (var g in list)
            {
                // グループのインデックスは全てのグループに指定されているか
                // または全てのグループに指定されていないかのどちらかでなければならない
                // index == -1 なら一番最初のグループのなので判定を保留する
                // index == 0 ならそれ以前のグループは全てインデックスが指定されていた
                // index > 0 ならそれ以前のグループは全てインデックスが指定されていなかった
                if (index < 0)
                {
                    // 一番最初のグループ
                    index = 0;
                }
                else
                {
                    if (index > 0 && g.Index >= 0)
                    {
                        // index が指定されていてはいけないのに指定されている
                        throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_GROUP_INDEX);
                    }
                    else if (index == 0 && g.Index < 0)
                    {
                        // index が指定されていなければならないのに指定されていない
                        throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_GROUP_INDEX);
                    }
                }

                if (g.Index < 0)
                {
                    g.Index = index++;
                }

                var found = map.ContainsKey(g.Index);
                if (found)
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_GROUP_INDEX_DUPLICATE, g.Index);
                }

                map[g.Index] = 1;

                GroupList_Normalize(g.Groups, ref index, map);
            }
        }

        public class CharCodeList : List<uint>
        {
        }

        public class GroupList : List<Group>
        {
        }

        public class Group
        {
            private readonly GroupList groups = new GroupList();
            private readonly CharCodeList chars = new CharCodeList();

            public Group()
            {
                this.Index = -1;
                this.Parent = null;
            }

            public int Index { get; set; }

            public string Name { get; set; }

            public Group Parent { get; set; }

            public GroupList Groups
            {
                get { return this.groups; }
            }

            public CharCodeList Chars
            {
                get { return this.chars; }
            }
        }
    }
}
