﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Xml.Serialization;

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Diagnostics;
    using System.IO;
    using System.Reflection;

    /// <summary>
    /// プラットフォームプリファレンス
    /// </summary>
    public class PlatformPreference
    {
        /// <summary>
        /// プラットフォーム名です。
        /// </summary>
        public string ConverterPlatformName { get; set; }

        public string CacheDirectory { get; set; }
        public int MaxCacheItemCount { get; set; }
        public string IsGpuEncodingEnabled { get; set; }

        // 隠しオプション Proxy 専用
        public string OriginalFontConverterConsolePath { get; set; }

        public PlatformPreference()
        {
            MaxCacheItemCount = 50;
            IsGpuEncodingEnabled = "true";
        }
    }

    /// <summary>
    /// 環境依存の情報を集約するクラスです。
    /// </summary>
    public class ConverterEnvironment
    {
        private static IPlatformDetails _PlatformDetails;
        private static string _PlatformName;
        private static string _TileMode;
        private static PlatformPreference platformPreference;

        /// <summary>
        /// テクスチャデータのアライメントサイズです。
        /// </summary>
        public static int TextureDataAlignSize
        {
            get
            {
                // Bntx 形式なら、ファイル内部のメモリープールに対して実機が要求する 4k をアライメント制約とします。
                // Bntx 形式は複数プラットフォームをサポートするので、この対応は暫定的なものです。
                return UseBntx ? 4 * 1024 : 8 * 1024;
            }
        }

        public const string PlatformDllPrefix = "FontConverterCommon";
        public const string GenericPlatformDllName = "Generic";
        public const string PreferenceXmlFileName = "preference.xml";

        /// <summary>
        /// バイトオーダーマークです。
        /// </summary>
        public const ushort ByteOrderMark = 0xFEFF;
        public const string PlatformChar = "F";
        public const string PlatformCharSmall = "f";

        //----------------------------------------------------------

        /// <summary>
        /// 実機側での複数バイトを扱う際のバイト順を示します。
        /// リトルエンディアンの場合は true、ビッグエンディアンの場合は falseです。
        /// </summary>
        public static bool IsInputTargetLittleEndian { get; set; }

        /// <summary>
        /// 出力データのエンディアンです。
        /// ビッグエンディアン(false)指定された場合は、
        /// 画像フォーマットも連動して実機テクスチャフォーマットに切り替わります。
        /// </summary>
        public static bool IsOutputTargetLittleEndian { get; set; }

        /// <summary>
        /// テクスチャ詳細情報
        /// </summary>
        public static IPlatformDetails PlatformDetails
        {
            get { return _PlatformDetails; }
        }

        /// <summary>
        /// タイルモード
        /// </summary>
        public static string TileMode
        {
            get { return _TileMode; }
        }

        /// <summary>
        /// Bntx テクスチャを使うかどうか？（Generic 動作でのみ有効です）
        /// </summary>
        public static bool UseBntx
        {
            get ; set;
        }

        /// <summary>
        /// ログを出力しないかどうか
        /// </summary>
        public static bool IsLogSilent
        {
            get;
            set;
        }

        /// <summary>
        /// Gpu エンコーディングが有効かどうか
        /// </summary>
        public static string IsGpuEncodingEnabled
        {
            get;
            set;
        }

        public static PlatformPreference PlatformPreference
        {
            get { return platformPreference; }
        }

        /// <summary>
        /// プリファレンスで設定される、UI に表示するプラットフォーム名です。
        /// </summary>
        public static string UIPlatformPreferenceName
        {
            get
            {
                return (platformPreference == null) ? "" : platformPreference.ConverterPlatformName;
            }
        }

        public static string PlatformCacheDirectory
        {
            get
            {
                return platformPreference == null || string.IsNullOrEmpty(platformPreference.CacheDirectory) ?
                    Path.Combine(
                        System.Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData),
                        "Nintendo",
                        "FontConverter") :
                        System.Environment.ExpandEnvironmentVariables(platformPreference.CacheDirectory);
            }
        }

        public static int PlatformMaxCacheItemCount
        {
            get
            {
                return (platformPreference == null) ? 50 : platformPreference.MaxCacheItemCount;
            }
        }

        static ConverterEnvironment()
        {
            IsInputTargetLittleEndian = false;
            IsOutputTargetLittleEndian = false;
            UseBntx = true;
            IsGpuEncodingEnabled = "true";
        }

        /// <summary>
        /// プラットフォーム名を列挙します。
        /// </summary>
        public static IEnumerable<string> EnumratePlatFormNames(string platformDllPrefix)
        {
            foreach (var dllFilePath in EnumratePlatFormDLLFilePaths_((fileName) => fileName.StartsWith(platformDllPrefix)))
            {
                string dllFileName = Path.GetFileNameWithoutExtension(dllFilePath);

                dllFileName = dllFileName.Replace(platformDllPrefix, "");
                if (string.IsNullOrEmpty(dllFileName))
                {
                    continue;
                }

                yield return dllFileName;
            }
        }

        public static string GetTileModeFromPlatformName(string platformName)
        {
            if (string.Equals(platformName, "Win", StringComparison.CurrentCultureIgnoreCase) ||
                        string.Equals(platformName, "Win32", StringComparison.CurrentCultureIgnoreCase) ||
                        string.Equals(platformName, "Generic", StringComparison.CurrentCultureIgnoreCase))
            {
                return "linear";
            }
            else if (string.Equals(platformName, "Cafe", StringComparison.CurrentCultureIgnoreCase))
            {
                return "Cafe";
            }
            else if (string.Equals(platformName, "NX", StringComparison.CurrentCultureIgnoreCase))
            {
                return "NX";
            }
            else
            {
                return "Unknown";
            }
        }

        /// <summary>
        /// 対象プラットフォームを初期化します。
        /// </summary>
        public static bool InitializeTargetPlatform(string platformName, string tileMode, string platformDllName = null, ConvertImageFontCache cache = null)
        {
            _PlatformName = platformName;
            _TileMode = tileMode;

            if (platformDllName != null)
            {
                platformName = platformDllName;
            }

            string platformSufix = string.IsNullOrEmpty(platformName)
               || platformName.Equals("Win", StringComparison.CurrentCultureIgnoreCase) ? GenericPlatformDllName : platformName;

            string platformDllFileName = PlatformDllPrefix + platformSufix;

            // GX2TextureWriter をプラットフォーム中立なインタフェース経由で利用するよう変更して、実装を切り替えられるようにしています。
            Type type = EnumrateInterfaceTypesFromDLL_((dllFileName) => string.Compare(dllFileName, platformDllFileName, true) == 0, typeof(IPlatformDetails)).FirstOrDefault();
            if (type != null)
            {
                _PlatformDetails = Activator.CreateInstance(type) as IPlatformDetails;

                IsInputTargetLittleEndian = _PlatformDetails.IsInputFontLittleEndian;
                IsOutputTargetLittleEndian = _PlatformDetails.IsOutputFontLittleEndian;
            }

            // プラットフォームがジェネリックの時だけプリファレンスで上書きする
            if (string.Compare(_PlatformName, GenericPlatformDllName, StringComparison.OrdinalIgnoreCase) != 0)
            {
                return _PlatformDetails != null;
            }

            // プリファレンス設定ファイルが存在するか探す。
            // あれば読み込む。読み込んだら、各種処理にその値を利用する。
            string executingFileName = Assembly.GetExecutingAssembly().Location;
            string executingFolder = Path.GetDirectoryName(executingFileName);
            Debug.Assert(executingFolder != null);
            string platformPreferenceFilePath = Path.Combine(executingFolder, PreferenceXmlFileName);

            // 実行ファイルと同じフォルダになかったら、直下のPreferencesフォルダから探す
            if (!File.Exists(platformPreferenceFilePath))
            {
                var prefFolderPath = Path.Combine(executingFolder, "Preferences");
                if (Directory.Exists(prefFolderPath))
                {
                    // Preferences内のサブフォルダを探し、1つしかなければそれを読み、
                    // 2つ以上ある場合は名前がGenericではない最初のものを読む
                    var folders =
                        Directory.EnumerateDirectories(prefFolderPath)
                            .Where(x => File.Exists(Path.Combine(x, PreferenceXmlFileName)))
                            .ToArray();
                    if (folders.Length == 1)
                    {
                        platformPreferenceFilePath = Path.Combine(
                            executingFolder,
                            "Preferences",
                            folders.First(),
                            PreferenceXmlFileName);
                    }
                    else if (folders.Length >= 2)
                    {
                        platformPreferenceFilePath = Path.Combine(
                            executingFolder,
                            "Preferences",
                            folders.First(f => !f.EndsWith("Generic", StringComparison.InvariantCultureIgnoreCase)),
                            PreferenceXmlFileName);
                    }
                }
            }

            // TODO: ここに来ない分岐があるので、cache.AddCacheKey はここで行いたくない。
            // プリファレンスが存在するときはデシリアライズする。
            if (File.Exists(platformPreferenceFilePath))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(PlatformPreference));
                var bytes = File.ReadAllBytes(platformPreferenceFilePath);
                using (var fs = new MemoryStream(bytes))
                {
                    platformPreference = (PlatformPreference)serializer.Deserialize(fs);
                    _PlatformName = platformPreference.ConverterPlatformName;
                    _TileMode = GetTileModeFromPlatformName(_PlatformName);
                    IsGpuEncodingEnabled = platformPreference.IsGpuEncodingEnabled;
                }
                if (cache != null)
                {
                    cache.AddCacheKey(bytes);
                }
            }
            else
            {
                if (cache != null)
                {
                    cache.AddCacheKey(string.Empty);
                }
            }

            return _PlatformDetails != null;
        }

        // 本来はこれはfalseであるべきだが、現状ではまだIsCtrの方を用いる
        public static bool IsCtr { get { return true; } }

        /// <summary>
        ///
        /// </summary>
        static IEnumerable<string> EnumratePlatFormDLLFilePaths_(Predicate<string> passPredicate)
        {
            foreach (var dllFilePath in Directory.GetFiles(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "*.dll"))
            {
                string dllFileName = Path.GetFileNameWithoutExtension(dllFilePath);

                if (!passPredicate(dllFileName))
                {
                    continue;
                }

                yield return dllFilePath;
            }
        }

        /// <summary>
        ///
        /// </summary>
        static IEnumerable<Type> EnumrateInterfaceTypesFromDLL_(Predicate<string> passPredicate, Type interfaceType)
        {
            foreach (var dllFilePath in EnumratePlatFormDLLFilePaths_(passPredicate))
            {
                Assembly platformAssembly = Assembly.LoadFrom(dllFilePath);
                if (platformAssembly != null)
                {
                    var types = platformAssembly.GetTypes().Where(p => interfaceType.IsAssignableFrom(p));

                    foreach (var type in types)
                    {
                        yield return type;
                    }
                }
            }
        }
    }
}
