﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using u32 = System.UInt32;

    /// <summary>
    /// ビット列へ毎回bit数を指定して格納するクラスです。
    /// </summary>
    public class BitWriter : IDisposable
    {
        private int bitFree;
        private byte[] buf;
        private int pos;
        private byte data;    // 上位ビットから詰めていく

        public BitWriter(byte[] buf)
        {
            this.buf = buf;
            this.bitFree = 8;
            this.data = 0;
        }

        public void Dispose()
        {
            {
                this.Flush();
            }
        }

        public void Flush()
        {
            if (this.bitFree < 8)
            {
                this.FlushImpl();
            }
        }

        public void Write(int bits, u32 data)
        {
            while (bits > 8)
            {
                bits -= 8;
                this.Write8(8, (byte)(data >> bits));
            }

            this.Write8(bits, (byte)data);
        }

        public void Fill(int bits, int num, u32 data)
        {
            while (num-- > 0)
            {
                this.Write(bits, data);
            }
        }

        public void Skip(int bits)
        {

            if (bits > this.bitFree)
            {
                bits -= this.bitFree;

                this.pos += bits / 8;
                this.bitFree = 8 - (bits % 8);
                this.data = this.buf[this.pos];
            }
            else
            {
                this.bitFree -= bits;
            }
        }

        private void FlushImpl()
        {
            this.buf[this.pos++] = this.data;
            this.data = 0;
            this.bitFree = 8;
        }

        private void Write8(int bits, byte data)
        {
            Debug.Assert(bits <= 8);

            if (this.bitFree < bits)
            {
                int over = bits - this.bitFree;
                this.Write8(this.bitFree, (byte)(data >> over));
                this.Write8(over, data);
            }
            else
            {
                data &= (byte)(0xFF >> (8 - bits));
                this.data |= (byte)(data << (this.bitFree - bits));

                this.bitFree -= bits;

                if (this.bitFree <= 0)
                {
                    this.FlushImpl();
                }
            }
        }
    }
}
