﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.IO;
    using NintendoWare.Font.Runtime;

    /// <summary>
    /// フォントバイナリ出力設定用ViewModelです。
    /// </summary>
    public class OutNitroViewModel : OutSettingViewModel
    {
        private const string FileFilterStr = "font resource (*." + LibFormat.ExtensionFont + ", *." + LibFormat.ExtensionArchiveFont + ")|*." + LibFormat.ExtensionFont + ";*." + LibFormat.ExtensionArchiveFont
                                        + "|normal brfnt (*." + LibFormat.ExtensionFont + ")|*." + LibFormat.ExtensionFont
                                        + "|archived brfnt (*." + LibFormat.ExtensionArchiveFont + ")|*." + LibFormat.ExtensionArchiveFont
                                        + "|all files (*.*)|*.*";

        private const string NormalFontExt = "." + LibFormat.ExtensionFont;

        private const string ArchiveFontExt = "." + LibFormat.ExtensionArchiveFont;

        private readonly OutRuntimeBinarySettings settings;
        private CharEncodingViewModel[] fontEncodingInfos;
        private SheetPixelsViewModel[] sheetPixelsInfos;
        private IPlatformDetails _platformDetails;

        public OutNitroViewModel(IPlatformDetails platformDetails,  OutRuntimeBinarySettings settings)
        {
            if (settings == null)
            {
                throw new ArgumentNullException("settings");
            }

            this.settings = settings;
            this.DisplayName = MainWindow.FontBinaryTabTitleStr;
            this.SetPathHistory(this.settings.FilePaths);

            _platformDetails = platformDetails;
        }

        /// <summary>
        /// 文字コードの表示名を取得します。
        /// </summary>
        public string CharEncodingDisplayName
        {
            get
            {
                return ConverterEnvironment.IsCtr ?
                    Strings.DisplayNameCharacterEncoding :
                    Strings.DisplayNameCharacterEncodingOld;
            }
        }

        public CharEncoding Encoding
        {
            get
            {
                return this.settings.Encoding;
            }

            set
            {
                if (value == this.settings.Encoding)
                {
                    return;
                }

                this.settings.Encoding = value;
                this.OnPropertyChanged("Encoding");
            }
        }

        public int SheetPixels
        {
            get
            {
                return this.settings.SheetPixels;
            }

            set
            {
                if (value == this.settings.SheetPixels)
                {
                    return;
                }

                this.settings.SheetPixels = value;
                this.OnPropertyChanged("SheetPixels");
            }
        }

        public bool UseDefaultAlternateChar
        {
            get
            {
                return this.settings.UseDefaultAlternateChar;
            }

            set
            {
                if (value == this.settings.UseDefaultAlternateChar)
                {
                    return;
                }

                this.settings.UseDefaultAlternateChar = value;
                this.OnPropertyChanged("UseDefaultAlternateChar");
            }
        }

        public string AlternateChar
        {
            get
            {
                return this.settings.AlternateChar;
            }

            set
            {
                if (value == this.settings.AlternateChar)
                {
                    return;
                }

                this.settings.AlternateChar = value;
                this.OnPropertyChanged("AlternateChar");
            }
        }

        public bool UseDefaultLineHeight
        {
            get
            {
                return this.settings.UseDefaultLineHeight;
            }

            set
            {
                if (value == this.settings.UseDefaultLineHeight)
                {
                    return;
                }

                this.settings.UseDefaultLineHeight = value;
                this.OnPropertyChanged("UseDefaultLineHeight");
            }
        }

        public int LineHeight
        {
            get
            {
                return this.settings.LineHeight;
            }

            set
            {
                if (value == this.settings.LineHeight)
                {
                    return;
                }

                this.settings.LineHeight = value;
                this.OnPropertyChanged("LineHeight");
            }
        }

        public bool UseDefaultLeft
        {
            get
            {
                return this.settings.UseDefaultLeft;
            }

            set
            {
                if (value == this.settings.UseDefaultLeft)
                {
                    return;
                }

                this.settings.UseDefaultLeft = value;
                this.OnPropertyChanged("UseDefaultLeft");
            }
        }

        public int DefaultLeft
        {
            get
            {
                return this.settings.DefaultLeft;
            }

            set
            {
                if (value == this.settings.DefaultLeft)
                {
                    return;
                }

                this.settings.DefaultLeft = value;
                this.OnPropertyChanged("DefaultLeft");
            }
        }

        public bool UseDefaultWidth
        {
            get
            {
                return this.settings.UseDefaultWidth;
            }

            set
            {
                if (value == this.settings.UseDefaultWidth)
                {
                    return;
                }

                this.settings.UseDefaultWidth = value;
                this.OnPropertyChanged("UseDefaultWidth");
            }
        }

        public int DefaultWidth
        {
            get
            {
                return this.settings.DefaultWidth;
            }

            set
            {
                if (value == this.settings.DefaultWidth)
                {
                    return;
                }

                this.settings.DefaultWidth = value;
                this.OnPropertyChanged("DefaultWidth");
            }
        }

        public bool UseDefaultRight
        {
            get
            {
                return this.settings.UseDefaultRight;
            }

            set
            {
                if (value == this.settings.UseDefaultRight)
                {
                    return;
                }

                this.settings.UseDefaultRight = value;
                this.OnPropertyChanged("UseDefaultRight");
            }
        }

        public int DefaultRight
        {
            get
            {
                return this.settings.DefaultRight;
            }

            set
            {
                if (value == this.settings.DefaultRight)
                {
                    return;
                }

                this.settings.DefaultRight = value;
                this.OnPropertyChanged("DefaultRight");
            }
        }

        public bool IsOutKerningData
        {
            get
            {
                return this.settings.IsOutKerningData;
            }

            set
            {
                if (value == this.settings.IsOutKerningData)
                {
                    return;
                }

                this.settings.IsOutKerningData = value;
                this.OnPropertyChanged("IsOutKerningData");
            }
        }

        /// <summary>
        /// Returns a list of strings used to populate the Customer Type selector.
        /// </summary>
        public CharEncodingViewModel[] FontEncodingInfos
        {
            get
            {
                if (this.fontEncodingInfos == null)
                {
                    var list = new List<CharEncodingViewModel>();
                    if (ConverterEnvironment.IsCtr)
                    {
                        // CTR版では、UTF-16, UTF-8を分けない。
                        list.Add(new CharEncodingViewModel(
                            string.Format(Strings.UIStringDefault, "Unicode"),
                            CharEncoding.UTF16));
                    }
                    else
                    {
                        list.Add(new CharEncodingViewModel("UTF-8", CharEncoding.UTF8));
                        list.Add(new CharEncodingViewModel(
                            string.Format(Strings.UIStringDefault, "UTF-16"),
                            CharEncoding.UTF16));
                    }

                    list.Add(new CharEncodingViewModel("Shift_JIS", CharEncoding.SJIS));
                    list.Add(
                        new CharEncodingViewModel(
                            "CodePage 1252 (ISO 8859-1)",
                            CharEncoding.CP1252));
                    //// list.Add(new FontEncodingInfo("CodePage 949 (UHC/KS X 1001)", FontEncoding.CP949));
                    this.fontEncodingInfos = list.ToArray();
                }

                return this.fontEncodingInfos;
            }
        }

        public SheetPixelsViewModel[] SheetPixelsInfos
        {
            get
            {
                if (this.sheetPixelsInfos == null)
                {
                    this.sheetPixelsInfos = new SheetPixelsViewModel[]
                    {
                        new SheetPixelsViewModel(
                            string.Format(
                                Strings.UIStringDefault,
                                Strings.IDS_OUTNITRO_SHEETPIXELS_AUTO),
                            0),
                        new SheetPixelsViewModel(Strings.IDS_OUTNITRO_SHEETPIXELS_16KP,     16 * 1024),
                        new SheetPixelsViewModel(Strings.IDS_OUTNITRO_SHEETPIXELS_32KP,     32 * 1024),
                        new SheetPixelsViewModel(Strings.IDS_OUTNITRO_SHEETPIXELS_64KP,     64 * 1024),
                        new SheetPixelsViewModel(Strings.IDS_OUTNITRO_SHEETPIXELS_128KP,   128 * 1024),
                        new SheetPixelsViewModel(Strings.IDS_OUTNITRO_SHEETPIXELS_256KP,   256 * 1024),
                        new SheetPixelsViewModel(Strings.IDS_OUTNITRO_SHEETPIXELS_512KP,   512 * 1024),
                        new SheetPixelsViewModel(Strings.IDS_OUTNITRO_SHEETPIXELS_1024KP, 1024 * 1024),
                    };
                }

                return this.sheetPixelsInfos;
            }
        }

        public string FileFilter
        {
            get { return FileFilterStr; }
        }

        public string FileExtension
        {
            get { return LibFormat.ExtensionFont; }
        }

        public override void SaveState()
        {
            this.settings.FilePaths = new StringList(this.PathHistory);
        }

        public override FontWriter GetFontWriter()
        {
            var sheetPixels = this.settings.SheetPixels;

#if UI_VALUE_VALIDATION
            int encIdx = IDC_OUT_NITRO_ENCODING.SelectedIndex;
            if (encIdx < 0 || FontEncodingInfos.Length <= encIdx)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ENCODING_NOT_SELECTED);
            }
#endif

            if (sheetPixels < 0)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_INVALID_SHEETPIXELS, sheetPixels);
            }

            var file = this.CurrentPath;
            var alter = this.GetAlterChar(NitroFontWriter.UseDefault);
            var linefeed = this.GetLineHeight(NitroFontWriter.UseDefault);
            var width = this.GetDefaultWidth(NitroFontWriter.UseDefault);
            var left = this.GetDefaultLeft(NitroFontWriter.UseDefault);
            var right = this.GetDefaultRight(NitroFontWriter.UseDefault);

            // file の拡張子を調整します。
            {
                var resExt = NormalFontExt;

                var ext = Path.GetExtension(file);

                if (string.Equals(ext, NormalFontExt, StringComparison.OrdinalIgnoreCase)
                    || string.Equals(ext, ArchiveFontExt, StringComparison.OrdinalIgnoreCase))
                {
                    ext = string.Empty;
                }

                ext += resExt;
                file = Path.ChangeExtension(file, ext);
                this.CurrentPath = file;
            }

            // ファイルの上書きチェック
            // キャンセルされたら例外が飛ぶ
            FileOverwriteConfirmDialog.Show(file);

            return _platformDetails.GetFontWriter(
                file,
                "",
                GlyphDataType.Texture,
                alter,
                linefeed,
                width,
                left,
                right,
                this.settings.Encoding,
                0,
                0,
                sheetPixels,
                ConverterEnvironment.IsOutputTargetLittleEndian,
                this.settings.IsOutKerningData);
        }

        private uint GetAlterChar()
        {
            uint alter;
            var c = this.settings.AlternateChar;
            if (c.Length == 1)
            {
                alter = c[0];
            }
            else
            {
                uint local = (uint)GlCm.ParseHexNumber(c);

                alter = GlCm.EncodingToUnicode(this.settings.Encoding, local);

                if (alter == RtConsts.InvalidCharCode)
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ALTER_TO_UNICODE, local);
                }
            }

            return alter;
        }

        private uint GetAlterChar(uint def)
        {
            return this.settings.UseDefaultAlternateChar ? def : this.GetAlterChar();
        }

        private int GetLineHeight(int def)
        {
            return this.settings.UseDefaultLineHeight ? def : this.settings.LineHeight;
        }

        private int GetDefaultWidth(int def)
        {
            return this.settings.UseDefaultWidth ? def : this.settings.DefaultWidth;
        }

        private int GetDefaultLeft(int def)
        {
            return this.settings.UseDefaultLeft ? def : this.settings.DefaultLeft;
        }

        private int GetDefaultRight(int def)
        {
            return this.settings.UseDefaultRight ? def : this.settings.DefaultRight;
        }
    }
}
