﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.IO;
    using System.Windows.Input;

    /// <summary>
    /// アプリケーションのメインウィンドウ用ViewModelです。
    /// </summary>
    public class MainWindowViewModel : WorkspaceViewModel, IProgressControl
    {
        private const string FileFilterStr = "Filter List (*.xllt)|*.xllt|all files (*.*)|*.*";

        private const string FileExtensionStr = "xllt";

        private const int ProgressBarMaxTick = 200;

        private readonly OrderFileListProxy orders = new OrderFileListProxy();
        private ProgressBarViewModel progressViewModel;
        private ReadOnlyCollection<InSettingViewModel> inpSettings;
        private ReadOnlyCollection<OutSettingViewModel> outSettings;
        private ConvertSettings settings;
        private RelayCommand convertCommand;
        private RelayCommand cancelCommand;
        private bool isCanceld;
        private bool isConvertReady = true;
        private int progressStep;
        private int progressMultiply;
        private string stateString;

        public MainWindowViewModel(ConvertSettings settings)
        {
            // プリファレンスが指定されているときは、そちらの名前を使う。
            var platformName = string.IsNullOrEmpty(ConverterEnvironment.UIPlatformPreferenceName)
                           ? ConverterEnvironment.PlatformDetails.UIPlatformName
                           : ConverterEnvironment.UIPlatformPreferenceName;
            this.DisplayName =
                GlCm.GetProductInfo()
                + " Version " + GlCm.GetVersionString(platformName);

            this.settings = settings;

            this.inpSettings = new ReadOnlyCollection<InSettingViewModel>(this.CreateInSettings());
            this.outSettings = new ReadOnlyCollection<OutSettingViewModel>(this.CreateOutSettings());

            this.SetPathHistory(this.settings.FilePaths);

            ProgressControl.SetInstance(this);
        }

        public ReadOnlyCollection<InSettingViewModel> InSettings
        {
            get
            {
                return this.inpSettings;
            }
        }

        /// <summary>
        /// 出力設定用のViewModelの読み取り専用のリストを取得します。
        /// </summary>
        public ReadOnlyCollection<OutSettingViewModel> OutSettings
        {
            get
            {
                return this.outSettings;
            }
        }

        public int InputMode
        {
            get
            {
                return this.settings.InputMode;
            }

            set
            {
                if (value == this.settings.InputMode)
                {
                    return;
                }

                this.settings.InputMode = value;
                this.OnPropertyChanged("InputMode");
            }
        }

        public int OutputMode
        {
            get
            {
                return this.settings.OutputMode;
            }

            set
            {
                if (value == this.settings.OutputMode)
                {
                    return;
                }

                this.settings.OutputMode = value;
                this.OnPropertyChanged("OutputMode");
            }
        }

        public bool UseFilter
        {
            get
            {
                return this.settings.UseFilter;
            }

            set
            {
                if (value == this.settings.UseFilter)
                {
                    return;
                }

                this.settings.UseFilter = value;
                this.OnPropertyChanged("UseFilter");
            }
        }

        public ProgressBarViewModel ProgressViewModel
        {
            get
            {
                if (null == this.progressViewModel)
                {
                    this.progressViewModel = new ProgressBarViewModel()
                    {
                        Value = 0,
                        Minimum = 0,
                        Maximum = 100,
                    };
                }

                return this.progressViewModel;
            }
        }

        /// <summary>
        /// 変換コマンドを取得します。
        /// </summary>
        public ICommand ConvertCommand
        {
            get
            {
                if (this.IsConvertReady)
                {
                    if (this.convertCommand == null)
                    {
                        this.convertCommand = new RelayCommand(
                            param => this.Convert(),
                            param => this.CanConvert);
                        this.convertCommand.Text = Strings.IDS_EXEC_BUTTON_DEFAULT;
                    }

                    return this.convertCommand;
                }
                else
                {
                    if (this.cancelCommand == null)
                    {
                        this.cancelCommand = new RelayCommand(
                            param => this.Cancel(),
                            param => this.CanCancel);
                        this.cancelCommand.Text = Strings.IDS_EXEC_BUTTON_CONVERTING;
                    }

                    return this.cancelCommand;
                }
            }
        }

        public bool IsConvertReady
        {
            get
            {
                return this.isConvertReady;
            }

            set
            {
                if (value == this.isConvertReady)
                {
                    return;
                }

                this.isConvertReady = value;
                this.OnPropertyChanged("IsConvertReady");
                this.OnPropertyChanged("ConvertCommand");
            }
        }

        public string StateString
        {
            get
            {
                return this.stateString;
            }

            set
            {
                this.stateString = value;
                this.OnPropertyChanged("StateString");
                ProcessMessage();
            }
        }

        public string FileFilter
        {
            get { return FileFilterStr; }
        }

        public string FileExtension
        {
            get { return FileExtensionStr; }
        }

        /// <summary>
        /// 変換を実行できるかどうかを判別する値を取得します。
        /// </summary>
        private bool CanConvert
        {
            get
            {
                return this.IsConvertReady;
            }
        }

        /// <summary>
        /// キャンセルを実行できるかどうかを判別する値を取得します。
        /// </summary>
        private bool CanCancel
        {
            get
            {
                return ! this.IsConvertReady;
            }
        }

        public void StateSave()
        {
            this.settings.FilePaths = new StringList(this.PathHistory);

            foreach (var setting in this.inpSettings)
            {
                setting.SaveState();
            }

            foreach (var setting in this.outSettings)
            {
                setting.SaveState();
            }
        }

        public override void SavePathToHistory()
        {
            if (this.UseFilter)
            {
                base.SavePathToHistory();
            }

            this.inpSettings[this.settings.InputMode].SavePathToHistory();
            this.outSettings[this.settings.OutputMode].SavePathToHistory();
        }

        public FontReader GetFontReader()
        {
            return this.inpSettings[this.settings.InputMode].GetFontReader();
        }

        public FontWriter GetFontWriter()
        {
            return this.outSettings[this.settings.OutputMode].GetFontWriter();
        }

        public void LoadFilter(CharFilter filter)
        {
            var filterFile = this.CurrentPath;

            if (filterFile.Length == 0)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_USE_FILTER_BUT_NO_FILTER);
            }

            filter.Load(filterFile, false);
        }

        public bool IsCanceled()
        {
            ProcessMessage();

            return this.isCanceld;
        }

        public void SetStatusString(string msgId)
        {
            this.StateString = msgId;
        }

        public void ResetProgressBarPos()
        {
            this.ProgressViewModel.Value = 0;
            ProcessMessage();

            this.progressStep = 0;
            this.progressMultiply = 1;
        }

        public void SetProgressBarMax(int max)
        {
            this.progressMultiply = 1;

            this.ProgressViewModel.Minimum = 0;
            this.ProgressViewModel.Maximum = max;

            while (max > ProgressBarMaxTick)
            {
                max /= 2;
                this.progressMultiply *= 2;
            }

            ProcessMessage();
        }

        public void StepProgressBar(int step)
        {
            int oldStep = this.progressStep;

            this.progressStep += step;

            if ((this.progressStep / this.progressMultiply) != (oldStep / this.progressMultiply))
            {
                this.ProgressViewModel.Value = this.progressStep;

                if (this.IsCanceled())
                {
                    throw new ConvertCancelException();
                }

                ProcessMessage();
            }
        }

        private static void ProcessMessage()
        {
            DispatcherHelper.DoEvents();
        }

        /// <summary>
        /// Saves the customer to the repository.  This method is invoked by the SaveCommand.
        /// </summary>
        private void Convert()
        {
            FontData font = new FontData();
            CharFilter filter = new CharFilter();
            GlyphOrder outputOrder = new GlyphOrder();
            FontReader reader = null;
            FontWriter writer = null;

            this.isCanceld = false;
            this.IsConvertReady = false;

            try
            {
                //---- 変換準備
                this.SetStatusString(Strings.IDS_STATUS_PREPAIR);
                this.PrepairConvert();

                //---- reader, writer の取得と入力値チェック
                reader = this.GetFontReader();
                writer = this.GetFontWriter();
                reader.ValidateInput();
                writer.ValidateInput();

                //---- 出力順序ファイルロード
                this.SetStatusString(Strings.IDS_STATUS_LOAD_ORDER);
                writer.GetGlyphOrder(outputOrder);

                //---- フィルタロード＆出力順序ファイル統合
                if (this.UseFilter)
                {
                    this.SetStatusString(Strings.IDS_STATUS_LOAD_FILTER);
                    this.LoadFilter(filter);
                }

                filter.SetOrder(outputOrder);
                //// filter.DumpFilter();    // for Debug

                // フォントデータロード
                reader.ReadFontData(font, filter);

                // フィルタでpassになっている文字がすべてあるか確認
                if (this.UseFilter)
                {
                    filter.CheckEqual(font);
                }

                // フォントデータの調整
                font.ReflectGlyph();

                // フォントデータ書き出し
                writer.WriteFontData(font, outputOrder);

                //---- 変換終了時処理
                this.SetStatusString(Strings.IDS_STATUS_FINISH);

                this.SavePathToHistory();
            }
            catch (GeneralException ge)
            {
                ProgressControl.Error(ge.GetMsg());
                Rpt._RPT1("Error: {0}\n", ge.GetMsg());

                this.SetStatusString(Strings.IDS_STATUS_ERROR);
                this.ResetProgressBarPos();
            }
            catch (ConvertCancelException)
            {
                // cancel
                this.SetStatusString(Strings.IDS_STATUS_CANCELED);
                this.ResetProgressBarPos();
            }
            catch (OutOfMemoryException /*e*/)
            {
                ProgressControl.Error(Strings.IDS_ERR_OVER_MEMORY);

                this.SetStatusString(Strings.IDS_STATUS_ERROR);
                this.ResetProgressBarPos();
            }
#if ! DEBUG
            catch(Exception e)
            {
                ProgressControl.Error(Strings.IDS_ERR_UNKNOWN_EXCEPTION + Environment.NewLine + e.Message + Environment.NewLine + e.StackTrace);
                SetStatusString(Strings.IDS_STATUS_ERROR);
                ResetProgressBarPos();
            }
#endif

            this.ResetProgressBarPos();
            this.IsConvertReady = true;
        }

        private void Cancel()
        {
            this.isCanceld = true;
        }

        private void PrepairConvert()
        {
            // 各入力が有効かチェックする
            this.ValidateInput();
        }

        private void ValidateInput()
        {
            if (this.UseFilter)
            {
                var filterFile = this.CurrentPath;

                if (filterFile == null)
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter,Strings.IDS_ERR_FILTER_NOT_EXISTS, string.Empty);
                }

                if (filterFile.Length != 0)
                {
                    if (!File.Exists(filterFile))
                    {
                        throw GlCm.ErrMsg(
                            ErrorType.Parameter,
                            Strings.IDS_ERR_FILTER_NOT_EXISTS,
                            filterFile);
                    }
                }
            }
        }

        private List<InSettingViewModel> CreateInSettings()
        {
            return new List<InSettingViewModel>
            {
                new InImageViewModel(this.settings.InImageSettings, this.orders),
                new InNitroViewModel(ConverterEnvironment.PlatformDetails, this.settings.InRuntimeBinarySettings),
                new InWinViewModel(this.settings.InWinSettings),
            };
        }

        private List<OutSettingViewModel> CreateOutSettings()
        {
            return new List<OutSettingViewModel>
            {
                new OutImageViewModel(this.settings.OutImageSettings, this.orders),
                new OutNitroViewModel(ConverterEnvironment.PlatformDetails, this.settings.OutRuntimeBinarySettings),
            };
        }
    }
}
