﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using NintendoWare.Font.Properties;
    using System.Windows;

    public class InImageViewModel : InSettingViewModel
    {
        private const string FileFilterStr = "Supported Images (*.bmp, *.tga, *.tif)|*.bmp;*.tga;*.tif"
                                              + "|Bitmap Images (*.bmp)|*.bmp"
                                              + "|Targa Images (*.tga)|*.tga"
                                              + "|TIFF Images (*.tif)|*.tif"
                                              + "|all files (*.*)|*.*";

        private readonly InImageSettings settings;
        private static GlyphImageFormatViewModel[] glyphImageFormats;
        private List<NnsFileInfo> orderInfos;
        private OrderFileListProxy orderFileListProxy;
        private bool isOrderInfosReady;

        public InImageViewModel(InImageSettings settings, OrderFileListProxy orderFileListProxy)
        {
            this.DisplayName = "Image";
            this.settings = settings;
            this.orderFileListProxy = orderFileListProxy;

            this.SetPathHistory(this.settings.FilePaths);
        }

        public static GlyphImageFormatViewModel[] GlyphImageFormats
        {
            get
            {
                if (glyphImageFormats == null)
                {
                    var list = new List<GlyphImageFormatViewModel>((int)GlyphImageFormat.Max); //
                    var imageInputDescList = ConverterEnvironment.PlatformDetails.GetImageInputFormatFormatList();

                    foreach (var imageInputDesc in imageInputDescList)
                    {
                        if (list.Find((item) => item.Info.GlyphImageFormat == imageInputDesc.GlyphImageFormat) != null) { continue; }
                        list.Add(new GlyphImageFormatViewModel(imageInputDesc.Description, GlyphImageInfo.GetGlyphImageInfo(imageInputDesc.KeyGlyphImageFormat)));
                    }

                    glyphImageFormats = list.ToArray();
                }

                return glyphImageFormats;
            }
        }

        public string OrderFile
        {
            get
            {
                if (!this.isOrderInfosReady)
                {
                    return null;
                }

                return this.settings.OrderFile;
            }

            set
            {
                if (value == null)
                {
                    Debug.WriteLine("OrderFile property set - null");
                    return;
                }

                if (!this.isOrderInfosReady)
                {
                    return;
                }

                if (value == this.settings.OrderFile)
                {
                    return;
                }

                this.settings.OrderFile = value;
                this.OnPropertyChanged("OrderFile");
            }
        }

        public List<NnsFileInfo> OrderInfos
        {
            get
            {
                if (null == this.orderInfos)
                {
                    this.orderFileListProxy.Loaded += this.OrderFileListProxy_Loaded;
                    this.orderInfos = this.orderFileListProxy.OrderFiles;
                    this.isOrderInfosReady = this.orderFileListProxy.IsLoaded;
                    this.IsReady = this.isOrderInfosReady;
                }

                return this.orderInfos;
            }

            private set
            {
                this.orderInfos = value;
                this.OnPropertyChanged("OrderInfos");
                this.isOrderInfosReady = true;
            }
        }

        public GlyphImageFormat GlyphImageFormat
        {
            get
            {
                return this.settings.GlyphImageFormat;
            }

            set
            {
                if (value == this.settings.GlyphImageFormat)
                {
                    return;
                }

                this.settings.GlyphImageFormat = value;
                this.OnPropertyChanged("GlyphImageFormat");
                this.OnPropertyChanged("IsGlyphImageFormatPacked");
            }
        }

        public bool IsGlyphImageFormatPacked
        {
            get { return this.GlyphImageFormat == GlyphImageFormat.LA4Packed; }
        }

        public bool AntiLinear
        {
            get
            {
                return this.settings.AntiLinear;
            }

            set
            {
                if (value == this.settings.AntiLinear)
                {
                    return;
                }

                this.settings.AntiLinear = value;
                this.OnPropertyChanged("AntiLinear");
            }
        }

        public string FileFilter
        {
            get { return FileFilterStr; }
        }

        public override FontReader GetFontReader()
        {
            if (!this.IsReady)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_NOW_READING);
            }

            var orderPath = this.OrderFile;
            if (orderPath == null)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ORDER_NOT_SELECTED);
            }

            orderPath = Path.Combine(this.orderFileListProxy.OrderFilesDir, orderPath);

            return
                new ImageFontReader(
                    this.CurrentPath,
                    orderPath,
                    this.GlyphImageFormat,
                    this.AntiLinear);
        }

        public override void SaveState()
        {
            this.settings.FilePaths = new StringList(this.PathHistory);
        }

        private void OrderFileListProxy_Loaded(object sender, EventArgs e)
        {
            this.OrderInfos = this.orderFileListProxy.OrderFiles;

            // 現在の値に一致する項目が無い場合は先頭の内容にする。
            var currentOrderFile = this.settings.OrderFile;
            var found = this.OrderInfos.Find((orderInfo) => orderInfo.FileName == currentOrderFile);
            if (null == found)
            {
                this.settings.OrderFile = this.OrderInfos[0].FileName;
            }

            this.OnPropertyChanged("OrderFile");

            this.IsReady = true;
        }
    }
}
