﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font.Properties
{
    internal sealed partial class Settings
    {
        public static Settings DefaultInstance { get; set; }
    }
}

namespace NintendoWare.Font
{
    using System;
    using System.IO;
    using System.Runtime.InteropServices;
    using System.Windows;
    using System.Xml.Serialization;

    using NintendoWare.Font.Properties;

    /// <summary>
    /// Settingsの内容をXMLに保存するためのクラスです。
    /// </summary>
    public class UserSettings
    {
        const string _SettingFileName = "setting.xml";

        static UserSettingParams _UserSettingParams = new UserSettingParams();

        /// <summary>
        /// ユーザーのアプリケーションデータのパスを取得
        /// </summary>
        private static string LocalAppDataDirectory
        {
            get
            {
                var folder = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData), "Nintendo", "FontConverter");
                if (!Directory.Exists(folder))
                {
                    Directory.CreateDirectory(folder);
                }
                return folder;
            }
        }

        /// <summary>
        /// Settingファイルのパスを取得
        /// </summary>
        private static string SettingFilePath
        {
            get
            {
                return LocalAppDataDirectory + "\\" + _SettingFileName;
            }
        }

        /// <summary>
        /// SettingsをXMLに出力します。
        /// </summary>
        public static void SaveSettings()
        {
            try
            {
                XmlSerializer serializer = XmlSerializerCache.GetXmlSerializer(typeof(UserSettingParams));
                using (FileStream fs = new FileStream(SettingFilePath, FileMode.Create))
                {
                    serializer.Serialize(fs, _UserSettingParams.GetParams());
                }
            }
            catch (Exception e)
            {
                MessageBox.Show(
                    e.Message,
                    Strings.IDS_WARNING,
                    MessageBoxButton.OK,
                    MessageBoxImage.Warning);
            }
        }

        /// <summary>
        /// XMLからSettingsを読み込みます
        /// </summary>
        public static void LoadSettings()
        {
            try
            {
                XmlSerializer serializer = XmlSerializerCache.GetXmlSerializer(typeof(UserSettingParams));
                using (FileStream fs = new FileStream(SettingFilePath, FileMode.Open, FileAccess.Read))
                {
                    UserSettingParams settings = XmlSerializerCache.Deserialize(serializer, fs) as UserSettingParams;
                    _UserSettingParams.SetParams(settings);
                }
            }
            catch (FileNotFoundException)
            {
                // FileNotFoundExceptionの場合はスルーします
            }
            catch (Exception e)
            {
                MessageBox.Show(
                    e.Message,
                    Strings.IDS_WARNING,
                    MessageBoxButton.OK,
                    MessageBoxImage.Warning);
            }
        }
    }

    /// <summary>
    /// Settingsを包括するクラスです。
    /// FontConverterがバージョンアップした場合Settingsの内容が失われてしまうため、
    /// XMLファイルとしてSettingsの内容を保持します。
    /// Settingsにパラメータを追加する場合はこのクラスにもパラメータを追加してください。
    /// </summary>
    [Serializable]
    [StructLayout(LayoutKind.Sequential)]
    public class UserSettingParams
    {
        const string currentVersion = "1.0";

        [XmlAttribute]
        public string version { get; set; }

        public global::NintendoWare.Font.Win32.WINDOWPLACEMENT WindowPlacement;
        public bool IsWarningFontLicence;
        public global::NintendoWare.Font.Win32.WINDOWPLACEMENT WarningWindowPlacement;
        public global::NintendoWare.Font.ConvertSettings ConvertSettings;

        public UserSettingParams()
        {
            // Settingsインスタンスを生成します
            if (Settings.DefaultInstance == null)
            {
                Settings.DefaultInstance = new Settings();
            }

            // バージョンによる排他を行なわないメンバは初期化します
            IsWarningFontLicence = false;
            ConvertSettings = new global::NintendoWare.Font.ConvertSettings();
        }

        public void SetParams(UserSettingParams settings)
        {
            // バージョンによる排他を行うメンバ
            if (string.Compare(settings.version, currentVersion) == 0)
            {
                Settings.DefaultInstance.WindowPlacement = settings.WindowPlacement;
                Settings.DefaultInstance.WarningWindowPlacement = settings.WarningWindowPlacement;
            }

            // バージョンによる排他を行なわないメンバ
            Settings.DefaultInstance.IsWarningFontLicence = settings.IsWarningFontLicence;
            Settings.DefaultInstance.ConvertSettings = settings.ConvertSettings;
        }

        public UserSettingParams GetParams()
        {
            UserSettingParams retParam = new UserSettingParams();
            retParam.version = currentVersion;

            retParam.WindowPlacement = Settings.DefaultInstance.WindowPlacement;
            retParam.IsWarningFontLicence = Settings.DefaultInstance.IsWarningFontLicence;
            retParam.WarningWindowPlacement = Settings.DefaultInstance.WarningWindowPlacement;
            retParam.ConvertSettings = Settings.DefaultInstance.ConvertSettings;

            return retParam;
        }
    }
}
