﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.IO;
    using System.Xml;

    public class NnsFileInfo : IComparable<NnsFileInfo>
    {
        public string Title { get; set; }

        public string Comment { get; set; }

        public string Path { get; set; }

        public string FileName { get; set; }

        public int CompareTo(NnsFileInfo rhs)
        {
            var result = string.Compare(this.Title, rhs.Title, StringComparison.Ordinal);
            if (result != 0)
            {
                return result;
            }

            return string.Compare(this.FileName, rhs.FileName, StringComparison.OrdinalIgnoreCase);
        }
    }

    /// <summary>
    /// 文字順序ファイル xlor
    /// </summary>
    public class NnsXmlFile : HandlerBase
    {
        private const string TagLetterOrder = "letter-order";
        private const string TagHeader = "head";
        private const string TagTitle = "title";
        private const string TagComment = "comment";

        private const string AttlibVersion = "version";
        private const string XlorVersion = "1.0";

        private bool isSuccess;
        private bool isInHeader;
        private bool isInComment;
        private bool isInTitle;
        private int tagLevel;
        private NnsFileInfo nnsFileInfo;

        public NnsXmlFile()
        {
            this.isSuccess = false;
            this.tagLevel = 0;
            this.isInHeader = false;
            this.isInComment = false;
            this.isInTitle = false;
            this.nnsFileInfo = null;
        }

        public bool IsSuccess()
        {
            return this.isSuccess;
        }

        // 順序定義ファイルを読み込む
        public void Load(NnsFileInfo nnsFileInfo, string filePath)
        {
            this.isSuccess = false;
            this.isInHeader = false;
            this.isInComment = false;
            this.isInTitle = false;
            this.tagLevel = 0;
            this.nnsFileInfo = nnsFileInfo;
            nnsFileInfo.Path = filePath;
            nnsFileInfo.FileName = Path.GetFileName(filePath);

            try
            {
                using (var reader = CreateReader(filePath, false))
                {
                    Parse(reader);
                }
            }
            catch (HandlerBaseException)
            {
                this.nnsFileInfo = null;

                if (Error != null)
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_MSG, Error);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_UNKNOWN);
                }
            }
            catch (XmlException ex)
            {
                this.nnsFileInfo = null;
                throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_XERCESC_MSG, ex.Message);
            }
            catch
            {
                this.nnsFileInfo = null;
                throw;
            }

            this.nnsFileInfo = null;
            this.isSuccess = true;
        }

        // ---------------------------------------------------------------------------
        //  PParseHandlers: Overrides of the SAX DocumentHandler interface
        // ---------------------------------------------------------------------------
        protected override void StartElement(string name, AttributeList attributes)
        {
            if (name == TagComment)
            {
                // <comment>
                if (this.isInHeader && this.tagLevel == 2)
                {
                    this.isInComment = true;
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagComment, TagHeader);
                }
            }
            else if (name == TagTitle)
            {
                // <title>
                if (this.isInHeader && this.tagLevel == 2)
                {
                    this.isInTitle = true;
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagTitle, TagHeader);
                }
            }
            else if (name == TagHeader)
            {
                // <header>
                if (this.tagLevel == 1)
                {
                    this.isInHeader = true;
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.Xml, Strings.IDS_ERR_TAG_MUST_BE_IN, TagHeader, TagLetterOrder);
                }
            }

            this.tagLevel++;
        }

        protected override void Characters(string chars, int length)
        {
            if (this.isInComment)
            {
                this.nnsFileInfo.Comment += chars;
            }
            else if (this.isInTitle)
            {
                this.nnsFileInfo.Title += chars;
            }
        }

        protected override void EndElement(string name)
        {
            this.tagLevel--;

            if (name == TagComment)
            {
                // <comment>
                this.isInComment = false;
            }
            else if (name == TagTitle)
            {
                // <title>
                this.isInTitle = false;
            }
            else if (name == TagHeader)
            {
                // <header>
                this.isInHeader = false;
            }
        }
    }
}
