﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.ComponentModel;
    using System.Runtime.InteropServices;
    using System.Windows;
    using System.Windows.Interop;
    using NintendoWare.Font.Properties;
    using NintendoWare.Font.Win32;

    /// <summary>
    /// アプリケーションのメインウィンドウのViewです。
    /// </summary>
    public partial class MainWindow : Window
    {
        public const string FontBinaryTabTitleStr = LibFormat.ExtensionFont + " / " + LibFormat.ExtensionArchiveFont;

        private WarningWindow warnWnd;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MainWindow()
        {
            this.warnWnd = new WarningWindow(this);

            InitializeComponent();
        }

        /// <summary>
        /// コンストラクタのタイミングで行えない初期化を行います。
        /// </summary>
        public void Init()
        {
#if ORIGINAL_TOOLTIP_CODE
            //---- Init ToolTip
            {
                m_tip.Create(m_hInstance, this);
            }
#endif
            var viewModel = (MainWindowViewModel)this.DataContext;
            viewModel.PropertyChanged += this.ViewModel_PropertyChanged;

            ProgressControl.GetInstance().SetStatusString(Strings.IDS_STATUS_INITIAL);

#if UI_VALUE_VALIDATION
            //---- tab dialogs
            this.InitTabCtrlInput();
            this.InitTabCtrlOutput();
#endif

            return;
        }

        internal void StateSave()
        {
            ((MainWindowViewModel)this.DataContext).StateSave();
        }

        private void CDlgWin_SourceInitialized(object sender, EventArgs e)
        {
            var wp = Settings.DefaultInstance.WindowPlacement;

            // まだ設定ファイルが一度も保存されていない
            if (wp == null)
            {
                return;
            }

            wp.length = (uint)Marshal.SizeOf(typeof(WINDOWPLACEMENT));
            wp.flags = 0;
            wp.showCmd =
                wp.showCmd == ShowWindowCommand.ShowMinimized ?
                    ShowWindowCommand.ShowNormal :
                    wp.showCmd;
            User.SetWindowPlacement(new WindowInteropHelper(this).Handle, wp);
        }

        // WARNING - Not fired when Application.SessionEnding is fired
        private void CDlgWin_Closing(object sender, CancelEventArgs e)
        {
            this.warnWnd.Close();

            // Persist window placement details to application settings
            var wp = new WINDOWPLACEMENT();
            User.GetWindowPlacement(new WindowInteropHelper(this).Handle, wp);
            Settings.DefaultInstance.WindowPlacement = wp;
        }

        private void ViewModel_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            var isConvertReady = "IsConvertReady";

            var viewModel = (MainWindowViewModel)sender;
            viewModel.VerifyPropertyName(isConvertReady);

            if (e.PropertyName == isConvertReady)
            {
                if (!viewModel.IsConvertReady)
                {
                    // 警告ウインドウをクリアする
                    this.warnWnd.ClearMessage();
                }
                else
                {
                    // メッセージが無い場合は隠す
                    if (this.warnWnd.IsVisible && !this.warnWnd.HasMessage)
                    {
                        this.warnWnd.Hide();
                    }
                }
            }
        }

#if UI_VALUE_VALIDATION
        private enum IN
        {
            BMP,
            NITRO,
            WIN,
            NUM_OF_IN_PLANES,
        }

        private enum OUT
        {
            BMP,
            NITRO,
            NUM_OF_OUT_PLANES,
        }

        private void InitTabCtrlInput()
        {
            var inputSelect = Properties.Settings.Default.InputMode;
            LimitRange(ref inputSelect, 0, (int)IN.NUM_OF_IN_PLANES - 1);

            IDC_INPUT.SelectedIndex = inputSelect;
        }

        private void InitTabCtrlOutput()
        {
            var outputSelect = Settings.Default.OutputMode;
            LimitRange(ref outputSelect, 0, (int)OUT.NUM_OF_OUT_PLANES - 1);

            IDC_OUTPUT.SelectedIndex = outputSelect;
        }

        private static void LimitRange(ref int num, int min, int max)
        {
            if (num < min)
            {
                num = min;
            }

            if (num > max)
            {
                num = max;
            }
        }
#endif
    }
}
