﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <perf_MeterDrawer.h>

namespace nn
{
namespace perf
{

float MeterDrawer::GetAdjustedHeight() const NN_NOEXCEPT
{
    if ( m_BarHeight > 0.f )
    {
        if ( m_RootMeter->m_LoadMeterList.size() > 0 )
        {
            return m_BarHeight * m_RootMeter->m_LoadMeterList.size();
        }
        else
        {
            return m_BarHeight;
        }
    }

    return m_Height;
}

void MeterDrawer::Draw( nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* renderer, LoadMeterBase* rootMeter) NN_NOEXCEPT
{
    m_RootMeter = rootMeter;
    if ( !m_Visible )
    {
        return;
    }

    // フレームの区切り本数を計算します。
    if ( m_SectionNum == 0 )
    {
        uint32_t currentNum = CalculateMaxSectionNum();

        if ( m_CurrentSectionNum <= currentNum )
        {
            m_AdjustCounter = m_AdjustTime;
            m_CurrentSectionNum = currentNum;
        }
        else
        {
            if ( m_AdjustCounter > 0 )
            {
                m_AdjustCounter--;
            }
            else
            {
                m_CurrentSectionNum = currentNum;
            }
        }
    }

    uint32_t sectionNum = m_SectionNum;

    if ( sectionNum == 0 )
    {
        sectionNum = m_CurrentSectionNum;
    }
    if ( sectionNum < m_MinSectionNum )
    {
        sectionNum = m_MinSectionNum;
    }

    // メーター全体の高さ、各メーターの高さを計算します。
    float height = GetAdjustedHeight();
    float barHeight = ( m_BarHeight > 0.f ? m_BarHeight : height / m_RootMeter->m_LoadMeterList.size() );

    const LoadMeterBase::Section& sectionFrame = m_RootMeter->GetLastResult( 0 );
    uint64_t referenceFrameTime = nn::os::Tick( m_ReferenceFrame ).GetInt64Value();

    renderer->SetDefaultParameters();
    renderer->SetDepthStencilState( pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWrite );

    {
        // 全体の枠を描画します。
        {
            nn::util::Uint8x4 color;
            color.v[0] = m_BorderColor.GetR();
            color.v[1] = m_BorderColor.GetG();
            color.v[2] = m_BorderColor.GetB();
            color.v[3] = m_BorderColor.GetA();
            renderer->Draw2DRect( pCommandBuffer,
                m_Position.x, m_Position.y,
                m_Width, height );
        }

        // 親負荷メーターの処理時間を描画します。
        {
            for ( int i = m_RootMeter->GetLastSectionNum() - 1 ; i >= 0; --i )
            {
                const LoadMeterBase::Section& section = m_RootMeter->GetLastResult( i );

                float positionLeft = CalculateTimeLinePosition( m_Width, sectionNum, section.begin.GetInt64Value() );
                float positionRight = CalculateTimeLinePosition( m_Width, sectionNum, section.end.GetInt64Value() );

                const float rectX = m_Position.x + positionLeft;
                const float rectY = m_Position.y;
                const float rectWidth  = positionRight - positionLeft;
                const float rectHeight = height;

                nn::util::Uint8x4 color;
                color.v[0] = section.color.GetR();
                color.v[1] = section.color.GetG();
                color.v[2] = section.color.GetB();
                color.v[3] = section.color.GetA();
                renderer->Draw2DRect( pCommandBuffer,
                    rectX, rectY,
                    rectWidth, rectHeight );
            }
        }

        // 登録されている子負荷メーターを描画します。
        int listNum = 0;
        for ( LoadMeterList::iterator it = m_RootMeter->m_LoadMeterList.begin(); it != m_RootMeter->m_LoadMeterList.end(); ++it )
        {
            for ( int i = it->GetLastSectionNum() - 1 ; i >= 0; --i )
            {
                const LoadMeterBase::Section& section = it->GetLastResult( i );

                float positionLeft = CalculateTimeLinePosition( m_Width, sectionNum, section.begin.GetInt64Value() );
                float positionRight = CalculateTimeLinePosition( m_Width, sectionNum, section.end.GetInt64Value() );
                float adjustedBarHeight = barHeight * ( section.parent >= 0 ? 0.7f : 0.9f );;

                const float rectX = m_Position.x + positionLeft;
                const float rectY = m_Position.y + barHeight * listNum + ( barHeight - adjustedBarHeight );
                const float rectWidth  = positionRight - positionLeft;
                const float rectHeight = adjustedBarHeight;

                nn::util::Uint8x4 color;
                color.v[0] = section.color.GetR();
                color.v[1] = section.color.GetG();
                color.v[2] = section.color.GetB();
                color.v[3] = section.color.GetA();
                if(rectWidth > m_DrawThreshold)
                {
                    renderer->Draw2DRect( pCommandBuffer,
                        rectX, rectY,
                        rectWidth, rectHeight );
                }
            }
            ++listNum;
        }

        // フレームの区切り線を描画します。
        {
            uint64_t tick = sectionFrame.begin.GetInt64Value();
            uint32_t sectionLineNum = sectionNum;

            if ( m_MaxSectionSeparatorNum != 0 && sectionLineNum > m_MaxSectionSeparatorNum )
            {
                sectionLineNum = m_MaxSectionSeparatorNum;
            }

            for ( uint32_t i = 0; i < sectionLineNum - 1; ++i )
            {
                tick += referenceFrameTime;

                float posX = CalculateTimeLinePosition( m_Width, sectionNum, tick );

                nn::util::Uint8x4 color;
                color.v[0] = nn::util::Color4u8::Black().GetR();
                color.v[1] = nn::util::Color4u8::Black().GetG();
                color.v[2] = nn::util::Color4u8::Black().GetB();
                color.v[3] = nn::util::Color4u8::Black().GetA();
                renderer->Draw2DRect( pCommandBuffer,
                    m_Position.x + posX, m_Position.y,
                    2, height );
            }
        }
    }

    // メーターの名前と処理率を出力します。
    //未動作確認のためコメントアウト
//    if ( m_TextVisible && m_TextWriter )
//    {
//        m_TextWriter->SetFontSize( m_FontSize );
//        float scale = barHeight / m_TextWriter->GetLineHeight();
//        m_TextWriter->SetScale( scale < 1.f ? scale : 1.f );
//        m_TextWriter->SetTextColor( m_TextColor );
//        m_TextWriter->SetShadowAlpha( 128 );
//
//        int listNum = 0;
//        for ( LoadMeterList::iterator it = m_RootMeter->m_LoadMeterList.begin(); it != m_RootMeter->m_LoadMeterList.end(); ++it )
//        {
//            float rate = 100.f * it->GetLastTotalSpan().GetInt64Value() / referenceFrameTime;
//            m_TextWriter->SetCursor( m_Position.x + 2.f, m_Position.y + barHeight * listNum );
//            m_TextWriter->SetCursor( m_Position.x, m_Position.y );
//            //m_TextWriter->Print( "%s ( %.2f%% )", it->GetName(), rate );
//            m_TextWriter->Print( it->GetName(), 20 );
//            ++listNum;
//        }
//        m_TextWriter->Flush();
//    }
} //NOLINT(impl/function_size)

float MeterDrawer::CalculateTimeLinePosition( float maxWidth, uint32_t sectionNum, uint64_t time ) NN_NOEXCEPT
{
    const LoadMeterBase::Section& sectionFrame = m_RootMeter->GetLastResult( 0 );
    return CalculateTimeLineWidth( maxWidth, sectionNum, time - static_cast<uint64_t>( sectionFrame.begin.GetInt64Value() ) );
}

float MeterDrawer::CalculateTimeLineWidth( float maxWidth, uint32_t sectionNum, uint64_t time ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( sectionNum > 0 );

    uint64_t totalTime = nn::os::Tick( m_ReferenceFrame ).GetInt64Value() * sectionNum;
    return maxWidth * time / totalTime;
}

uint32_t MeterDrawer::CalculateMaxSectionNum() NN_NOEXCEPT
{
    uint64_t referenceFrameTime = nn::os::Tick( m_ReferenceFrame ).GetInt64Value();
    uint32_t sectionNum = 1;

    {
        uint64_t  rootMeterTotalTime = static_cast<uint64_t>( m_RootMeter->GetLastFinalEnd().GetInt64Value() - m_RootMeter->GetLastResult( 0 ).begin.GetInt64Value() );
        sectionNum = static_cast<uint32_t>(  rootMeterTotalTime / referenceFrameTime ) + 1;
    }

    for ( LoadMeterList::iterator it = m_RootMeter->m_LoadMeterList.begin(); it != m_RootMeter->m_LoadMeterList.end(); ++it )
    {
        uint64_t meterTotalTime = static_cast<uint64_t>( it->GetLastFinalEnd().GetInt64Value() - it->GetLastResult( 0 ).begin.GetInt64Value() );

        if (  meterTotalTime > 0 )
        {
            uint32_t num = static_cast<uint32_t>(  meterTotalTime / referenceFrameTime ) + 1;

            if ( sectionNum < num )
            {
                sectionNum = num;
            }
        }
    }
    return sectionNum;
}

} // namespace perf
} // namespace nns
