﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "detail/util_VisualizeParticleFillHotSpot.h"
#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif


detail::VisualizeParticleFillHotSpot* detail::VisualizeParticleFillHotSpot::g_pVisualizeParticleFillHotSpot = nullptr;

namespace detail {

//---------------------------------------------------------------------------
//      パーティクルの重なりチェック用クラスを初期化する
//---------------------------------------------------------------------------
void VisualizeParticleFillHotSpot::Initialize( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_pVisualizeParticleFillHotSpot == nullptr );

    void* buffer = pHeap->Alloc( sizeof( detail::VisualizeParticleFillHotSpot ) );
    if ( !buffer )
    {
        nn::vfx::detail::OutputWarning( "VisualizeParticleFillHotSpot : Memory Allocation Error!! : %d\n", sizeof( detail::VisualizeParticleFillHotSpot ) );
        return;
    }

    g_pVisualizeParticleFillHotSpot = new ( buffer )detail::VisualizeParticleFillHotSpot();
    if ( !g_pVisualizeParticleFillHotSpot )
    {
        nn::vfx::detail::OutputWarning( "VisualizeParticleFillHotSpot : Creation Error!!\n" );
        return;
    }

    // パーティクル重なりチェック用のテクスチャ作成
    g_pVisualizeParticleFillHotSpot->GenerateHotSpotAnalyzeTexture( pDevice, pHeap );

    // パーティクル重なりチェック用サンプラを作成
    {
        nn::gfx::Sampler::InfoType info;
        info.SetDefault();
        info.SetFilterMode( nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint );
        info.SetAddressU( nn::gfx::TextureAddressMode_Mirror );
        info.SetAddressV( nn::gfx::TextureAddressMode_Mirror );
        info.SetAddressW( nn::gfx::TextureAddressMode_Mirror );
        g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeSampler.Initialize( pDevice, info );
    }

    // ブレンドステート作成
    g_pVisualizeParticleFillHotSpot->GenerateAddBlendState( pDevice );

    // ラスタライザステートを初期化
    {
        nn::gfx::RasterizerState::InfoType info;
        info.SetDefault();
        info.SetCullMode( nn::gfx::CullMode_None );
        g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeRasterizerState.Initialize( pDevice, info );
    }

    // 深度ステンシルステートを初期化
    {
        nn::gfx::DepthStencilState::InfoType info;
        info.SetDefault();
        info.SetDepthTestEnabled( false );
        info.SetDepthWriteEnabled( false );
        g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeDepthStencilState.Initialize( pDevice, info );
    }
}

//---------------------------------------------------------------------------
//      パーティクルの重なりチェック用クラスを破棄する
//---------------------------------------------------------------------------
void VisualizeParticleFillHotSpot::Finalize( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_pVisualizeParticleFillHotSpot != nullptr );

    g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeSampler.Finalize( pDevice );
    g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeTexture.Finalize( pDevice );
    g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeTextureView.Finalize( pDevice );
    g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeTextureMemoryPool.Finalize( pDevice );
    g_pVisualizeParticleFillHotSpot->m_AddBlendState.Finalize( pDevice );

    g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeRasterizerState.Finalize( pDevice );
    g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeDepthStencilState.Finalize( pDevice );

    if ( g_pVisualizeParticleFillHotSpot->m_pHotSpotAnalyzeTextureBufferPtr )
    {
        pHeap->Free( g_pVisualizeParticleFillHotSpot->m_pHotSpotAnalyzeTextureBufferPtr );
        g_pVisualizeParticleFillHotSpot->m_pHotSpotAnalyzeTextureBufferPtr = nullptr;
    }
}

//------------------------------------------------------------------------------
//      描画パス 描画設定コールバック
//------------------------------------------------------------------------------
void VisualizeParticleFillHotSpot::DrawPathRenderStateSetCallback( nn::vfx::RenderStateSetArg& arg ) NN_NOEXCEPT
{
    if ( g_pVisualizeParticleFillHotSpot )
    {
        g_pVisualizeParticleFillHotSpot->SetParticleHotSpotAnalyzingTexture( arg );
    }
}

//---------------------------------------------------------------------------
//      半透明のテクスチャを設定
//---------------------------------------------------------------------------
void VisualizeParticleFillHotSpot::SetParticleHotSpotAnalyzingTexture( nn::vfx::RenderStateSetArg& arg ) NN_NOEXCEPT
{
    nn::vfx::Emitter* pEmitter = arg.pEmitter;
    nn::vfx::detail::Shader* pShader = pEmitter->GetShader( arg.pDrawParameterArg->m_ShaderType );
    // テクスチャを入れ替える
    for( int i = 0; i < 3; i++ )
    {
        nn::gfx::DescriptorSlot textureDescSlot = pEmitter->GetEmitterResource()->m_TextureDescSlot[ i ];
        if ( textureDescSlot.IsValid() )
        {
            int pixelLoc  = pShader->GetPixelTextureSamplerLocation( static_cast<nn::vfx::TextureSlotId>( i ) );
            if( pixelLoc != -1 )
            {
                arg.pCommandBuffer->SetTextureAndSampler( pixelLoc, nn::gfx::ShaderStage_Pixel, m_TextureDesSlot, m_SamplerDescSlot );
            }
        }
    }

    // ブレンドステートを入れ替える
    arg.pCommandBuffer->SetBlendState( &m_AddBlendState );

    // ラスタライザステートを入れ替える
    arg.pCommandBuffer->SetRasterizerState( &m_HotSpotAnalyzeRasterizerState );

    // 深度ステンシルステートを入れ替える
    arg.pCommandBuffer->SetDepthStencilState( &m_HotSpotAnalyzeDepthStencilState );
}

//---------------------------------------------------------------------------
//      パーティクルの重なりチェック用のテクスチャ作成
//---------------------------------------------------------------------------
void VisualizeParticleFillHotSpot::GenerateHotSpotAnalyzeTexture( nn::gfx::Device* pDevice, nn::vfx::Heap* heap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );

    static const int OverDrawTextureWidth  = 32;
    static const int OverDrawTextureHeight = 32;

    nn::gfx::Texture::InfoType texInfo;
    texInfo.SetDefault();
    texInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_Texture );
    texInfo.SetWidth( OverDrawTextureWidth );
    texInfo.SetHeight( OverDrawTextureHeight );
    texInfo.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    texInfo.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );

    nn::gfx::MemoryPool::InfoType mpInfo;
    mpInfo.SetDefault();
    mpInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached );

    size_t texAlign     = nn::gfx::Texture::CalculateMipDataAlignment( pDevice, texInfo );
    size_t poolAlign    = nn::gfx::MemoryPool::GetPoolMemoryAlignment( pDevice, mpInfo );

    size_t textureSize  = OverDrawTextureWidth * OverDrawTextureHeight * 4;
    size_t allocedSize  = textureSize + texAlign;
    allocedSize         = nn::util::align_up( allocedSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( pDevice, mpInfo ) );

    m_pHotSpotAnalyzeTextureBufferPtr = heap->Alloc( allocedSize, poolAlign );
    NN_SDK_ASSERT_NOT_NULL( m_pHotSpotAnalyzeTextureBufferPtr );

    mpInfo.SetPoolMemory( m_pHotSpotAnalyzeTextureBufferPtr, allocedSize );
    m_HotSpotAnalyzeTextureMemoryPool.Initialize( pDevice, mpInfo );

#if NN_GFX_IS_TARGET_GL
    void* pTextureImage = nn::util::BytePtr( m_pHotSpotAnalyzeTextureBufferPtr ).Advance( texAlign ).Get();
#endif

#if NN_GFX_IS_TARGET_NVN
    uint32_t* tempBuff = reinterpret_cast<uint32_t *>( heap->Alloc( OverDrawTextureHeight * OverDrawTextureWidth * sizeof( uint32_t  ) ) );
    NN_SDK_ASSERT_NOT_NULL( tempBuff );
#endif
    const int pitchY = OverDrawTextureHeight;
    // アルファの入った薄い赤いダミーテクスチャ作成
    for( int y = 0; y < OverDrawTextureHeight; y++ )
    {
        for( int x = 0; x < OverDrawTextureWidth; x++ )
        {
            const int i = ( y * pitchY ) + x;
            const uint8_t r = 25;
            const uint8_t g = 0;
            const uint8_t b = 0;
            const uint8_t a = 80;

#if NN_GFX_IS_TARGET_GL
            // ARGB の順番で配置
            const uint32_t color = ( a << 24 ) | ( b << 16 ) | ( g << 8 ) | ( r << 0 );
            ( reinterpret_cast< uint32_t* >( pTextureImage ) )[ i ] = color;
#elif NN_GFX_IS_TARGET_NVN
            // ARGB の順番で配置
            const uint32_t color = ( a << 24 ) | ( b << 16 ) | ( g << 8 ) | ( r << 0 );
            tempBuff[ i ] = color;
#else
    #error "未対応の低レベル API"
#endif
        }
    }
    texInfo.SetMipCount( 1 );
    m_HotSpotAnalyzeTexture.Initialize( pDevice, texInfo, &m_HotSpotAnalyzeTextureMemoryPool, texAlign, textureSize );

    {
        nn::gfx::TextureView::InfoType info;
        info.SetDefault();
        info.SetTexturePtr( &m_HotSpotAnalyzeTexture );
        info.SetImageDimension( nn::gfx::ImageDimension_2d );
        info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );

        m_HotSpotAnalyzeTextureView.Initialize( pDevice, info );
    }
#if NN_GFX_IS_TARGET_NVN
    // 一時領域から Texture へ画像データを CPU 上で転送
    NVNcopyRegion region;
    region.xoffset = region.yoffset = region.zoffset = 0;
    region.width = region.height = 32;
    region.depth = 1;
    nvnTextureWriteTexels( m_HotSpotAnalyzeTexture.ToData()->pNvnTexture, nullptr, &region, tempBuff );
    nvnTextureFlushTexels( m_HotSpotAnalyzeTexture.ToData()->pNvnTexture, nullptr, &region );
    heap->Free( tempBuff );
    tempBuff = NULL;
#endif

}


//---------------------------------------------------------------------------
//   ブレンドステート作成
//---------------------------------------------------------------------------
void VisualizeParticleFillHotSpot::GenerateAddBlendState( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();
    nn::gfx::BlendTargetStateInfo targetInfo;

    targetInfo.SetDefault();
    targetInfo.SetBlendEnabled( ( true ) );

    targetInfo.SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    targetInfo.SetDestinationColorBlendFactor( nn::gfx::BlendFactor_One );
    targetInfo.SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    targetInfo.SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    targetInfo.SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_One );
    targetInfo.SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );
    info.SetIndependentBlendEnabled( false );
    info.SetBlendTargetStateInfoArray( &targetInfo, 1 );

    size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize( info );
    NN_SDK_ASSERT( memorySize < static_cast< size_t >( RequiredMemorySize_BlendState ) );
    m_AddBlendState.SetMemory( &m_BlendStateBuffer, memorySize );
    m_AddBlendState.Initialize( pDevice, info );
}

//---------------------------------------------------------------------------
//  テクスチャービューをディスクリプタプールに登録します。
//---------------------------------------------------------------------------
void VisualizeParticleFillHotSpot::RegisterTextureViewToDescriptorPool( nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    pRegisterTextureSlotCallback( &g_pVisualizeParticleFillHotSpot->m_TextureDesSlot, g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeTextureView, pUserData );
    NN_SDK_ASSERT( g_pVisualizeParticleFillHotSpot->m_TextureDesSlot.IsValid() );
}

//---------------------------------------------------------------------------
//  テクスチャービューをディスクリプタプールから解放します。
//---------------------------------------------------------------------------
void VisualizeParticleFillHotSpot::UnregisterTextureViewFromDescriptorPool( nn::vfx::UnregisterTextureViewSlot pUnregisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    pUnregisterTextureSlotCallback( &g_pVisualizeParticleFillHotSpot->m_TextureDesSlot, g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeTextureView, pUserData );
}

//---------------------------------------------------------------------------
//  テクスチャーサンプラをディスクリプタプールに登録します。
//---------------------------------------------------------------------------
void VisualizeParticleFillHotSpot::RegisterSamplerToDescriptorPool( nn::vfx::RegisterSamplerSlot pRegisterSamplerSlot, void* pUserData ) NN_NOEXCEPT
{
    pRegisterSamplerSlot( &g_pVisualizeParticleFillHotSpot->m_SamplerDescSlot, g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeSampler, pUserData );
    NN_SDK_ASSERT( g_pVisualizeParticleFillHotSpot->m_SamplerDescSlot.IsValid() );
}

//---------------------------------------------------------------------------
//  テクスチャーサンプラを登録解除します。
//---------------------------------------------------------------------------
void VisualizeParticleFillHotSpot::UnregisterSamplerFromDescriptorPool( nn::vfx::UnregisterSamplerSlot pUnregisterSamplerSlot, void* pUserData ) NN_NOEXCEPT
{
    pUnregisterSamplerSlot( &g_pVisualizeParticleFillHotSpot->m_SamplerDescSlot, g_pVisualizeParticleFillHotSpot->m_HotSpotAnalyzeSampler, pUserData );
}
}  // namespace detail
