﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Log.h>
#include "detail/util_GpuStatistics.h"
#include "detail/util_DebugDrawUtil.h"

#if defined(NN_GFX_CONFIG_INCLUDE_NVN)
#include <nvn/nvn_FuncPtrInline.h>
#endif

void GpuStatistics::Initialize( nn::gfx::Device* pDevice, GpuStatisticsInfo info, void* pMemory, size_t memorySize, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pMemory != NULL);
    NN_SDK_ASSERT(memorySize >= GpuStatistics::CalculateBufferSize());
    NN_SDK_ASSERT(memoryPoolSize >= GpuStatistics::CalculateQueryBufferSize(pDevice, info));
    NN_UNUSED(memorySize);
    NN_UNUSED(memoryPoolSize);
    m_Info = info;
    size_t initialOffset = memoryPoolOffset;

    int statisticsCount = nn::gfx::QueryTarget_End;

    if(m_Info.GetBufferCount() == 1)
    {
        this->m_CurrentBufferIndex = 0;
        this->m_ReferenceBufferIndex = 0;
    }

    nn::util::BytePtr memory( pMemory );
    nn::util::BytePtr initMemory = memory;
    NN_SDK_ASSERT(memory.IsAligned(GpuStatistics::GetBufferAlignment()));

    size_t size = m_GpuBuffer.CalculateWorkMemorySize(statisticsCount);
    m_GpuBuffer.ResetWorkMemory(memory.Get(), size, statisticsCount);
    memory.Advance(size);

    size_t queryBufferSize = sizeof( nn::gfx::QueryBuffer ) * statisticsCount;
    nn::gfx::Buffer::InfoType bufferInfo;
    bufferInfo.SetDefault();
    bufferInfo.SetSize( queryBufferSize );
    bufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_QueryBuffer );
    memoryPoolOffset = nn::util::align_up( memoryPoolOffset, nn::gfx::Buffer::GetBufferAlignment( pDevice, bufferInfo ) );
    for(int i = 0; i < info.GetBufferCount(); ++i)
    {
        if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
        {
            m_GpuBuffer[i].Initialize( pDevice, bufferInfo, pMemoryPool, memoryPoolOffset, bufferInfo.GetSize() );
            memoryPoolOffset += bufferInfo.GetSize();
            memoryPoolOffset = nn::util::align_up( memoryPoolOffset, nn::gfx::Buffer::GetBufferAlignment( pDevice, bufferInfo ) );
        }
        else
        {
            m_GpuBuffer[i].Initialize( pDevice, bufferInfo, NULL, 0, 0 );
        }
    }

    memory.AlignUp(NN_ALIGNOF(nn::gfx::QueryBuffer));
    size = m_CounterBuffer.CalculateWorkMemorySize(statisticsCount);
    m_CounterBuffer.ResetWorkMemory(memory.Get(), size, statisticsCount);
    memset(memory.Get(), 0, size);
    memory.Advance(size);

    NN_SDK_ASSERT(initMemory.Distance(memory.Get()) <= static_cast<ptrdiff_t>(GpuStatistics::CalculateBufferSize()), "memory shortage");
    NN_SDK_ASSERT(memoryPoolOffset - initialOffset <= GpuStatistics::CalculateQueryBufferSize(pDevice, info),  "memoryPool shortage");
    NN_UNUSED(initMemory);
    NN_UNUSED(initialOffset);
}

void GpuStatistics::Finalize( nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    m_CounterBuffer.ResetWorkMemory();
    for(int i = 0; i < m_Info.GetBufferCount(); ++i)
    {
        m_GpuBuffer[i].Finalize( pDevice );
    }
    m_GpuBuffer.ResetWorkMemory();
}

size_t GpuStatistics::GetBufferAlignment() NN_NOEXCEPT
{
    return NN_ALIGNOF(nn::gfx::Buffer);
}

size_t GpuStatistics::CalculateBufferSize() NN_NOEXCEPT
{
    int statisticsCount = nn::gfx::QueryTarget_End;
    size_t memorySize = 0;
    memorySize += nn::util::PlacementArray<nn::gfx::Buffer>::CalculateWorkMemorySize(statisticsCount);
    memorySize += NN_ALIGNOF(nn::gfx::QueryBuffer);
    memorySize +=  nn::util::PlacementArray<nn::gfx::QueryBuffer>::CalculateWorkMemorySize(statisticsCount);
    return memorySize;
}

size_t GpuStatistics::GetQueryBufferAlignment( nn::gfx::Device* pDevice, GpuStatisticsInfo info) NN_NOEXCEPT
{
    int statisticsCount = nn::gfx::QueryTarget_End;
    size_t g_QueryBufferSize = sizeof( nn::gfx::QueryBuffer ) * statisticsCount;
    nn::gfx::Buffer::InfoType bufferInfo;
    bufferInfo.SetDefault();
    bufferInfo.SetSize( g_QueryBufferSize );
    bufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_QueryBuffer );

    NN_UNUSED(info);
    return nn::gfx::Buffer::GetBufferAlignment( pDevice, bufferInfo );
}

size_t GpuStatistics::CalculateQueryBufferSize( nn::gfx::Device* pDevice, GpuStatisticsInfo info) NN_NOEXCEPT
{
    int statisticsCount = nn::gfx::QueryTarget_End;
    size_t queryBufferSize = sizeof( nn::gfx::QueryBuffer ) * statisticsCount;
    nn::gfx::Buffer::InfoType bufferInfo;
    bufferInfo.SetDefault();
    bufferInfo.SetSize( queryBufferSize );
    bufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_QueryBuffer );

    size_t bufferSize = bufferInfo.GetSize();
    size_t alignment = nn::gfx::Buffer::GetBufferAlignment( pDevice, bufferInfo );

    bufferSize += alignment;
    return bufferSize * info.GetBufferCount();
}

void GpuStatistics::BeginQuery( nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    for(int i = 1; i < nn::gfx::QueryTarget_End; ++i)
    {
        pCommandBuffer->BeginQuery(static_cast<nn::gfx::QueryTarget>(i));
    }
}

void GpuStatistics::EndQuery( nn::gfx::CommandBuffer* pCommandBuffer ) NN_NOEXCEPT
{
    nn::gfx::GpuAddress GpuAddress;
    m_GpuBuffer[m_CurrentBufferIndex].GetGpuAddress( &GpuAddress );
    ptrdiff_t bufferOffset = sizeof( nn::gfx::QueryBuffer );

    // time stamp 分スキップ
    GpuAddress.Offset( bufferOffset );

    for(int i = 1; i < nn::gfx::QueryTarget_End; ++i)
    {
        pCommandBuffer->EndQuery(GpuAddress, static_cast<nn::gfx::QueryTarget>(i));
        GpuAddress.Offset( bufferOffset );
    }
}

void GpuStatistics::Next() NN_NOEXCEPT
{
    //　フレームの結果を格納するバッファを入れ替える
    if ( m_Info.GetBufferCount() > 1 )
    {
        this->m_CurrentBufferIndex = this->m_ReferenceBufferIndex;
        if ( this->m_ReferenceBufferIndex == m_Info.GetBufferCount() - 1 )
        {
            this->m_ReferenceBufferIndex = 0;
        }
        else
        {
            this->m_ReferenceBufferIndex++;
        }
    }

    // GPU バッファに書き込まれたタイムスタンプをコピー
    int statisticsCount = nn::gfx::QueryTarget_End;
    nn::gfx::TimestampBuffer timestampBuffer;
    size_t currentBufferSize = sizeof( nn::gfx::QueryBuffer ) * statisticsCount;
    void* pMapped = m_GpuBuffer[ this->m_ReferenceBufferIndex ].Map();
    m_GpuBuffer[ this->m_ReferenceBufferIndex ].InvalidateMappedRange( 0, currentBufferSize );
    memcpy( m_CounterBuffer.data(), pMapped, currentBufferSize );
    memcpy( &timestampBuffer, pMapped, sizeof( nn::gfx::TimestampBuffer ) );
    m_GpuBuffer[ this->m_ReferenceBufferIndex ].Unmap();
}

void GpuStatistics::DrawDebugInfo( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, float x, float y ) NN_NOEXCEPT
{
    // 各エミッタ単位のGpu処理コスト
    // GPU処理表示
    const float frameWidth  = 200.f;
    const float frameHeight = 20.f;

    const nn::util::Color4u8 captionColor( 79, 129, 189, 255 );
    const nn::util::Color4u8 groundColor( 32, 32, 32, 255 );
    const nn::util::Color4u8 frameColor( 128, 128, 128, 255 );

    // 詳細処理情報
    pTextWriter->SetCursor( x, y );

    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "GpuStatistics", x, pTextWriter->GetCursorY(), frameWidth, frameHeight, 1, captionColor );
    pTextWriter->Print( "\n" );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "InVert", static_cast<int>( GetLastResult( nn::gfx::QueryTarget::QueryTarget_InputVertices ) ), x, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "InPrim", static_cast<int>( GetLastResult( nn::gfx::QueryTarget::QueryTarget_InputPrimitives ) ), x, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "VtxShaderInvoke", static_cast<int>( GetLastResult( nn::gfx::QueryTarget::QueryTarget_VertexShaderInvocations ) ), x, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "ClipInPrim", static_cast<int>( GetLastResult( nn::gfx::QueryTarget::QueryTarget_ClippingInputPrimitives ) ), x, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "ClipOutPrim", static_cast<int>( GetLastResult( nn::gfx::QueryTarget::QueryTarget_ClippingOutputPrimitives ) ), x, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "PixShaderInvoke", static_cast<int>( GetLastResult( nn::gfx::QueryTarget::QueryTarget_PixelShaderInvocations ) ), x, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "SamplesPassed", static_cast<int>( GetLastResult( nn::gfx::QueryTarget::QueryTarget_SamplesPassed ) ), x, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor );
}
