﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include "detail/util_FileSystem.h"

namespace detail {

//------------------------------------------------------------------------------
//  唯一のインスタンスを取得
//------------------------------------------------------------------------------
VfxViewerUtilFileSystem& VfxViewerUtilFileSystem::GetInstance() NN_NOEXCEPT
{
    static VfxViewerUtilFileSystem s_instance;
    return s_instance;
}

//---------------------------------------------------------------------------
//  ヒープを設定
//---------------------------------------------------------------------------
void VfxViewerUtilFileSystem::SetHeap( nn::lmem::HeapHandle pHeapHandle ) NN_NOEXCEPT
{
    m_HeapHandle = pHeapHandle;
}

//------------------------------------------------------------------------------
//  ファイルから読み込んだデータを破棄
//------------------------------------------------------------------------------
void VfxViewerUtilFileSystem::ReleaseData( uint8_t* pData ) NN_NOEXCEPT
{
    nn::lmem::FreeToExpHeap(m_HeapHandle, pData);
}

//------------------------------------------------------------------------------
//  ファイル読み込み
//------------------------------------------------------------------------------
bool VfxViewerUtilFileSystem::Read( uint8_t** ppDest, size_t* pSize, const char* pPath, int alignment ) NN_NOEXCEPT
{
    *pSize = 0;

    nn::fs::FileHandle fsHandle;
    void* pData = nullptr;

    // ファイルオープン
    nn::Result ret = nn::fs::OpenFile( &fsHandle, pPath, nn::fs::OpenMode_Read );
    if ( ret.IsFailure() )
    {
        return false;
    }

    // ファイルサイズの取得
    int64_t fsSize;
    nn::fs::GetFileSize( &fsSize, fsHandle );
    *pSize = static_cast<size_t>(fsSize);

    // 読み込みバッファの確保
    pData = nn::lmem::AllocateFromExpHeap(m_HeapHandle, *pSize, alignment);
    if ( pData == nullptr )
    {
        nn::fs::CloseFile( fsHandle );
        return false;
    }

    memset( pData, 0, *pSize );

    // ファイル読み込み
    ret = nn::fs::ReadFile( fsHandle, 0, pData, *pSize );
    nn::fs::CloseFile( fsHandle );

    if ( ret.IsFailure() )
    {
        return false;
    }

    *ppDest = static_cast<uint8_t*>(pData);

    return true;
}

//------------------------------------------------------------------------------
//  ファイル書き込み
//------------------------------------------------------------------------------
bool VfxViewerUtilFileSystem::Write( const uint8_t* pData, size_t size, const char* pPath ) NN_NOEXCEPT
{
    nn::fs::FileHandle fsHandle;

    // ファイルオープン
    nn::Result ret = nn::fs::OpenFile( &fsHandle, pPath, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend );
    if (ret.IsFailure())
    {
        ret = nn::fs::CreateFile( pPath, size );
        if (ret.IsFailure())
        {
            return false;
        }

        ret = nn::fs::OpenFile( &fsHandle, pPath, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend );
        if (ret.IsFailure())
        {
            return false;
        }
    }

    nn::fs::WriteOption option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush);

    // ファイル書き込み
    ret = nn::fs::WriteFile( fsHandle, 0, pData, size, option );
    nn::fs::CloseFile( fsHandle );

    if (ret.IsFailure())
    {
        return false;
    }

    return true;
}

}
