﻿/*--------------------------------------------------------------------------------*
Copyright (C)Nintendo All rights reserved.

These coded instructions, statements, and computer programs contain proprietary
information of Nintendo and/or its licensed developers and are protected by
national and international copyright laws. They may not be disclosed to third
parties or copied or duplicated in any form, in whole or in part, without the
prior written consent of Nintendo.

The content herein is highly confidential and should be handled accordingly.
*--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/g3d.h>
#include <nn/g3d/g3d_Configuration.h>
#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_SkeletalAnimObj.h>
#include <nn/g3d/g3d_BoneVisibilityAnimObj.h>
#include <nn/g3d/g3d_MaterialAnimObj.h>
#include <nn/g3d/g3d_ShapeAnimObj.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/util/util_BitArray.h>
#include <nn/util/util_MatrixApi.h>

#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/vfx_Random.h>


// ige によるパラメータ編集を行う
#define _USE_IGE_UI
#ifdef _USE_IGE_UI
#include <nn/ige/ige_UiService.h>
#include "detail/util_FileSystem.h"
#endif

#include <DrawParam.h>
#include <CombModelPreview.h>



namespace nw {
namespace eftdemo {

#if defined( NN_BUILD_CONFIG_OS_WIN )

// モデル、テクスチャ、シェーダ読み込みフォルダ
const char* g_AssetPath = "D:\\CombinerEditorAssets";

#ifdef _USE_IGE_UI
void* _AlignedAllocateFunc( size_t size, size_t alignment, void* ) NN_NOEXCEPT
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    return aligned_alloc( alignment, nn::util::align_up( size, alignment ) );
#else
    return _aligned_malloc( size, alignment );
#endif
}

void _FreeFunc( void* ptr, void* ) NN_NOEXCEPT
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    free( ptr );
#else
    _aligned_free( ptr );
#endif
}

nn::ige::UiService g_IgeService;
#endif

#define SHADER_SOURCE( ... ) #__VA_ARGS__

//---------------------------------------------------------------------------
// 頂点シェーダソースコード
//---------------------------------------------------------------------------
static const char VERTEX_SHADER_SOURCE[] =
"#version 440 \n"
SHADER_SOURCE(
layout( location = 0 ) in vec4  i_Position;
layout( location = 1 ) in vec3  i_Normal;
layout( location = 2 ) in vec4  i_Tangent;
layout( location = 3 ) in vec4  i_Binormal;
layout( location = 4 ) in vec2  i_UV;
layout( location = 5 ) in ivec4 i_Index;
layout( location = 6 ) in vec4  i_Weight;

layout( location = 0 ) out Raster
{
    vec3 pos_w;
    vec3 normal;
    vec3 normal_w;
    vec3 tangent_w;
    vec3 binormal_w;
    vec3 eye2pos_w;
    vec4 color;
    vec4 uv;
    vec4 projection;
} OUT;

layout( std140 ) uniform Skeleton
{
    vec4 mtxPalette[ 3 * 128 ];
};

layout( std140 ) uniform Shape
{
    vec4 shapeMtx[ 3 ];
    int vtxSkinCount;
};

layout( std140 ) uniform View
{
    vec4 projMtx[ 4 ];
    vec4 cameraMtx[ 4 ];
    vec4 cameraPos;
    vec4 cameraVec;
};

out gl_PerVertex
{
    vec4 gl_Position;
};

mat4 sysTexCoordTransformMatrix =
    mat4( 0.5, 0.0, 0.0, 0.0,
        0.0, 0.5, 0.0, 0.0,
        0.0, 0.0, 0.5, 0.0,
        0.5, 0.5, 0.5, 1.0 );

void main()
{
    vec4 pos_w = vec4( 0, 0, 0, 1 );
    vec3 nrm_w = vec3( 0, 0, 0 );

    // 分岐によるパフォーマンス度外視の実装。
    if ( vtxSkinCount == 0 )
    {
        pos_w.x = dot( shapeMtx[ 0 ], i_Position );
        pos_w.y = dot( shapeMtx[ 1 ], i_Position );
        pos_w.z = dot( shapeMtx[ 2 ], i_Position );
        nrm_w.x = dot( shapeMtx[ 0 ].xyz, i_Normal );
        nrm_w.y = dot( shapeMtx[ 1 ].xyz, i_Normal );
        nrm_w.z = dot( shapeMtx[ 2 ].xyz, i_Normal );

        OUT.tangent_w.x = dot( shapeMtx[ 0 ].xyz, i_Tangent.xyz );
        OUT.tangent_w.y = dot( shapeMtx[ 1 ].xyz, i_Tangent.xyz );
        OUT.tangent_w.z = dot( shapeMtx[ 2 ].xyz, i_Tangent.xyz );
        OUT.binormal_w.x = dot( shapeMtx[ 0 ].xyz, i_Binormal.xyz );
        OUT.binormal_w.y = dot( shapeMtx[ 1 ].xyz, i_Binormal.xyz );
        OUT.binormal_w.z = dot( shapeMtx[ 2 ].xyz, i_Binormal.xyz );
    }
    else if ( vtxSkinCount == 1 )
    {
        int mtxIndex = i_Index.x * 3;
        pos_w.x = dot( mtxPalette[ mtxIndex + 0 ], i_Position );
        pos_w.y = dot( mtxPalette[ mtxIndex + 1 ], i_Position );
        pos_w.z = dot( mtxPalette[ mtxIndex + 2 ], i_Position );
        nrm_w.x = dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Normal );
        nrm_w.y = dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Normal );
        nrm_w.z = dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Normal );
        OUT.tangent_w.x = dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Tangent.xyz );
        OUT.tangent_w.y = dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Tangent.xyz );
        OUT.tangent_w.z = dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Tangent.xyz );
        OUT.binormal_w.x = dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Binormal.xyz );
        OUT.binormal_w.y = dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Binormal.xyz );
        OUT.binormal_w.z = dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Binormal.xyz );
    }
    else
    {
        for ( int i = 0; i < vtxSkinCount; ++i )
        {
            int mtxIndex = i_Index[ i ] * 3;
            float weight = i_Weight[ i ];
            pos_w.x += weight * dot( mtxPalette[ mtxIndex + 0 ], i_Position );
            pos_w.y += weight * dot( mtxPalette[ mtxIndex + 1 ], i_Position );
            pos_w.z += weight * dot( mtxPalette[ mtxIndex + 2 ], i_Position );
            nrm_w.x += weight * dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Normal );
            nrm_w.y += weight * dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Normal );
            nrm_w.z += weight * dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Normal );
            OUT.tangent_w.x = dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Tangent.xyz );
            OUT.tangent_w.y = dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Tangent.xyz );
            OUT.tangent_w.z = dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Tangent.xyz );
            OUT.binormal_w.x = dot( mtxPalette[ mtxIndex + 0 ].xyz, i_Binormal.xyz );
            OUT.binormal_w.y = dot( mtxPalette[ mtxIndex + 1 ].xyz, i_Binormal.xyz );
            OUT.binormal_w.z = dot( mtxPalette[ mtxIndex + 2 ].xyz, i_Binormal.xyz );
        }
    }

    nrm_w = normalize( nrm_w );

    OUT.pos_w = pos_w.xyz;

    vec4 pos_v;
    pos_v.x = dot( cameraMtx[ 0 ], pos_w );
    pos_v.y = dot( cameraMtx[ 1 ], pos_w );
    pos_v.z = dot( cameraMtx[ 2 ], pos_w );
    pos_v.w = 1;

    gl_Position.x = dot( projMtx[ 0 ], pos_v );
    gl_Position.y = dot( projMtx[ 1 ], pos_v );
    gl_Position.z = dot( projMtx[ 2 ], pos_v );
    gl_Position.w = dot( projMtx[ 3 ], pos_v );

    OUT.projection = sysTexCoordTransformMatrix * gl_Position;

    OUT.normal_w = nrm_w;
    OUT.normal.x = dot( cameraMtx[ 0 ].xyz, nrm_w );
    OUT.normal.y = dot( cameraMtx[ 1 ].xyz, nrm_w );
    OUT.normal.z = dot( cameraMtx[ 2 ].xyz, nrm_w );

    OUT.uv.xy = i_UV;
    OUT.uv.zw = i_UV;

    vec3 up = vec3( 0, 1, 0 );
    float rate = dot( nrm_w, up ) * 0.5 + 0.5;
    OUT.color.rgb = rate * vec3( 0.7, 0.7, 0.9 ) + ( 1 - rate ) * vec3( 0.3, 0.2, 0.2 );
    OUT.color.a = pow( 1 - abs( OUT.normal.z ), 2 ) * 0.2;


    OUT.eye2pos_w = pos_w.xyz - cameraPos.xyz;
}
);

//---------------------------------------------------------------------------
// フラグメントシェーダソースコード
//---------------------------------------------------------------------------
static const char FRAGMENT_SHADER_SOURCE[] =
    "#version 440 \n"
    SHADER_SOURCE(
        layout( location = 0 ) in Raster
{
    vec3 pos_w;
    vec3 normal;
    vec3 normal_w;
    vec3 tangent_w;
    vec3 binormal_w;
    vec3 eye2pos_w;
    vec4 color;
    vec4 uv;
    vec4 projection;
} IN;

layout( location = 0 ) out vec4 o_Color;

uniform sampler2D nwAlbedoTexture0;

layout( std140 ) uniform Material
{
    vec4 offset;
};

void main()
{
    vec4 albedo = texture( nwAlbedoTexture0, IN.uv.xy );
    o_Color = albedo;
}
);

//---------------------------------------------------------------------------
// フラグメントシェーダ　ソースコードヘッダ
//---------------------------------------------------------------------------
static const char FRAGMENT_SHADER_HEADER[] =
"#version 440\n"
"mat4 IdentityMatrix = \n"
"mat4( 0.5, 0.0, 0.0, 0.0, \n"
"      0.0, 0.5, 0.0, 0.0, \n"
"      0.0, 0.0, 0.5, 0.0, \n"
"      0.5, 0.5, 0.5, 1.0 );\n"
"layout( location = 0 ) in Raster\n"
"{\n"
"    vec3 pos_w;\n"
"    vec3 normal;\n"
"    vec3 normal_w;\n"
"    vec3 tangent_w;"
"    vec3 binormal_w;"
"    vec3 eye2pos_w;\n"
"    vec4 color;\n"
"    vec4 uv;\n"
"    vec4 projection;\n"
"} IN;\n"
"layout( std140 ) uniform Material\n"
"{\n"
"    vec4 nwParam;\n"
"    vec4 nwParam1;\n"
"    vec4 nwParam2;\n"
"    vec4 nwParam3;\n"
"};\n"
"layout( std140 ) uniform View\n"
"{\n"
"    vec4 projMtx[ 4 ];\n"
"    mat4 cameraMtx;\n"
"    vec4 cameraPos;\n"
"    vec4 cameraVec;\n"
"};\n"
"#define _USE_NN_VFX\n"
"layout( location = 0 ) out vec4 nwOutputAlbedoColor;\n"
"#define OUTPUT_COLOR nwOutputAlbedoColor\n"
"uniform sampler2D nwAlbedoTexture0;\n"
"uniform sampler2D nwAlbedoTexture1;\n"
"uniform sampler2D nwAlbedoTexture2;\n"
"uniform sampler2D nwAlbedoTexture3;\n"
"uniform sampler2D nwColorBufferTexture;\n"
"uniform sampler2D nwDepthBufferTexture;\n"
"#define sysTextureSampler0 nwAlbedoTexture0\n"
"#define sysTextureSampler1 nwAlbedoTexture1\n"
"#define sysTextureSampler2 nwAlbedoTexture2\n"
"#define sysFrameBufferTexture nwColorBufferTexture\n"
"#define sysDepthBufferTexture nwDepthBufferTexture\n"
"#define nwTextureCoord0 IN.uv.xy\n"
"#define nwTextureCoord1 IN.uv.xy\n"
"#define nwTextureCoord2 IN.uv.xy\n"
"#define nwTextureCoord3 IN.uv.xy\n"
"#define sysTexCoord01Vary IN.uv\n"
"#define sysTexCoord2Vary IN.uv.xy\n"
"#define sysLifeRatio nwParam.x\n"
"#define sysFragCoordVary IN.projection\n"
"#define sysWorldPositionVary IN.pos_w\n"
"#define sysWorldNormalVary IN.normal_w\n"
"#define sysWorldTangentVary IN.tangent_w\n"
"#define sysWorldBinormalVary IN.binormal_w\n"
"#define v_outWorldNormal IN.normal_w\n"
"#define sysViewMatrix cameraMtx\n"
"#define v_outPrimitiveColor IN.color\n"
"#define viewMat cameraMtx\n"
"#define sysShaderAnim nwParam.x\n"
"#define sysShaderAnimY nwParam.y\n"
"#define sysShaderAnimZ nwParam.z\n"
"#define emitterMatrixRT IdentityMatrix\n"
"#define sysEmitterRTMatrix IdentityMatrix\n"
"#define eyePos cameraPos\n"
"#define eyeVec cameraVec\n"
"#define sysEyeVecVary eye2pos_w\n"
"#define v_outColor0 nwParam1\n"
"#define v_outColor1 nwParam2\n"
"#define sysParticleRandom nwParam3\n"
"#define sysFragRandomVary nwParam3\n"
SHADER_SOURCE(
float       nwTimeRatio;                          // nw_def コンパイル通し用
uniform sampler3D sysCurlNoiseTextureArray;       // カールノイズテクスチャアレイサンプラ―( コンパイル通し用 )
float CalcAlphaProcess( float inAlpha ) { return inAlpha; }
void FinalAdjustmentFragmentColor() {}
float GetDepthValue( vec4 position ) { return 0.0; }
float GetDepthValueFromTexture( vec4 projection ) { return 0.0; }
);




//---------------------------------------------------------------------------
//  テクスチャ
//---------------------------------------------------------------------------
Texture::Texture() NN_NOEXCEPT
{
    m_TextureFace = 0;
    m_pDevice = nullptr;
    m_pResTextureFile[ 0 ] = nullptr;
    m_pResTextureFile[ 1 ] = nullptr;
    m_TextureBinaryPtr[ 0 ] = nullptr;
    m_TextureBinaryPtr[ 1 ] = nullptr;
}

Texture::~Texture() NN_NOEXCEPT
{
    if ( m_pResTextureFile[ 0 ] )
    {
        m_pResTextureFile[ 0 ]->Finalize( m_pDevice );
    }
    if ( m_pResTextureFile[ 1 ] )
    {
        m_pResTextureFile[ 1 ]->Finalize( m_pDevice );
    }

    if ( m_TextureBinaryPtr[ 0 ]  )
    {
        detail::VfxViewerUtilFileSystem::GetInstance().ReleaseData( m_TextureBinaryPtr[ 0 ] );
    }
    if ( m_TextureBinaryPtr[ 1 ] )
    {
        detail::VfxViewerUtilFileSystem::GetInstance().ReleaseData( m_TextureBinaryPtr[ 1 ] );
    }

}

bool Texture::Initialize( nn::gfx::Device* pDevice, const char* pFtxbFilePath ) NN_NOEXCEPT
{
    if ( !pFtxbFilePath )
    {
        return false;
    }
    if ( pDevice )
    {
        m_pDevice = pDevice;
    }
    if ( !m_pDevice )
    {
        return false;
    }

    Finalize( pDevice );
    m_TextureFace = 1 - m_TextureFace;

    size_t textureDataSize;

    char command[ 1024 ];

    // テクスチャコンバート
    sprintf_s( command, "%%NINTENDO_SDK_ROOT%%\\Tools\\Graphics\\3dTools\\3dTextureConverter.exe %s -o %s\\combinerPreviewTexture.bntx --tile-mode NX", pFtxbFilePath, g_AssetPath );
    system( command );

    // テクスチャロード
    sprintf_s( command, "%s\\combinerPreviewTexture.bntx", g_AssetPath );
    bool resRead = detail::VfxViewerUtilFileSystem::GetInstance().Read( &m_TextureBinaryPtr[ m_TextureFace ], &textureDataSize, command, 1024 * 64 );
    if ( !resRead )
    {
        return false;
    }


    m_pResTextureFile[ m_TextureFace ] = nn::gfx::ResTextureFile::ResCast( m_TextureBinaryPtr[ m_TextureFace ] );
    NN_SDK_ASSERT_NOT_NULL( m_pResTextureFile );

    m_pResTextureFile[ m_TextureFace ]->Initialize( pDevice );

    int textureCount = m_pResTextureFile[ m_TextureFace ]->GetTextureDic()->GetCount();
    for ( int i = 0; i < textureCount; i++ )
    {
        nn::gfx::ResTexture* resTexture = m_pResTextureFile[ m_TextureFace ]->GetResTexture( i );
        resTexture->Initialize( pDevice );
    }

    return true;
}

void Texture::UpdateDescriptorSlot( vfxdemo::TextureDescriptorIndexAllocator* pTextureDescriptorIndexAllocator ) NN_NOEXCEPT
{
    if ( !m_pResTextureFile[ m_TextureFace ] )
    {
        return;
    }

    int textureCount = m_pResTextureFile[ m_TextureFace ]->GetTextureDic()->GetCount();
    for ( int i = 0; i < textureCount; i++ )
    {
        nn::gfx::ResTexture* resTexture = m_pResTextureFile[ m_TextureFace ]->GetResTexture( i );
        const nn::gfx::TextureView* textureView = resTexture->GetTextureView();
        pTextureDescriptorIndexAllocator->AllocateTransientTextureDescriptorSlot( &m_DescriptorSlot[ m_TextureFace ], *textureView, NULL );
    }
}

void Texture::Finalize( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    if ( m_pResTextureFile[ 1 - m_TextureFace ] )
    {
        m_pResTextureFile[ 1 - m_TextureFace ]->Finalize( pDevice );
        m_pResTextureFile[ 1 - m_TextureFace ] = nullptr;

        detail::VfxViewerUtilFileSystem::GetInstance().ReleaseData( m_TextureBinaryPtr[ 1 - m_TextureFace ] );
        m_TextureBinaryPtr[ 1 - m_TextureFace ] = nullptr;
    }
}



//---------------------------------------------------------------------------
//  シェーダ
//---------------------------------------------------------------------------
Shader::Shader() NN_NOEXCEPT
{
    m_pDevice = nullptr;
    m_IsInitialized[ 0 ] = false;
    m_IsInitialized[ 1 ] = false;
    m_ShaderFace = 0;
}

Shader::~Shader() NN_NOEXCEPT
{
    if ( m_IsInitialized[ 0 ] )
    {
        m_Shader[ 0 ].Finalize( m_pDevice );
    }
    if ( m_IsInitialized[ 1 ] )
    {
        m_Shader[ 1 ].Finalize( m_pDevice );
    }
}

bool Shader::Initialize( nn::gfx::Device* pDevice, nn::gfx::ShaderStage shaderStage, const char* shaderCode ) NN_NOEXCEPT
{
    if ( pDevice )
    {
        m_pDevice = pDevice;
    }
    if ( !m_pDevice )
    {
        return false;
    }

    Finalize( m_pDevice );
    m_ShaderFace = 1 - m_ShaderFace;

    // シェーダ生成
    nn::gfx::Shader::InfoType info;
    info.SetDefault();
    info.SetSeparationEnabled( true );

    nn::gfx::ShaderCode psCode;
    psCode.codeSize = static_cast< uint32_t >( strlen( shaderCode ) );
    psCode.pCode = shaderCode;

    info.SetShaderCodePtr( shaderStage, &psCode );
    info.SetSourceFormat( nn::gfx::ShaderSourceFormat_Glsl );
    info.SetCodeType( nn::gfx::ShaderCodeType_Source );
    nn::gfx::ShaderInitializeResult result = m_Shader[ m_ShaderFace ].Initialize( m_pDevice, info );
    NN_SDK_ASSERT( result == nn::gfx::ShaderInitializeResult_Success, "nn::gfx::Shader::Initialize() failed (%d)\n", result );
    NN_UNUSED( result );

    // TODO:シェーダコンパイルエラー時の処理

    m_IsInitialized[ m_ShaderFace ] = true;

    return true;
}

void Shader::Finalize( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    if ( m_IsInitialized[ 1 - m_ShaderFace ] )
    {
        m_Shader[ 1 - m_ShaderFace ].Finalize( pDevice );
        m_IsInitialized[ 1 - m_ShaderFace ] = false;
    }
}






//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
CombinerPreviewModel::CombinerPreviewModel() NN_NOEXCEPT
{
    m_pDevice = nullptr;
    m_pHeap = nullptr;
    m_Reload = false;
    m_pBinaryLoadBuffer = nullptr;

    m_CurrentFace = 0;

    m_pMemoryPoolPtr = nullptr;
    m_MemoryPoolOffset = 0;

    m_pG3dModelResFile = nullptr;
    m_pVertexStateArray = nullptr;
    m_pVertexStateBuffer = nullptr;

    m_pModelObj = nullptr;

    m_ShaderParameter[ 0 ] = 0.0f;
    m_ShaderParameter[ 1 ] = 0.0f;
    m_ShaderParameter[ 2 ] = 0.0f;
    m_ShaderParameter[ 3 ] = 0.0f;

    m_VectorParameter0.x = 1.0f;
    m_VectorParameter0.y = 1.0f;
    m_VectorParameter0.z = 1.0f;
    m_VectorParameter0.w = 1.0f;

    m_VectorParameter1.x = 1.0f;
    m_VectorParameter1.y = 1.0f;
    m_VectorParameter1.z = 1.0f;
    m_VectorParameter1.w = 1.0f;

    m_VectorParameter2.x = nn::vfx::detail::Random::GetGlobalRandom()->GetFloat();
    m_VectorParameter2.y = nn::vfx::detail::Random::GetGlobalRandom()->GetFloat();
    m_VectorParameter2.z = nn::vfx::detail::Random::GetGlobalRandom()->GetFloat();
    m_VectorParameter2.w = nn::vfx::detail::Random::GetGlobalRandom()->GetFloat();


    memset( m_TextureFilePath[ 0 ], 0, _MAX_PATH );
    memset( m_TextureFilePath[ 1 ], 0, _MAX_PATH );
    memset( m_TextureFilePath[ 2 ], 0, _MAX_PATH );
    memset( m_TextureFilePath[ 3 ], 0, _MAX_PATH );
    memset( m_ReloadFshFilePath, 0, _MAX_PATH );
}

//---------------------------------------------------------------------------
//  デストラクタ
//---------------------------------------------------------------------------
CombinerPreviewModel::~CombinerPreviewModel()
{
    if ( m_pG3dModelResFile )
    {
        m_pG3dModelResFile->Cleanup( m_pDevice );
    }
}


void _CreateFileLoad( nn::ige::Panel* parent, const char* header, char* path, nn::ige::ClickCallback callback, void* userData )
{
    nn::ige::PanelOption panelOption;
    panelOption.tooltip = "Parameter_Label";
    //panelOption.background = "LightGray";
    panelOption.orientation = nn::ige::Orientation_Horizontal;
    panelOption.panelType = nn::ige::PanelType_Wrap;
    panelOption.height = 40;

    nn::ige::LabelOption labelOption;
    labelOption.tooltip = "file load";
    labelOption.header = header;

    nn::ige::ButtonOption buttonOption;
    buttonOption.header = "Load";
    buttonOption.tooltip = header;
    buttonOption.pClickCallback = callback;
    buttonOption.pUserData = userData;

    nn::ige::TextBoxOption textBoxOption;
    textBoxOption.header = "TextBox Header";

    auto* panel = parent->CreatePanel( panelOption );

    panelOption.width = 60;
    auto* panel_0 = panel->CreatePanel( panelOption );
    panelOption.width = 150;
    auto* panel_1 = panel->CreatePanel( panelOption );
    panelOption.width = 60;
    auto* panel_2 = panel->CreatePanel( panelOption );

    panel_0->CreateLabel( labelOption );
    panel_1->CreateTextBox( path, sizeof( path ), textBoxOption );
    panel_2->CreateButton( buttonOption );
}

void _CreateSlider( nn::ige::Panel* parent, const char* header, float* value )
{
    nn::ige::PanelOption panelOption;
    panelOption.tooltip = "Parameter_Label";
    //panelOption.background = "LightGray";
    panelOption.orientation = nn::ige::Orientation_Horizontal;
    panelOption.panelType = nn::ige::PanelType_Wrap;
    panelOption.height = 26;

    nn::ige::LabelOption labelOption;
    labelOption.tooltip = "min:0 max:1";
    nn::ige::SliderOption floatSliderOption;
    floatSliderOption.tooltip = "min:0 max:1";

    auto* panel = parent->CreatePanel( panelOption );

    panelOption.width = 100;
    //panelOption.background = "White";
    auto* panel_0 = panel->CreatePanel( panelOption );

    panelOption.width = 220;
    //panelOption.background = "White";
    auto* panel_1 = panel->CreatePanel( panelOption );

    labelOption.header = header;
    panel_0->CreateLabel( labelOption );
    panel_1->CreateSlider( value, 0, 1, floatSliderOption );
}

//---------------------------------------------------------------------------
// Ige システム初期化
//---------------------------------------------------------------------------
void CombinerPreviewModel::InitializeIgeService() NN_NOEXCEPT
{
#ifdef _USE_IGE_UI
    // UiService の初期化
    nn::ige::UiService::InitializeArg arg;
    arg.portName = "IgeSimple";
    arg.allocateFunc = _AlignedAllocateFunc;
    arg.freeFunc = _FreeFunc;

    g_IgeService.Initialize( arg );
    g_IgeService.Start();

    nn::ige::PanelOption panelOption;

    // モデル
    nn::ige::GroupBoxOption groupBoxOption;
    groupBoxOption.header = "Texture";
    groupBoxOption.tooltip = "Texture";
    auto* textureGroupBox = g_IgeService.CreateGroupBox( groupBoxOption );

    panelOption.tooltip = "TexLoad";
    panelOption.orientation = nn::ige::Orientation_Vertical;
    panelOption.panelType = nn::ige::PanelType_Stack;
    auto* panel_tex_load = textureGroupBox->CreatePanel( panelOption );

    static int textureIndex0 = 0;
    static int textureIndex1 = 1;
    static int textureIndex2 = 2;
    static int textureIndex3 = 3;

    _CreateFileLoad( panel_tex_load, "Texture 0", m_TextureFilePath[ 0 ], TextureReload, &textureIndex0 );
    _CreateFileLoad( panel_tex_load, "Texture 1", m_TextureFilePath[ 1 ], TextureReload, &textureIndex1 );
    _CreateFileLoad( panel_tex_load, "Texture 2", m_TextureFilePath[ 2 ], TextureReload, &textureIndex2 );
    _CreateFileLoad( panel_tex_load, "Texture 3", m_TextureFilePath[ 3 ], TextureReload, &textureIndex3 );

    groupBoxOption.header = "Particle Parameter";
    groupBoxOption.tooltip = "Particle Parameter";
    auto* shaderParam0GroupBox = g_IgeService.CreateGroupBox( groupBoxOption );

    panelOption.tooltip = "Parameter";
    panelOption.orientation = nn::ige::Orientation_Vertical;
    panelOption.panelType = nn::ige::PanelType_Stack;
    auto* panel_0 = shaderParam0GroupBox->CreatePanel( panelOption );
    _CreateSlider( panel_0, "Life Ratio", &m_ShaderParameter[ 0 ] );
    _CreateSlider( panel_0, "Animation0_X", &m_ShaderParameter[ 1 ] );
    _CreateSlider( panel_0, "Animation0_Y", &m_ShaderParameter[ 2 ] );
    _CreateSlider( panel_0, "Animation0_Z", &m_ShaderParameter[ 3 ] );


    groupBoxOption.header = "Particle Color0";
    groupBoxOption.tooltip = "Particle Color0";
    auto* shaderParam1GroupBox = g_IgeService.CreateGroupBox( groupBoxOption );

    panelOption.tooltip = "Particle Color0";
    auto* panel_1 = shaderParam1GroupBox->CreatePanel( panelOption );
    _CreateSlider( panel_1, "R :", &m_VectorParameter0.x );
    _CreateSlider( panel_1, "G :", &m_VectorParameter0.y);
    _CreateSlider( panel_1, "B :", &m_VectorParameter0.z);
    _CreateSlider( panel_1, "A :", &m_VectorParameter0.w);


    groupBoxOption.header = "Particle Color1";
    groupBoxOption.tooltip = "Particle Color1";
    auto* shaderParam2GroupBox = g_IgeService.CreateGroupBox( groupBoxOption );

    panelOption.tooltip = "Particle Color1";
    auto* panel_2 = shaderParam2GroupBox->CreatePanel( panelOption );
    _CreateSlider( panel_2, "R :", &m_VectorParameter1.x );
    _CreateSlider( panel_2, "G :", &m_VectorParameter1.y );
    _CreateSlider( panel_2, "B :", &m_VectorParameter1.z );
    _CreateSlider( panel_2, "A :", &m_VectorParameter1.w );


    groupBoxOption.header = "Particle Random";
    groupBoxOption.tooltip = "Particle Random";
    auto* shaderParam3GroupBox = g_IgeService.CreateGroupBox( groupBoxOption );

    panelOption.tooltip = "Particle Random";
    auto* panel_3 = shaderParam3GroupBox->CreatePanel( panelOption );
    _CreateSlider( panel_3, "X :", &m_VectorParameter2.x );
    _CreateSlider( panel_3, "Y :", &m_VectorParameter2.y );
    _CreateSlider( panel_3, "Z :", &m_VectorParameter2.z );
    _CreateSlider( panel_3, "W :", &m_VectorParameter2.w );
#endif
}

//---------------------------------------------------------------------------
// Ige システム破棄
//---------------------------------------------------------------------------
void CombinerPreviewModel::FinalizeIgeService() NN_NOEXCEPT
{
#ifdef _USE_IGE_UI


#endif
}



//---------------------------------------------------------------------------
//! @brief  初期化処理を行います。
//---------------------------------------------------------------------------
bool CombinerPreviewModel::Initialize( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    m_pDevice = pDevice;
    m_pHeap = pHeap;

    // デフォルトシェーダを初期化
    m_VertexShader.Initialize( m_pDevice, nn::gfx::ShaderStage_Vertex, VERTEX_SHADER_SOURCE );
    m_FragmentShader.Initialize( m_pDevice, nn::gfx::ShaderStage_Pixel, FRAGMENT_SHADER_SOURCE );

    // メモリプールの初期化
    static const uint64_t cpuUncachedMemoryPoolSize = 4 * 1024 * 1024; // ダブり
    nn::gfx::MemoryPool::InfoType memoryPoolInfo;
    memoryPoolInfo.SetDefault();
    memoryPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );
    const size_t memoryPoolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( m_pDevice, memoryPoolInfo );
    m_pMemoryPoolPtr = m_pHeap->Alloc( cpuUncachedMemoryPoolSize, memoryPoolGranularity );
    memoryPoolInfo.SetPoolMemory( m_pMemoryPoolPtr, cpuUncachedMemoryPoolSize );
    m_MemoryPool.Initialize( m_pDevice, memoryPoolInfo );

    // 共通テクスチャサンプラ初期化
    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    m_Sampler.Initialize( m_pDevice, samplerInfo );

    // バイナリロード用バッファ
    const size_t binaryLoadBufferSize = 1024 * 1024;
    m_pBinaryLoadBuffer = pHeap->Alloc( binaryLoadBufferSize );
    m_HeapHandle = nn::lmem::CreateExpHeap( m_pBinaryLoadBuffer, binaryLoadBufferSize, nn::lmem::CreationOption_NoOption );
    detail::VfxViewerUtilFileSystem::GetInstance().SetHeap( m_HeapHandle );

    return true;
}

//---------------------------------------------------------------------------
//! @brief  メモリプールから領域を確保します
//---------------------------------------------------------------------------
ptrdiff_t CombinerPreviewModel::AllocateFromMemoryPool( size_t size, size_t alignment /*= DEFAULT_ALIGNMENT*/ ) NN_NOEXCEPT
{
    static const uint64_t cpuUncachedMemoryPoolSize = 4 * 1024 * 1024;

    ptrdiff_t offset = nn::util::align_up( m_MemoryPoolOffset, alignment );
    NN_ASSERT( cpuUncachedMemoryPoolSize - offset >= size, "AllocateMemoryPool failed.\n" );
    m_MemoryPoolOffset = offset + size;
    return offset;
}

//---------------------------------------------------------------------------
//! @brief  モデルファイルをロードします。
//---------------------------------------------------------------------------
bool CombinerPreviewModel::LoadG3dModelBinary( const char* pFmdbFilePath ) NN_NOEXCEPT
{
    if ( !m_VertexShader.GetGfxShader() ) return false;
    if ( !m_FragmentShader.GetGfxShader() ) return false;

    if ( !pFmdbFilePath )
    {
        const char* fmdbPath = "D:\\CombinerEditorAssets\\combinerPreviewModel0.fmdb";
        pFmdbFilePath = fmdbPath;
    }

    // bfresコンバート
    char command[ 1024 ];
    sprintf_s( command, "%%NINTENDO_SDK_ROOT%%\\Tools\\Graphics\\3dTools\\3dBinaryConverter.exe -o %s\\combinerPreviewModel.bfres %s\\combinerPreviewModel0.fmdb -i", g_AssetPath, g_AssetPath );
    system( command );

    // bfres ロード
    uint8_t*    pBfresBinary = nullptr;
    size_t      pBfresBinarySize;
    sprintf_s( command, "%s\\combinerPreviewModel.bfres", g_AssetPath );
    bool resRead = detail::VfxViewerUtilFileSystem::GetInstance().Read( &pBfresBinary, &pBfresBinarySize, command, 1024 * 64 );
    if ( !resRead )
    {
        return false;
    }

    NN_SDK_ASSERT( nn::g3d::ResFile::IsValid( pBfresBinary ) );
    m_pG3dModelResFile = nn::g3d::ResFile::ResCast( pBfresBinary );
    NN_SDK_ASSERT( m_pG3dModelResFile->GetModelCount() == 1 );
    m_pG3dModelResFile->Setup( m_pDevice );

    nn::g3d::ResModel* pResModel = m_pG3dModelResFile->GetModel( 0 );

    // 頂点ステートを初期化
    {
        const char* attribName[] = { "_p0", "_n0", "_t0", "_b0", "_u0", "_i0", "_w0" };
        const char* attribId[] = { "i_Position", "i_Normal", "i_Tangent", "i_Binormal", "i_UV", "i_Index", "i_Weight" };
        const int attribCount = sizeof( attribName ) / sizeof( char* );
        const int bufferCount = 16;

        // VertexStateは ResVertex と1対1で対応します。
        // 複数の ResShape が ResVertex を共有している場合があります。
        int numVertex = pResModel->GetVertexCount();
        size_t vertexStateArraySize = sizeof( nn::gfx::VertexState ) * numVertex;
        m_pVertexStateArray = static_cast< nn::gfx::VertexState* >( m_pHeap->Alloc( vertexStateArraySize, 4 ) );
        NN_ASSERT_NOT_NULL( m_pVertexStateArray );

        for ( int idxVertex = 0; idxVertex < numVertex; ++idxVertex )
        {
            nn::gfx::VertexState* pVertexState = new( &m_pVertexStateArray[ idxVertex ] ) nn::gfx::VertexState;
            nn::gfx::VertexState::InfoType info;
            info.SetDefault();
            nn::g3d::ResVertex* pResVertex = pResModel->GetVertex( idxVertex );

            // 頂点属性を設定
            int idxElement = 0;
            nn::gfx::VertexAttributeStateInfo vertexAttribs[ attribCount ];
            for ( int idxAttrib = 0; idxAttrib < attribCount; ++idxAttrib )
            {
                const char* name = attribName[ idxAttrib ];
                vertexAttribs[ idxAttrib ].SetDefault();
                nn::g3d::ResVertexAttr* pResVertexAttr = pResVertex->FindVertexAttr( name );
                if ( pResVertexAttr )
                {
                    vertexAttribs[ idxElement ].SetBufferIndex( pResVertexAttr->GetBufferIndex() );
                    vertexAttribs[ idxElement ].SetFormat( pResVertexAttr->GetFormat() );
                    vertexAttribs[ idxElement ].SetOffset( pResVertexAttr->GetOffset() );
                    int slotAttr = m_VertexShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, attribId[ idxAttrib ] );
                    vertexAttribs[ idxElement ].SetShaderSlot( slotAttr );
                    if ( slotAttr != -1 )
                    {
                        idxElement++;
                    }
                }
            }

            // 頂点バッファ情報を設定
            int numVertexBuffer = pResVertex->GetVertexBufferCount();
            nn::gfx::VertexBufferStateInfo bufferStateInfo[ bufferCount ];
            NN_ASSERT( numVertexBuffer < bufferCount );
            for ( int idxBuffer = 0; idxBuffer < numVertexBuffer; ++idxBuffer )
            {
                bufferStateInfo[ idxBuffer ].SetDefault();
                bufferStateInfo[ idxBuffer ].SetStride( pResVertex->GetVertexBufferStride( idxBuffer ) );
            }

            // 頂点属性と頂点バッファ情報を頂点ステートに設定
            info.SetVertexAttributeStateInfoArray( vertexAttribs, idxElement );
            info.SetVertexBufferStateInfoArray( bufferStateInfo, numVertexBuffer );

            size_t memorySize = nn::gfx::VertexState::GetRequiredMemorySize( info );
            m_pVertexStateBuffer = m_pHeap->Alloc( memorySize, nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
            NN_ASSERT_NOT_NULL( m_pVertexStateBuffer );
            pVertexState->SetMemory( m_pVertexStateBuffer, memorySize );
            pVertexState->Initialize( m_pDevice, info, NULL );
            pResVertex->SetUserPtr( pVertexState );
        }
    }

    // モデルを初期化
    {
        m_pModelObj = new( m_pHeap->Alloc( sizeof( nn::g3d::ModelObj ), nn::g3d::ModelObj::Alignment_Buffer ) ) nn::g3d::ModelObj();
        NN_SDK_ASSERT_NOT_NULL( m_pModelObj );
        nn::g3d::ModelObj::Builder builder( m_pG3dModelResFile->GetModel( 0 ) );

        builder.MaterialBufferingCount( 2 );
        builder.SkeletonBufferingCount( 2 );
        builder.ShapeBufferingCount( 2 );
        builder.ViewCount( 2 );
        builder.CalculateMemorySize();
        size_t bufferSize = builder.GetWorkMemorySize();
        void* pBuffer = m_pHeap->Alloc( bufferSize );
        NN_SDK_ASSERT_NOT_NULL( pBuffer );
        bool success = builder.Build( m_pModelObj, pBuffer, bufferSize );
        NN_SDK_ASSERT( success );

        // Uniform Block のバッファは所定のアライメントが必要です。
        size_t blockBufferSize = m_pModelObj->CalculateBlockBufferSize( m_pDevice );
        ptrdiff_t blockBufferOffset = -1;
        blockBufferOffset = this->AllocateFromMemoryPool( blockBufferSize, m_pModelObj->GetBlockBufferAlignment( m_pDevice ) );
        success = m_pModelObj->SetupBlockBuffer( m_pDevice, &m_MemoryPool, blockBufferOffset, blockBufferSize );
        NN_ASSERT( success );
    }

    // 原点で計算処理
    nn::util::Matrix4x3fType drawMatrix;
    nn::util::MatrixIdentity( &drawMatrix );
    m_pModelObj->CalculateWorld( drawMatrix );

    m_Reload = true;

    return true;
}


//---------------------------------------------------------------------------
//! @brief  テクスチャファイルをロードします。
//---------------------------------------------------------------------------
bool CombinerPreviewModel::LoadTextureBinary( const char* pFtxbFilePath ) NN_NOEXCEPT
{
    NN_UNUSED( pFtxbFilePath );

    m_Reload = true;

    m_Texture[ 0 ].Initialize( m_pDevice, "D:\\CombinerEditorAssets\\combinerPreviewTexture0.ftxb" );
    m_Texture[ 1 ].Initialize( m_pDevice, "D:\\CombinerEditorAssets\\combinerPreviewTexture1.ftxb" );
    m_Texture[ 2 ].Initialize( m_pDevice, "D:\\CombinerEditorAssets\\combinerPreviewTexture2.ftxb" );
    m_Texture[ 3 ].Initialize( m_pDevice, "D:\\CombinerEditorAssets\\combinerPreviewTexture3.ftxb" );

    return true;
}


//---------------------------------------------------------------------------
// テクスチャリロードファイルパスを設定
//---------------------------------------------------------------------------
void CombinerPreviewModel::ReloadTextureFilePath( int index, const char* ftxbFilePath ) NN_NOEXCEPT
{
    if ( index > 4 ) return;

    m_Texture[ index ].Initialize( m_pDevice, ftxbFilePath );

    m_Reload = true;
}

//---------------------------------------------------------------------------
// テクスチャリロードファイルパスを設定
//---------------------------------------------------------------------------
void CombinerPreviewModel::UpdateShaderParameter(const char* name, float value) NN_NOEXCEPT
{
    if (!strcmp(name, "m_ShaderParameter[ 0 ]"))
    {
        m_ShaderParameter[0] = value;
    }
    else if (!strcmp(name, "m_ShaderParameter[ 1 ]"))
    {
        m_ShaderParameter[1] = value;
    }
    else if (!strcmp(name, "m_ShaderParameter[ 2 ]"))
    {
        m_ShaderParameter[2] = value;
    }
    else if (!strcmp(name, "m_ShaderParameter[ 3 ]"))
    {
        m_ShaderParameter[3] = value;
    }
    else if (!strcmp(name, "m_VectorParameter0.x"))
    {
        m_VectorParameter0.x = value;
    }
    else if (!strcmp(name, "m_VectorParameter0.y"))
    {
        m_VectorParameter0.y = value;
    }
    else if (!strcmp(name, "m_VectorParameter0.z"))
    {
        m_VectorParameter0.z = value;
    }
    else if (!strcmp(name, "m_VectorParameter0.w"))
    {
        m_VectorParameter0.w = value;
    }
    else if (!strcmp(name, "m_VectorParameter1.x"))
    {
        m_VectorParameter1.x = value;
    }
    else if (!strcmp(name, "m_VectorParameter1.y"))
    {
        m_VectorParameter1.y = value;
    }
    else if (!strcmp(name, "m_VectorParameter1.z"))
    {
        m_VectorParameter1.z = value;
    }
    else if (!strcmp(name, "m_VectorParameter1.w"))
    {
        m_VectorParameter1.w = value;
    }
    else if (!strcmp(name, "m_VectorParameter2.x"))
    {
        m_VectorParameter2.x = value;
    }
    else if (!strcmp(name, "m_VectorParameter2.y"))
    {
        m_VectorParameter2.y = value;
    }
    else if (!strcmp(name, "m_VectorParameter2.z"))
    {
        m_VectorParameter2.z = value;
    }
    else if (!strcmp(name, "m_VectorParameter2.w"))
    {
        m_VectorParameter2.w = value;
    }
}
//---------------------------------------------------------------------------
// テクスチャリロードファイルパスを設定
//---------------------------------------------------------------------------
void CombinerPreviewModel::ReloadTextureFilePath(const char* name, const char* ftxbFilePath) NN_NOEXCEPT
{
    if (!strcmp(name, "nwAlbedoTexture0"))
    {
        ReloadTextureFilePath(0, ftxbFilePath);
    }
    else if (!strcmp(name, "nwAlbedoTexture1"))
    {
        ReloadTextureFilePath(1, ftxbFilePath);
    }
    else if (!strcmp(name, "nwAlbedoTexture2"))
    {
        ReloadTextureFilePath(2, ftxbFilePath);
    }
    else if (!strcmp(name, "nwAlbedoTexture3"))
    {
        ReloadTextureFilePath(3, ftxbFilePath);
    }
    else
    {
        return;
    }
}
//---------------------------------------------------------------------------
// テクスチャリロードファイルパスを設定
//---------------------------------------------------------------------------
void CombinerPreviewModel::ReceiveUpdateParameter(const char* receiveData) NN_NOEXCEPT
{

    int nameSize;
    memcpy(&nameSize, receiveData, sizeof(uint32_t));
    receiveData += sizeof(uint32_t);
    char name[_MAX_PATH];
    memset(name, 0, nameSize);
    memcpy(name, receiveData, nameSize);
    receiveData += nameSize;

    int valueTypeSize;
    memcpy(&valueTypeSize, receiveData, sizeof(uint32_t));
    receiveData += sizeof(uint32_t);
    char valueType[_MAX_PATH];
    memset(valueType, 0, valueTypeSize);
    memcpy(valueType, receiveData, valueTypeSize);
    receiveData += valueTypeSize;


    if (!strcmp(valueType, "file"))
    {
        int filePathSize;
        memcpy(&filePathSize, receiveData, sizeof(uint32_t));
        receiveData += sizeof(uint32_t);

        char filePath[_MAX_PATH];
        memset(filePath, 0, filePathSize);
        memcpy(filePath, receiveData, filePathSize);

        ReloadTextureFilePath(name, filePath);
    }
    else if (!strcmp(valueType, "float"))
    {
        float value;
        memcpy(&value, receiveData, sizeof(float_t));
        receiveData += sizeof(float_t);

        UpdateShaderParameter(name, value);
    }
    else if (!strcmp(valueType, "vec2"))
    {
        float valueX;
        memcpy(&valueX, receiveData, sizeof(float_t));
        receiveData += sizeof(float_t);

        float valueY;
        memcpy(&valueY, receiveData, sizeof(float_t));
        receiveData += sizeof(float_t);
        name[nameSize] = '.';
        name[nameSize + 1] = 'x';
        UpdateShaderParameter(name, valueX);
        name[nameSize + 1] = 'y';
        UpdateShaderParameter(name, valueY);
    }
    else if (!strcmp(valueType, "vec3"))
    {
        float valueX;
        memcpy(&valueX, receiveData, sizeof(float_t));
        receiveData += sizeof(float_t);

        float valueY;
        memcpy(&valueY, receiveData, sizeof(float_t));
        receiveData += sizeof(float_t);

        float valueZ;
        memcpy(&valueZ, receiveData, sizeof(float_t));
        receiveData += sizeof(float_t);

        name[nameSize] = '.';
        name[nameSize + 1] = 'x';
        UpdateShaderParameter(name, valueX);
        name[nameSize + 1] = 'y';
        UpdateShaderParameter(name, valueY);
        name[nameSize + 1] = 'z';
        UpdateShaderParameter(name, valueZ);
    }
    else if (!strcmp(valueType, "vec4"))
    {
        float valueX;
        memcpy(&valueX, receiveData, sizeof(float_t));
        receiveData += sizeof(float_t);

        float valueY;
        memcpy(&valueY, receiveData, sizeof(float_t));
        receiveData += sizeof(float_t);

        float valueZ;
        memcpy(&valueZ, receiveData, sizeof(float_t));
        receiveData += sizeof(float_t);

        float valueW;
        memcpy(&valueW, receiveData, sizeof(float_t));
        receiveData += sizeof(float_t);

        name[nameSize] = '.';
        name[nameSize + 1] = 'x';
        UpdateShaderParameter(name, valueX);
        name[nameSize + 1] = 'y';
        UpdateShaderParameter(name, valueY);
        name[nameSize + 1] = 'z';
        UpdateShaderParameter(name, valueZ);
        name[nameSize + 1] = 'w';
        UpdateShaderParameter(name, valueW);
    }
}

//---------------------------------------------------------------------------
//! @brief  ブロックの更新処理を行います。
//---------------------------------------------------------------------------
void CombinerPreviewModel::CalcBlock( const nn::util::Matrix4x3fType* viewMtx ) NN_NOEXCEPT
{
#ifdef _USE_IGE_UI
    // データ送受信
    g_IgeService.Poll();

    // データ更新
    g_IgeService.Update();
#endif
    // フラグメントシェーダリロード
    UpdateFragmentShader();

    if ( !m_pModelObj )
    {
        return;
    }

    // バッファの更新
    m_CurrentFace = 1 - m_CurrentFace;

    m_pModelObj->CalculateSkeleton( m_CurrentFace );
    m_pModelObj->CalculateMaterial( m_CurrentFace );
    m_pModelObj->CalculateShape( m_CurrentFace );
    m_pModelObj->CalculateView( m_CurrentFace, *viewMtx, m_CurrentFace );
}


//---------------------------------------------------------------------------
//! @brief  描画処理を行います。
//---------------------------------------------------------------------------
void CombinerPreviewModel::Draw( nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::GpuBuffer* pGpuBuffer,
                                 const nn::util::Matrix4x4fType* projectionMatrix, const nn::util::Matrix4x3fType* viewMatrix, const nn::util::Vector3fType* camPos,
                                 nn::gfx::DescriptorSlot colorBufferDescSlot, nn::gfx::DescriptorSlot depthBufferDescSlot ) NN_NOEXCEPT
{
    if ( !m_pModelObj )
    {
        return;
    }

    nn::util::Matrix4x3fType identityViewMatrix;
    {
        nn::util::Vector3fType pos;
        nn::util::Vector3fType at;
        nn::util::Vector3fType up;
        nn::util::VectorSet( &pos, 0.0f, 0.0f, 2.0f );
        nn::util::VectorSet( &at, 0.0f, 0.0f, 0.0f );
        nn::util::VectorSet( &up, 0.0f, 1.0f, 0.0f );
        nn::util::MatrixLookAtRightHanded( &identityViewMatrix, pos, at, up );
    }

    // シェーダーを設定
    pCommandBuffer->SetShader( m_VertexShader.GetGfxShader(), nn::gfx::ShaderStageBit_Vertex );
    pCommandBuffer->SetShader( m_FragmentShader.GetGfxShader(), nn::gfx::ShaderStageBit_Pixel );

    int slotSkeleton = m_VertexShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Skeleton" );
    int slotShape = m_VertexShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Shape" );
    int slotVertexView = m_VertexShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "View" );
    int slotFragmentView = m_FragmentShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "View" );
    int slotMaterial = m_FragmentShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Material" );

    // ユニフォームブロック設定
    const nn::g3d::SkeletonObj* pSkeletonObj = m_pModelObj->GetSkeleton();
    nn::gfx::GpuAddress gpuAddress;
    if ( pSkeletonObj->GetMtxBlockSize() > 0 )
    {
        pSkeletonObj->GetMtxBlock( m_CurrentFace )->GetGpuAddress( &gpuAddress );
        pCommandBuffer->SetConstantBuffer( slotSkeleton, nn::gfx::ShaderStage_Vertex, gpuAddress, pSkeletonObj->GetMtxBlockSize() );
    }

    // ビュー
    ViewBlock* pViewBlockBuffer = reinterpret_cast<ViewBlock*>( pGpuBuffer->Allocate( sizeof( ViewBlock ) ) );
    if ( pViewBlockBuffer )
    {
        if ( m_Reload )
        {
            pViewBlockBuffer->Set( identityViewMatrix, *projectionMatrix, *camPos );
        }
        else
        {
            pViewBlockBuffer->Set( *viewMatrix, *projectionMatrix, *camPos );
        }
        pGpuBuffer->GetGpuAddress( &gpuAddress, pViewBlockBuffer );
        if ( slotVertexView != -1 )
        {
            pCommandBuffer->SetConstantBuffer( slotVertexView, nn::gfx::ShaderStage_Vertex, gpuAddress, sizeof( ViewBlock ) );
        }
        if ( slotFragmentView != -1 )
        {
            pCommandBuffer->SetConstantBuffer( slotFragmentView, nn::gfx::ShaderStage_Pixel, gpuAddress, sizeof( ViewBlock ) );
        }
    }

    int shapeCount = m_pModelObj->GetShapeCount();
    for ( int idxShape = 0; idxShape < shapeCount; ++idxShape )
    {
        const nn::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape( idxShape );

        if ( !m_pModelObj->IsShapeVisible( idxShape ) ) // ビジビリティの制御を行います。
        {
            continue;
        }

        // シェイプ
        pShapeObj->GetShapeBlock( 0, 0 )->GetGpuAddress( &gpuAddress );
        pCommandBuffer->SetConstantBuffer( slotShape, nn::gfx::ShaderStage_Vertex, gpuAddress, sizeof( nn::g3d::ShapeBlock ) );

        // マテリアル
        if ( slotMaterial != -1 )
        {
            nn::util::Float4* pMaterialBuffer = reinterpret_cast< nn::util::Float4* >( pGpuBuffer->Allocate( sizeof( nn::util::Float4 ) * 4 ) );
            if ( pMaterialBuffer )
            {
                pGpuBuffer->GetGpuAddress( &gpuAddress, pMaterialBuffer );

                pMaterialBuffer->x = m_ShaderParameter[ 0 ];
                pMaterialBuffer->y = m_ShaderParameter[ 1 ];
                pMaterialBuffer->z = m_ShaderParameter[ 2 ];
                pMaterialBuffer->w = m_ShaderParameter[ 3 ];
                pMaterialBuffer++;

                pMaterialBuffer->x = m_VectorParameter0.x;
                pMaterialBuffer->y = m_VectorParameter0.y;
                pMaterialBuffer->z = m_VectorParameter0.z;
                pMaterialBuffer->w = m_VectorParameter0.w;
                pMaterialBuffer++;

                pMaterialBuffer->x = m_VectorParameter1.x;
                pMaterialBuffer->y = m_VectorParameter1.y;
                pMaterialBuffer->z = m_VectorParameter1.z;
                pMaterialBuffer->w = m_VectorParameter1.w;
                pMaterialBuffer++;

                pMaterialBuffer->x = m_VectorParameter2.x;
                pMaterialBuffer->y = m_VectorParameter2.y;
                pMaterialBuffer->z = m_VectorParameter2.z;
                pMaterialBuffer->w = m_VectorParameter2.w;

                pCommandBuffer->SetConstantBuffer( slotMaterial, nn::gfx::ShaderStage_Pixel, gpuAddress, sizeof( ViewBlock ) );
            }
        }

        // 頂点属性を設定
        int idxVertex = pShapeObj->GetVertexIndex();
        nn::g3d::ResModel* pResModel = const_cast< nn::g3d::ResModel* >( m_pModelObj->GetResource() );
        nn::g3d::ResVertex* pResVertex = pResModel->GetVertex( idxVertex );
        nn::gfx::VertexState* pVertexState = static_cast< nn::gfx::VertexState* >( pResVertex->GetUserPtr() );
        pCommandBuffer->SetVertexState( pVertexState );

        // TODO:良い場所に
        const char* samplerId[] = { "nwAlbedoTexture0", "nwAlbedoTexture1", "nwAlbedoTexture2", "nwAlbedoTexture3" };
        const int samplerCount = sizeof( samplerId ) / sizeof( char* );

        // テクスチャ設定
        for ( int idxTex = 0; idxTex < samplerCount; ++idxTex )
        {
            const char* name = samplerId[ idxTex ];
            int slotSampler = m_FragmentShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, name );
            if ( slotSampler != -1 && m_Texture[ idxTex ].IsInitialized() )
            {
                pCommandBuffer->SetTextureAndSampler( slotSampler, nn::gfx::ShaderStage_Pixel,
                    m_Texture[ idxTex ].GetDescriptorSlot(), m_SamplerDescriptorSlot );
            }
        }

        // カラーバッファテクスチャ設定
        {
            int slotSampler = m_FragmentShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "nwColorBufferTexture" );
            if ( slotSampler != -1 && colorBufferDescSlot.IsValid() )
            {
                pCommandBuffer->SetTextureAndSampler( slotSampler, nn::gfx::ShaderStage_Pixel, colorBufferDescSlot, m_SamplerDescriptorSlot );
            }
        }

        // デプスバッファテクスチャ設定
        {
            int slotSampler = m_FragmentShader.GetGfxShader()->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "nwDepthBufferTexture" );
            if ( slotSampler != -1 && depthBufferDescSlot.IsValid() )
            {
                pCommandBuffer->SetTextureAndSampler( slotSampler, nn::gfx::ShaderStage_Pixel, depthBufferDescSlot, m_SamplerDescriptorSlot );
            }
        }

        // バッファを設定
        int numVertexBuffer = pResVertex->GetVertexBufferCount();
        for ( int idxBuffer = 0; idxBuffer < numVertexBuffer; ++idxBuffer )
        {
            pResVertex->GetVertexBuffer( idxBuffer )->GetGpuAddress( &gpuAddress );
            pCommandBuffer->SetVertexBuffer( idxBuffer, gpuAddress,
                pResVertex->GetVertexBufferStride( idxBuffer ),
                pResVertex->GetVertexBufferInfo( idxBuffer )->GetSize() );
        }

        // 描画コール
        pShapeObj->GetResMesh()->Draw( pCommandBuffer, 1 );
    }

    m_Reload = false;
}


//---------------------------------------------------------------------------
//! @brief  ディスクリプタスロットの更新を行います。
//---------------------------------------------------------------------------
void CombinerPreviewModel::UpdateDescriptorSlot( vfxdemo::TextureDescriptorIndexAllocator* pTextureDescriptorIndexAllocator,
    vfxdemo::SamplerDescriptorIndexAllocator* pSamplerDescriptorIndexAllocator ) NN_NOEXCEPT
{
    m_Texture[ 0 ].UpdateDescriptorSlot( pTextureDescriptorIndexAllocator );
    m_Texture[ 1 ].UpdateDescriptorSlot( pTextureDescriptorIndexAllocator );
    m_Texture[ 2 ].UpdateDescriptorSlot( pTextureDescriptorIndexAllocator );
    m_Texture[ 3 ].UpdateDescriptorSlot( pTextureDescriptorIndexAllocator );

    pSamplerDescriptorIndexAllocator->AllocateStaticSamplerDescriptorSlot( &m_SamplerDescriptorSlot, m_Sampler, nullptr );
}

//---------------------------------------------------------------------------
//! @brief  ディスクリプタスロットの更新を行います。
//---------------------------------------------------------------------------
bool CombinerPreviewModel::IsReload() const NN_NOEXCEPT
{
    return m_Reload;
}


//---------------------------------------------------------------------------
// シェーダリロード
//---------------------------------------------------------------------------
void CombinerPreviewModel::ShaderReload( void* ) NN_NOEXCEPT
{
    return GetInstance()->SetReloadShaderFilePath( "D:\\CombinerEditorAssets\\test.fsh" );
}

//---------------------------------------------------------------------------
// テクスチャリロード
//---------------------------------------------------------------------------
void CombinerPreviewModel::TextureReload( void* pTextureIndex ) NN_NOEXCEPT
{
    int* index = reinterpret_cast< int* >( pTextureIndex );
    return GetInstance()->ReloadTextureFilePath( *index, GetInstance()->m_TextureFilePath[ *index ] );
}


//---------------------------------------------------------------------------
// フラグメントシェーダの更新
//---------------------------------------------------------------------------
void CombinerPreviewModel::UpdateFragmentShader() NN_NOEXCEPT
{
    if ( strlen( m_ReloadFshFilePath ) == 0 )
    {
        return;
    }

    uint8_t* pCodeData;
    size_t codeSize;
    bool resRead = detail::VfxViewerUtilFileSystem::GetInstance().Read( &pCodeData, &codeSize, m_ReloadFshFilePath );
    if ( !resRead )
    {
        return;
    }

    size_t headerSize = strlen( FRAGMENT_SHADER_HEADER );
    void* codeBuffer = g_pCombinerPreviewModel->m_pHeap->Alloc( codeSize + 4098 );
    if ( !codeBuffer )
    {
        return;
    }

    strncpy( reinterpret_cast< char* >( codeBuffer ), FRAGMENT_SHADER_HEADER, headerSize );
    strncpy( reinterpret_cast< char* >( codeBuffer ) + headerSize, reinterpret_cast< const char* >( pCodeData ), codeSize );
    reinterpret_cast< char* >( codeBuffer )[ headerSize + codeSize ] = 0;
    g_pCombinerPreviewModel->m_FragmentShader.Initialize( nullptr, nn::gfx::ShaderStage_Pixel, reinterpret_cast< const char* >( codeBuffer ) );
    g_pCombinerPreviewModel->m_pHeap->Free( codeBuffer );
    detail::VfxViewerUtilFileSystem::GetInstance().ReleaseData( pCodeData );

    memset( m_ReloadFshFilePath, 0, _MAX_PATH );
}

//---------------------------------------------------------------------------
// シェーダリロードファイルパスを設定
//---------------------------------------------------------------------------
void CombinerPreviewModel::SetReloadShaderFilePath( const char* fshFilePath ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( static_cast< int >( strlen( fshFilePath ) ), 1, _MAX_PATH );
    strncpy_s( m_ReloadFshFilePath, fshFilePath, _MAX_PATH );
    m_Reload = true;
}

//---------------------------------------------------------------------------
//! @brief  コンバイナプレビューモデルの生成。
//---------------------------------------------------------------------------
void CombinerPreviewModel::InitializeCombinerPreviewModel( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    void* buffer = pHeap->Alloc( sizeof( CombinerPreviewModel ) );
    g_pCombinerPreviewModel = new ( buffer ) CombinerPreviewModel();

#ifdef _USE_IGE_UI
    g_pCombinerPreviewModel->InitializeIgeService();
#endif
    g_pCombinerPreviewModel->Initialize( pDevice, pHeap );
    g_pCombinerPreviewModel->LoadG3dModelBinary();
    g_pCombinerPreviewModel->LoadTextureBinary();
}

//---------------------------------------------------------------------------
//! @brief  コンバイナプレビューモデルの破棄。
//---------------------------------------------------------------------------
void CombinerPreviewModel::FinalizeCombinerPreviewModel( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    NN_UNUSED( pDevice );
    g_pCombinerPreviewModel->~CombinerPreviewModel();

    pHeap->Free( g_pCombinerPreviewModel );
}

#endif

CombinerPreviewModel* CombinerPreviewModel::g_pCombinerPreviewModel = nullptr;



} // namespace nw::eftdemo
} // namespace nw
