﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Utilities
{
    using System;
    using System.Collections.Generic;
    using System.Runtime.InteropServices;
    using System.Windows.Forms;

    /// <summary>
    /// ＵＩコントロールイベント抑制ブロッククラス。
    /// </summary>
    public sealed class ControlEventSuppressBlock : IDisposable
    {
        // インスタンスリスト
        #region Static Fields

        /// <summary>
        /// The instances.
        /// </summary>
        private static readonly List<ControlEventSuppressBlock> Instances = new List<ControlEventSuppressBlock>();

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="ControlEventSuppressBlock"/> class.
        /// コンストラクタ。
        /// </summary>
        public ControlEventSuppressBlock()
        {
            // リストに追加
            Instances.Add(this);
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// 抑制中かどうか。
        /// </summary>
        public static bool Enabled
        {
            get
            {
                // インスタンスがあれば抑制中
                return Instances.Count > 0;
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// 開放処理。
        /// </summary>
        public void Dispose()
        {
            // リストから削除
            Instances.Remove(this);
        }

        #endregion
    }

    /// <summary>
    /// The update block.
    /// </summary>
    public sealed class UpdateBlock : IDisposable
    {
        #region Fields

        /// <summary>
        /// The control.
        /// </summary>
        private readonly dynamic control;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="UpdateBlock"/> class.
        /// </summary>
        /// <param name="control">
        /// The control.
        /// </param>
        public UpdateBlock(dynamic control)
        {
            this.control = control;
            this.control.BeginUpdate();
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The dispose.
        /// </summary>
        public void Dispose()
        {
            this.control.EndUpdate();
        }

        #endregion
    }

    /// <summary>
    /// ウィンドウ内の描画を無効にするブロックです。
    /// </summary>
    public sealed class LockWindowUpdateBlock : IDisposable
    {
        /// <summary>
        /// The is locked.
        /// </summary>
        private bool isLocked = false;

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="LockWindowUpdateBlock"/> class.
        /// </summary>
        /// <param name="control">
        /// The control.
        /// </param>
        public LockWindowUpdateBlock(Control control)
        {
            if (control.IsDisposed)
            {
                return;
            }

            LockWindowUpdate(control.Handle);
            this.isLocked = true;
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The dispose.
        /// </summary>
        public void Dispose()
        {
            if (this.isLocked)
            {
                LockWindowUpdate(IntPtr.Zero);
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The lock window update.
        /// </summary>
        /// <param name="hwhd">
        /// The h whd.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        [DllImport("user32.dll")]
        private static extern bool LockWindowUpdate(IntPtr hwhd);

        #endregion
    }

    /// <summary>
    /// The layout suspend block.
    /// </summary>
    public sealed class LayoutSuspendBlock : IDisposable
    {
        #region Fields

        /// <summary>
        /// The control.
        /// </summary>
        private readonly Control control;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="LayoutSuspendBlock"/> class.
        /// </summary>
        /// <param name="control">
        /// The control.
        /// </param>
        public LayoutSuspendBlock(Control control)
        {
            this.control = control;
            this.control.SuspendLayout();
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The dispose.
        /// </summary>
        public void Dispose()
        {
            this.control.ResumeLayout(false);
            this.control.PerformLayout();
        }

        #endregion
    }

    /// <summary>
    /// The wait cursor block.
    /// </summary>
    public sealed class WaitCursorBlock : IDisposable
    {
        #region Fields

        /// <summary>
        /// The control.
        /// </summary>
        private readonly Control control;

        /// <summary>
        /// The old.
        /// </summary>
        private readonly Cursor old;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="WaitCursorBlock"/> class.
        /// </summary>
        /// <param name="control">
        /// The control.
        /// </param>
        public WaitCursorBlock(Control control)
        {
            this.control = control;
            this.old = this.control.Cursor;
            this.control.Cursor = Cursors.WaitCursor;
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The dispose.
        /// </summary>
        public void Dispose()
        {
            this.control.Cursor = this.old;
        }

        #endregion
    }
}
