﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.IO;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base;
using EffectMaker.UIControls.EffectBrowser.Controls.FileTreeView;
using EffectMaker.UIControls.EffectBrowser.Data.Favorites;
using EffectMaker.UIControls.EffectBrowser.Data;
using EffectMaker.UIControls.EffectBrowser.Utilities;

namespace EffectMaker.UIControls.EffectBrowser
{
    /// <summary>
    /// エフェクトブラウザのメインパネルです。
    /// </summary>
    public partial class EffectBrowserPanel
    {
        /// <summary>
        /// The update address bar.
        /// </summary>
        private void UpdateAddressBar()
        {
            using (new LockWindowUpdateBlock(this))
            using (new LayoutSuspendBlock(this))
            {
                switch (this.SelectedPageKind)
                {
                    case PageKind.Computer:
                        this.adbComputer.Visible = true;
                        this.adbFavorites.Visible = false;
                        this.adbSearch.Visible = false;

                        this.flvComputer.Visible = true;
                        this.flvFavorites.Visible = false;
                        this.flvSearch.Visible = false;
                        break;

                    case PageKind.Favorites:
                        this.adbComputer.Visible = false;
                        this.adbFavorites.Visible = true;
                        this.adbSearch.Visible = false;

                        this.flvComputer.Visible = false;
                        this.flvFavorites.Visible = true;
                        this.flvSearch.Visible = false;
                        break;

                    case PageKind.Search:
                        this.adbComputer.Visible = false;
                        this.adbFavorites.Visible = false;
                        this.adbSearch.Visible = true;

                        this.flvComputer.Visible = false;
                        this.flvFavorites.Visible = false;
                        this.flvSearch.Visible = true;
                        break;
                }
            }
        }

        /// <summary>
        /// The update context menu.
        /// </summary>
        private void UpdateContextMenu()
        {
            {
                // Computer
                this.tsmComputerShowInExplorer.Enabled = PathUtility.DirectoryExists(this.Directory);
                this.tsmComputerAdd.Enabled = this.tsmComputerShowInExplorer.Enabled;
            }

            {
                // Favorites
                this.tsmFavoritesShowInExplorer.Enabled = PathUtility.DirectoryExists(this.Directory);
                {
                    // ftvFavoritesAdd
                    if (this.ftvFavorites.SelectedNode != null)
                    {
                        this.tsmFavoritesAdd.Enabled = (this.ftvFavorites.SelectedNode.NodeKind
                                                        == EBFileTreeNode.NodeKindType.Directory)
                                                       && System.IO.Directory.Exists(
                                                           this.ftvFavorites.SelectedNode.Filepath);
                    }
                    else
                    {
                        this.tsmFavoritesAdd.Enabled = false;
                    }
                }

                {
                    // tsmFavoritesDelete
                    if (this.ftvFavorites.SelectedNode != null)
                    {
                        this.tsmFavoritesDelete.Enabled = (this.ftvFavorites.SelectedNode.NodeKind
                                                           == EBFileTreeNode.NodeKindType.FavFolder)
                                                          || (this.ftvFavorites.SelectedNode.NodeKind
                                                              == EBFileTreeNode.NodeKindType.Bookmark);
                    }
                    else
                    {
                        this.tsmFavoritesDelete.Enabled = false;
                    }
                }

                {
                    // tsmFavoritesRename
                    this.tsmFavoritesRename.Enabled = this.tsmFavoritesDelete.Enabled;
                }
            }

            // Files
            this.UpdateFilesContextMenu();
        }

        /// <summary>
        /// The update files context menu.
        /// </summary>
        private void UpdateFilesContextMenu()
        {
            var selectedFile = this.SelectedFileListView.SelectedFile;
            if (selectedFile == null)
            {
                this.tsmFilesAddFav.Enabled = System.IO.Directory.Exists(this.Directory);
                this.tsmFilesShowInExplorer.Enabled = this.tsmFilesAddFav.Enabled;
            }
            else
            {
                this.tsmFilesAddFav.Enabled = System.IO.Directory.Exists(selectedFile.FileFullPath);
                this.tsmFilesShowInExplorer.Enabled = File.Exists(selectedFile.FileFullPath)
                                                      || System.IO.Directory.Exists(selectedFile.FileFullPath);
            }

            {
                var canEdit = this.SelectedFileListView.CanEdit;
                var anyEsets = this.SelectedFileListView.SelectedFiles.Any(x => x.FileKind == FileKindType.EsetFile);
                var anyMetadataEditable = this.SelectedFileListView.SelectedFiles.Any(x => x.IsEditable);

                this.tsmFilesEdit.Enabled = canEdit && this.SelectedFileListView.SelectedFiles.Any(x => x.IsOpenable);
                this.tsmFilesExport.Enabled = canEdit && anyEsets;
                this.tsmFilesEditComment.Enabled = canEdit && anyMetadataEditable;
                this.tsmFilesEditLabelColor.Enabled = canEdit && anyMetadataEditable;
                if (this.SelectedFileListView.SelectedFiles.Any(x => x.IsBlockingEditByOpened))
                {
                    this.tsmFilesEditLabelColor.ToolTipText = Properties.Resources.ToolTipBlockingEdit;
                    this.tsmFilesEditComment.ToolTipText = Properties.Resources.ToolTipBlockingEdit;
                }
                else
                {
                    this.tsmFilesEditLabelColor.ToolTipText = Properties.Resources.ToolTipSetLabelColor;
                    this.tsmFilesEditComment.ToolTipText = Properties.Resources.ToolTipSetComment;
                }
            }
        }

        /// <summary>
        /// The update filter menu.
        /// </summary>
        private void UpdateFilterMenu()
        {
            var flv = this.FileListViews.First();

            this.tsmFileFilter.DropDownItems.OfType<ToolStripMenuItem>()
                .Where(x => x.Tag is EBFileListView.ColumnKinds)
                .ForEach(
                    x =>
                        {
                            var kind = (EBFileListView.ColumnKinds)x.Tag;
                            x.Checked = flv.GetColumnVisible(kind);
                        });
        }

        /// <summary>
        /// The update status bar.
        /// </summary>
        private void UpdateStatusBar()
        {
            using (new LayoutSuspendBlock(this))
            {
                switch (this.SelectedPageKind)
                {
                    case PageKind.Computer:
                        this.tpbComputer.Visible = true;
                        this.tpbFavorites.Visible = false;
                        this.tpbSearch.Visible = false;
                        break;

                    case PageKind.Favorites:
                        this.tpbComputer.Visible = false;
                        this.tpbFavorites.Visible = true;
                        this.tpbSearch.Visible = false;
                        break;

                    case PageKind.Search:
                        this.tpbSearch.Visible = false;
                        this.tpbFavorites.Visible = false;
                        this.tpbComputer.Visible = true;
                        break;
                }
            }

            this.tssAllCount.Text = string.Format(
                Properties.Resources.StatusBar_AllCount,
                this.SelectedFileListView.UnfilteredFilesCount);
            this.tssSelectedCount.Text = string.Format(
                Properties.Resources.StatusBar_SelectedCount,
                this.SelectedFileListView.SelectedFilesCount);
            this.tssFilename.Text = (this.SelectedFileListView == null)
                                        ? string.Empty
                                        : (this.SelectedFileListView.SelectedFile == null)
                                              ? string.Empty
                                              : (this.SelectedFileListView.SelectedFilesCount > 1)
                                                    ? Properties.Resources.StatusBat_MultipleSelected
                                                    : this.SelectedFileListView.SelectedFile.FileFullPath;
            this.UpdateStatusBarState();
        }

        /// <summary>
        /// The update status bar state.
        /// </summary>
        private void UpdateStatusBarState()
        {
            if (this.SelectedFileListView == null)
            {
                return;
            }

            switch (this.SelectedFileListView.UpdateState)
            {
                case EBFileListView.UpdateStateKind.EditDataLoad:
                    this.tssState.Text = this.SelectedFileListView.UpdateStateName
                                         + string.Format(
                                             "({0}/{1})",
                                             this.SelectedFileListView.LoadedEditDataCount,
                                             this.SelectedFileListView.OpenedAllFiles.Count(
                                                 x => x.FileKind == FileKindType.EsetFile));
                    break;

                case EBFileListView.UpdateStateKind.AssetsLoad:
                    this.tssState.Text = this.SelectedFileListView.UpdateStateName
                                         + string.Format(
                                             "({0}/{1})",
                                             this.SelectedFileListView.LoadedAssetDataCount,
                                             this.SelectedFileListView.OpenedAllFiles.Count(
                                                 x => x.FileKind == FileKindType.TextureFile));
                    break;

                default:
                    this.tssState.Text = this.SelectedFileListView.UpdateStateName;
                    break;
            }
        }

        /// <summary>
        /// The update tool bar.
        /// </summary>
        private void UpdateToolBar()
        {
            var current = (PageKind)this.tbcTrees.SelectedTab.Tag;
            {
                // カレントページから
                this.tsmAddFavorite.Visible = (current == PageKind.Computer) || (current == PageKind.Favorites);
                this.tsmAddFolder.Visible = current == PageKind.Favorites;
                this.tsmDeleteFavorite.Visible = current == PageKind.Favorites;
                this.tssSeparator.Visible = (current == PageKind.Computer) || (current == PageKind.Favorites);
                this.tsmAddSampleFav.Visible = (current == PageKind.Computer) || (current == PageKind.Favorites);
                this.tsmAddSampleFav.ToolTipText = PathUtility.DirectoryExists(FavoritesManager.SampleFolderPath) ?
                    Properties.Resources.Favorites_AddSampleFolderToolTip :
                    Properties.Resources.Favorites_SampleFolderNotFound;
            }

            {
                // 現在選択されているものから
                this.tsmAddFavorite.Enabled = PathUtility.DirectoryExists(this.Directory);
                this.tsmDeleteFavorite.Enabled = (this.ftvFavorites.SelectedNode != null)
                                                 && ((this.ftvFavorites.SelectedNode.NodeKind
                                                      == EBFileTreeNode.NodeKindType.Bookmark)
                                                     || (this.ftvFavorites.SelectedNode.NodeKind
                                                         == EBFileTreeNode.NodeKindType.FavFolder));
                this.tsmShowInExplorer.Enabled = (this.SelectedAddressBar != null)
                                                 && (this.SelectedFileListView != null)
                                                 && (this.SelectedFileListView != null)
                                                 && (this.SelectedFileListView.SelectedFile != null)
                                                 && (PathUtility.FileExists(
                                                     this.SelectedFileListView.SelectedFile.FileFullPath)
                                                     || PathUtility.DirectoryExists(
                                                         this.SelectedFileListView.SelectedFile.FileFullPath));
            }

            {
                if (this.SelectedFileListView == null)
                {
                    this.tsbEditEset.Enabled = false;
                    this.tsbExportEset.Enabled = false;
                    this.tsbSetLabelColorEset.Enabled = false;
                    this.tsbSetCommentEset.Enabled = false;
                }
                else
                {
                    var canEdit = this.SelectedFileListView.CanEdit;
                    var anyEsets = this.SelectedFileListView.SelectedFiles.Any(x => x.FileKind == FileKindType.EsetFile);
                    var anyOpenable = this.SelectedFileListView.SelectedFiles.Any(x => x.IsOpenable);
                    var anyEditable = this.SelectedFileListView.SelectedFiles.Any(x => x.IsEditable);

                    this.tsbEditEset.Enabled = canEdit && anyOpenable;
                    this.tsbExportEset.Enabled = canEdit && anyEsets;
                    this.tsbSetLabelColorEset.Enabled = canEdit && anyEditable;
                    this.tsbSetCommentEset.Enabled = canEdit && anyEditable;
                    if (this.SelectedFileListView.SelectedFiles.Any(x => x.IsBlockingEditByOpened))
                    {
                        this.tsbSetLabelColorEset.ToolTipText = Properties.Resources.ToolTipBlockingEdit;
                        this.tsbSetCommentEset.ToolTipText = Properties.Resources.ToolTipBlockingEdit;
                    }
                    else
                    {
                        this.tsbSetLabelColorEset.ToolTipText = Properties.Resources.ToolTipSetLabelColor;
                        this.tsbSetCommentEset.ToolTipText = Properties.Resources.ToolTipSetComment;
                    }
                }

                this.tsmAddSampleFav.Enabled = PathUtility.DirectoryExists(FavoritesManager.SampleFolderPath);
            }
        }

        /// <summary>
        /// The update view.
        /// </summary>
        private void UpdateView()
        {
            this.UpdateToolBar();

            this.UpdateFilterMenu();

            // ショートカットキーがあるので都度更新する
            this.UpdateContextMenu();
        }
    }
}
