﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Globalization;
using System.Threading;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.EffectBrowser.Data
{
    using System;
    using System.Drawing;
    using System.IO;
    using System.Linq;

    using EffectMaker.Foundation.Utility;
    using EffectMaker.UIControls.EffectBrowser.Data.FileCache;

    /// <summary>
    /// The file info.
    /// </summary>
    public partial class FileInfo
    {
        #region Fields

        /// <summary>
        /// The byte size.
        /// </summary>
        private int byteSize;

        /// <summary>
        /// The create timestamp.
        /// </summary>
        private DateTime createTimestamp;

        /// <summary>
        /// The dir name.
        /// </summary>
        private string dirName = string.Empty;

        /// <summary>
        /// The file kind.
        /// </summary>
        private FileKindType fileKind;

        /// <summary>
        /// The file name.
        /// </summary>
        private string fileName = string.Empty;

        /// <summary>
        /// The icon.
        /// </summary>
        private Bitmap icon;

        /// <summary>
        /// The large icon.
        /// </summary>
        private Bitmap largeIcon;

        /// <summary>
        /// The is opend.
        /// </summary>
        private bool isOpend;

        /// <summary>
        /// The label color.
        /// </summary>
        private LabelColorType labelColor;

        /// <summary>
        /// The update timestamp.
        /// </summary>
        private DateTime updateTimestamp;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="FileInfo"/> class.
        /// </summary>
        public FileInfo()
        {
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="FileInfo"/> class.
        /// </summary>
        /// <param name="cache">
        /// The cache.
        /// </param>
        public FileInfo(FileCacheValue cache)
        {
            System.Diagnostics.Debug.Assert(cache != null, "cache != null");

            if (System.Text.RegularExpressions.Regex.IsMatch(cache.FileFullPath, @"^\\\\[^\\]+\\[^\\]+\\?$"))
            {
                // Active Directory直下の場合は、Pathクラスのメソッドを使用できない
                string[] names = cache.FileFullPath.Split('\\');
                this.DirName = @"\\" + names[2];
                this.FileName = names.Last();
            }
            else
            {
                this.DirName = Path.GetDirectoryName(cache.FileFullPath);
                this.FileName = Path.GetFileName(cache.FileFullPath);
            }

            this.UpdateTimestamp = cache.UpdateTimestamp;
            this.CreateTimestamp = cache.CreateTimestamp;
            this.FileKind = cache.FileKind;
            this.ByteSize = cache.ByteSize;
            this.Icon = cache.Icon;
            this.LargeIcon = cache.LargeIcon;
            this.IsOldData = cache.IsOldData;
            this.LabelColor = cache.LabelColor;
            this.Comment = cache.Comment;
            this.EmitterNames = cache.EmitterNames;
            this.ThumnailFilePath = cache.ThumnailFilePath;

            switch (this.FileKind)
            {
                case FileKindType.TextureFile:
                case FileKindType.PrimitiveFile:
                    this.IsInternalLoaded = this.Icon != Constants.FileKindInfos[this.FileKind].Icon;
                    break;

                case FileKindType.EsetFile:
                    {
                        // 検索パスのからみがあるので毎回検索する必要がある
#if false
                    if ((cache.Textures != null) && (cache.Primitives != null))
                    {
                        LoadFromFile(FileKindType.EsetFile);
                        IsInternalLoaded = true;
                    }
#endif

                        break;
                    }
            }
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets the all items.
        /// </summary>
        public FileInfo[] AllItems
        {
            get
            {
                var items = new FileInfo[this.ItemCount];
                {
                    items[0] = this;

                    if (this.HasChildren)
                    {
                        var count = this.Children.Count();
                        for (var i = 0; i != count; ++i)
                        {
                            items[i + 1] = this.Children[i];
                        }
                    }
                }

                return items;
            }
        }

        /// <summary>
        /// Gets or sets the byte size.
        /// </summary>
        public int ByteSize
        {
            get
            {
                return this.byteSize;
            }

            set
            {
                this.byteSize = value;
                this.DisplayByteSize = MathUtility.XBytesText(this.ByteSize);
            }
        }

        /// <summary>
        /// Gets or sets the children.
        /// </summary>
        public FileInfo[] Children { get; set; }

        /// <summary>
        /// コメントを取得または設定します。
        /// </summary>
        public string Comment { get; set; }

        /// <summary>
        /// エミッタ名配列を取得または設定します。
        /// </summary>
        public string[] EmitterNames { get; set; }

        /// <summary>
        /// Gets or sets the create timestamp.
        /// </summary>
        public DateTime CreateTimestamp
        {
            get
            {
                return this.createTimestamp;
            }

            set
            {
                this.createTimestamp = value;
                this.DisplayCreateTimestamp = this.CreateTimestamp.ToShortDateString();
            }
        }

        /// <summary>
        /// Gets or sets the depth.
        /// </summary>
        public int Depth { get; set; }

        /// <summary>
        /// Gets or sets the dir name.
        /// </summary>
        public string DirName
        {
            get
            {
                return this.dirName;
            }

            set
            {
                this.dirName = value;
                this.FileFullPath = Path.Combine(this.DirName, this.FileName);
                this.IsDirectory = Utilities.PathUtility.DirectoryExists(this.FileFullPath);
            }
        }

        /// <summary>
        /// Gets the display byte size.
        /// </summary>
        public string DisplayByteSize { get; private set; }

        /// <summary>
        /// Gets the display comment.
        /// </summary>
        public string DisplayComment
        {
            get { return this.Comment; }
        }

        // 以下描画用

        /// <summary>
        /// Gets the display create timestamp.
        /// </summary>
        public string DisplayCreateTimestamp { get; private set; }

        /// <summary>
        /// Gets the display file kind.
        /// </summary>
        public string DisplayFileKind { get; private set; }

        /// <summary>
        /// Gets the display file name.
        /// </summary>
        public string DisplayFileName
        {
            get { return this.FileName; }
        }

        /// <summary>
        /// Gets the display label color.
        /// </summary>
        public string DisplayLabelColor { get; private set; }

        /// <summary>
        /// Gets the display path.
        /// </summary>
        public string DisplayPath
        {
            get { return this.DirName; }
        }

        /// <summary>
        /// Gets the display update timestamp.
        /// </summary>
        public string DisplayUpdateTimestamp { get; private set; }

        /// <summary>
        /// Gets the file full path.
        /// </summary>
        public string FileFullPath { get; private set; }

        /// <summary>
        /// Gets or sets the file kind.
        /// </summary>
        public FileKindType FileKind
        {
            get
            {
                return this.fileKind;
            }

            set
            {
                this.fileKind = value;
                this.DisplayFileKind = Constants.FileKindInfos[this.FileKind].Name;
            }
        }

        /// <summary>
        /// Gets or sets the file name.
        /// </summary>
        public string FileName
        {
            get
            {
                return this.fileName;
            }

            set
            {
                this.fileName = value;
                this.FileFullPath = Path.Combine(this.DirName, this.FileName);
                this.IsDirectory = Utilities.PathUtility.DirectoryExists(this.FileFullPath);
            }
        }

        /// <summary>
        /// Gets a value indicating whether has children.
        /// </summary>
        public bool HasChildren
        {
            get { return (this.Children != null) && this.Children.Any(); }
        }

        /// <summary>
        /// Gets or sets the icon.
        /// </summary>
        public Bitmap Icon
        {
            get
            {
                return this.icon ??
                    (this.FileKind == FileKindType.EsetFile && this.IsOldData ?
                        Properties.Resources.Icon_EmitterSet_Old :
                        Constants.FileKindInfos[this.FileKind].Icon);
            }

            set
            {
                this.icon = value;
            }
        }

        /// <summary>
        /// Gets or sets the icon.
        /// </summary>
        public Bitmap LargeIcon
        {
            get
            {
                return this.largeIcon ??
                    (this.FileKind == FileKindType.EsetFile && this.IsOldData ?
                        Properties.Resources.Icon_OldEmitterSet_LSize :
                        Constants.FileKindInfos[this.FileKind].LargeIcon);
            }

            set
            {
                this.largeIcon = value;
            }
        }

        /// <summary>
        /// FE1形式のデータか否かを表す値
        /// </summary>
        public bool IsOldData { get; set; }

        // 作業用

        /// <summary>
        /// Gets or sets the index.
        /// </summary>
        public int Index { get; set; }

        /// <summary>
        /// Gets a value indicating whether is cached.
        /// </summary>
        public bool IsCached
        {
            get { return FileCacheManager.Instance.IsCached(this.FileFullPath, this.UpdateTimestamp); }
        }

        /// <summary>
        /// Gets a value indicating whether is directory.
        /// </summary>
        public bool IsDirectory { get; private set; }

        /// <summary>
        /// Gets a value indicating whether is internal loaded.
        /// </summary>
        public bool IsInternalLoaded { get; private set; }

        /// <summary>
        /// Gets or sets a value indicating whether is opend.
        /// </summary>
        public bool IsOpend
        {
            get
            {
                return this.isOpend;
            }

            set
            {
                if (this.isOpend == value)
                {
                    return;
                }

                this.isOpend = value;
                if (this.isOpend)
                {
                    this.LoadFromFile(new [] { FileKindType.EsetFile });
                }
            }
        }

        /// <summary>
        /// Gets the item count.
        /// </summary>
        public int ItemCount
        {
            get
            {
                return this.HasChildren ? 1 + this.Children.Count() : 1;
            }
        }

        /// <summary>
        /// Gets or sets the label color.
        /// </summary>
        public LabelColorType LabelColor
        {
            get
            {
                return this.labelColor;
            }

            set
            {
                this.labelColor = value;

                // UIスレッド以外でリソースにアクセスする際は、スレッドを立てるごとにカルチャーの再設定が必要
                // スレッドの作成ルートが複数あり特定しきれないので、アクセス直前での再設定に変更
                var cultureInfo = CultureInfo.CreateSpecificCulture(OptionStore.RuntimeOptions.CurrentCulture);
                Thread.CurrentThread.CurrentCulture = cultureInfo;
                Thread.CurrentThread.CurrentUICulture = cultureInfo;

                this.DisplayLabelColor = string.Format(Properties.Resources.Color_Name, (int)this.LabelColor);
            }
        }

        /// <summary>
        /// Gets or sets the parent.
        /// </summary>
        public FileInfo Parent { get; set; }

        /// <summary>
        /// Gets or sets the update timestamp.
        /// </summary>
        public DateTime UpdateTimestamp
        {
            get
            {
                return this.updateTimestamp;
            }

            set
            {
                this.updateTimestamp = value;
                this.DisplayUpdateTimestamp = this.UpdateTimestamp.ToShortDateString() + " "
                                              + this.UpdateTimestamp.ToShortTimeString();
            }
        }

        /// <summary>
        /// サムネイル画像のファイルパスを取得または設定します。
        /// </summary>
        public string ThumnailFilePath { get; set; }

        /// <summary>
        /// エフェクトメーカーで編集可能なデータ種別かを取得します。
        /// </summary>
        public bool IsOpenable
        {
            get
            {
                return
                    this.FileKind == FileKindType.EsetFile ||
                    (this.FileKind == FileKindType.PreviewFile && !this.IsOldData) ||
                    this.FileKind == FileKindType.WorkspaceFile;
            }
        }

        /// <summary>
        /// コメントやラベルカラーが編集可能なデータ種別かを取得します。
        /// </summary>
        public bool IsEditable
        {
            get
            {
                var work = WorkspaceRootViewModel.Instance.WorkspaceViewModel;
                var esets = work.Children.OfType<EmitterSetViewModel>().ToArray();
                var prevs = esets.SelectMany(e => e.GetChildren<PreviewViewModel>());

                if (this.FileKind == FileKindType.EsetFile && !this.IsOldData)
                {
                    return esets.All(e => Path.Combine(e.FilePath, e.FileName + Constants.ExtEsetFile) != this.FileFullPath);
                }

                if (this.FileKind == FileKindType.PreviewFile && !this.IsOldData)
                {
                    return prevs.All(e => Path.Combine(e.FilePath, e.FileName + Constants.ExtPreviewFile) != this.FileFullPath);
                }

                if (this.FileKind == FileKindType.WorkspaceFile)
                {
                    return work.FilePath != this.FileFullPath;
                }

                return this.FileKind == FileKindType.TextureFile ||
                       this.FileKind == FileKindType.PrimitiveFile;
            }
        }

        /// <summary>
        /// 本来はコメントやラベルカラー編集が可能だけれど、EffectMakerで編集中であるためにそれが不可能になっているか否かを取得します。
        /// </summary>
        public bool IsBlockingEditByOpened
        {
            get
            {
                bool canOpenAndEditKind =  (this.FileKind == FileKindType.EsetFile && !this.IsOldData)
                    || (this.FileKind == FileKindType.PreviewFile && !this.IsOldData)
                    || (this.FileKind == FileKindType.WorkspaceFile);
                return canOpenAndEditKind && !this.IsEditable;
            }
        }

        #endregion
    }
}
