﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Xml;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.Foundation.Disposables;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.EffectBrowser.Utilities;

namespace EffectMaker.UIControls.EffectBrowser.Data.FileCache
{
    /// <summary>
    /// The file cache value.
    /// </summary>
    public partial class FileCacheValue
    {
        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="FileCacheValue"/> class.
        /// </summary>
        /// <param name="filePath">
        /// The file path.
        /// </param>
        /// <param name="isDeepLoad">
        /// The is deep load.
        /// </param>
        public FileCacheValue(string filePath, bool isDeepLoad)
        {
            this.IsValid = false;
            this.IsDirectory = Utilities.PathUtility.DirectoryExists(filePath);

            if (this.IsDirectory == false)
            {
                var ext = Path.GetExtension(filePath);
                System.Diagnostics.Debug.Assert(ext != null, "ext != null");
                if (Constants.SupportedExtTypes.ContainsKey(ext) == false)
                {
                    // todo:エラー通知を出す
                    this.IsValid = false;
                    return;
                }
            }

            this.FileFullPath = filePath;

            if (this.Load(this.IsDirectory, null, isDeepLoad) == false)
            {
                // todo:エラー通知を出す
                this.IsValid = false;
                return;
            }

            this.IsValid = true;
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets the byte size.
        /// </summary>
        public int ByteSize { get; private set; }

        /// <summary>
        /// Gets the comment.
        /// </summary>
        public string Comment { get; private set; }

        /// <summary>
        /// Gets the create timestamp.
        /// </summary>
        public DateTime CreateTimestamp { get; private set; }

        /// <summary>
        /// Gets or sets the file full path.
        /// </summary>
        public string FileFullPath { get; set; }

        /// <summary>
        /// Gets the file kind.
        /// </summary>
        public FileKindType FileKind
        {
            get
            {
                return this.IsDirectory
                           ? FileKindType.Folder
                           : FileUtility.GetFileKindFormExt(Path.GetExtension(this.FileFullPath));
            }
        }

        /// <summary>
        /// Gets the icon.
        /// </summary>
        public Bitmap Icon { get; private set; }

        /// <summary>
        /// Gets the large icon.
        /// </summary>
        public Bitmap LargeIcon { get; private set; }

        /// <summary>
        /// Gets a value indicating whether is deep loaded.
        /// </summary>
        public bool IsDeepLoaded { get; private set; }

        /// <summary>
        /// Gets a value indicating whether is directory.
        /// </summary>
        public bool IsDirectory { get; private set; }

        /// <summary>
        /// Gets a value indicating whether is valid.
        /// </summary>
        public bool IsValid { get; private set; }

        /// <summary>
        /// FE1形式のデータか否かを表す値
        /// </summary>
        public bool IsOldData { get; private set; }

        /// <summary>
        /// Gets the label color.
        /// </summary>
        public LabelColorType LabelColor { get; private set; }

        /// <summary>
        /// Gets the primitives.
        /// </summary>
        public string[] Primitives { get; private set; }

        /// <summary>
        /// Gets the textures.
        /// </summary>
        public string[] Textures { get; private set; }

        /// <summary>
        /// Gets the combiners.
        /// </summary>
        public string[] Combiners { get; private set; }

        /// <summary>
        /// エミッタ名配列を取得します。
        /// </summary>
        public string[] EmitterNames { get; private set; }

        /// <summary>
        /// サムネイル画像のファイルパスを取得または設定します。
        /// </summary>
        public string ThumnailFilePath { get; private set; }

        /// <summary>
        /// Gets the update timestamp.
        /// </summary>
        public DateTime UpdateTimestamp { get; private set; }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The get chidlen.
        /// </summary>
        /// <param name="kind">
        /// The kind.
        /// </param>
        /// <returns>
        /// The <see cref="string[]"/>.
        /// </returns>
        public string[] GetChidlen(FileKindType kind)
        {
            switch (kind)
            {
                case FileKindType.TextureFile:
                    return this.Textures;
                case FileKindType.PrimitiveFile:
                    return this.Primitives;
                case FileKindType.CombinerFile:
                    return this.Combiners;
                default:
                    throw new ArgumentException("Kind must be asset file type.");
            }
        }

        /// <summary>
        /// The load.
        /// </summary>
        /// <param name="isDirectory">
        /// The is directory.
        /// </param>
        /// <param name="updateTimestamp">
        /// The update timestamp.
        /// </param>
        /// <param name="isDeepLoad">
        /// The is deep load.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public bool Load(bool isDirectory, DateTime? updateTimestamp, bool isDeepLoad)
        {
            var ext = Path.GetExtension(this.FileFullPath);

            if (isDirectory == false)
            {
                System.Diagnostics.Debug.Assert(ext != null, "ext != null");
                if (Constants.SupportedExtTypes.ContainsKey(ext) == false)
                {
                    // todo:エラー通知を出す
                    return false;
                }
            }

            this.UpdateTimestamp = updateTimestamp ?? File.GetLastWriteTime(this.FileFullPath);
            this.CreateTimestamp = File.GetCreationTime(this.FileFullPath);

            if (isDirectory == false)
            {
                this.ByteSize = (int)new System.IO.FileInfo(this.FileFullPath).Length;

                if (isDeepLoad == false)
                {
                    return true;
                }

                using (new AnonymousDisposable(() => this.IsDeepLoaded = true))
                {
                    switch (FileUtility.GetFileKindFormExt(ext))
                    {
                        case FileKindType.EsetFile:
                            if (this.LoadEset() == false)
                            {
                                return false;
                            }

                            break;

                        case FileKindType.PreviewFile:
                            if (this.LoadPrev() == false)
                            {
                                return false;
                            }

                            break;

                        case FileKindType.WorkspaceFile:
                            if (this.LoadWork() == false)
                            {
                                return false;
                            }

                            break;

                        case FileKindType.TextureFile:
                            if (this.LoadTexture() == false)
                            {
                                return false;
                            }

                            break;

                        case FileKindType.PrimitiveFile:
                            this.LoadPrimitive();

                            break;
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// The read deep loaded.
        /// </summary>
        public void ReadDeepLoaded()
        {
            this.IsDeepLoaded = false;
        }

        /// <summary>
        /// The update cache timestamp.
        /// </summary>
        public void UpdateCacheTimestamp()
        {
            this.UpdateTimestamp = File.GetLastWriteTime(this.FileFullPath);
        }

        #endregion

        #region Methods

        /// <summary>
        /// The get label color from string.
        /// </summary>
        /// <param name="rgbStr">
        /// The rgb str.
        /// </param>
        /// <returns>
        /// The <see cref="LabelColorType"/>.
        /// </returns>
        private static LabelColorType GetLabelColorFromString(string rgbStr)
        {
            if (string.IsNullOrEmpty(rgbStr))
            {
                return LabelColorType.Color0;
            }

            var splitted = rgbStr.Split(' ');
            if (splitted.Length != 3)
            {
                return LabelColorType.Color0;
            }

            Color color;
            try
            {
                color = Color.FromArgb(
                    (int)(255 * float.Parse(splitted[0])),
                    (int)(255 * float.Parse(splitted[1])),
                    (int)(255 * float.Parse(splitted[2])));
            }
            catch
            {
                return LabelColorType.Color0;
            }

            return (LabelColorType)UIConstants.GetIndexFromRgb(color);
        }

        /// <summary>
        /// The load texture.
        /// </summary>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        private bool LoadTexture()
        {
            var resultWithData = BusinessLogic.Manager.TextureManager.Instance.LoadTexture(this.FileFullPath, true);

            if (resultWithData.ResultCode != LoadTextureResultCode.Success)
            {
                return false;
            }

            var combinedImage = new Bitmap(Constants.IconSize * 2, Constants.IconSize, PixelFormat.Format32bppArgb);

            System.Diagnostics.Debug.Assert(resultWithData.TextureData is FtxTextureData, "resultWithData.TextureData is FtxTextureData");
            var ftxTarget = (FtxTextureData)resultWithData.TextureData;
            var rgbMatrix = RenderUtility.CreateRgbColorMatrix(ftxTarget.CompSel, ftxTarget.PixelFormat);
            var alphaMatrix = RenderUtility.CreateAlphaColorMatrix(ftxTarget.CompSel, ftxTarget.PixelFormat);

            var iconSize = new Size(Constants.IconSize, Constants.IconSize);
            var srcImage = ftxTarget.GeneratePreviewBitmap(iconSize);
            var rgbImage = TextureData.ResizeBitmap(srcImage, iconSize, rgbMatrix);
            rgbImage = ColorUtility.ConvertBitmapWithGamma(
                rgbImage, rgbMatrix, true, resultWithData.TextureData.PixelFormat.IsSrgb());
            var alphaImage = TextureData.ResizeBitmap(srcImage, iconSize);
            alphaImage = ColorUtility.ConvertBitmapWithGamma(
                alphaImage, alphaMatrix, false, resultWithData.TextureData.PixelFormat.IsSrgb());

            using (var g = Graphics.FromImage(combinedImage))
            {
                g.DrawImage(rgbImage, 0, 0);
                g.DrawImage(alphaImage, Constants.IconSize, 0);
            }

            // 大きなアイコンイメージの作成
            var iconSizeLarge = new Size(Constants.LargeIconSize, Constants.LargeIconSize);
            var combinedImageLarge = new Bitmap(iconSizeLarge.Width * 2, iconSizeLarge.Height * 2, PixelFormat.Format32bppArgb);

            var srcImageLarge = ftxTarget.GeneratePreviewBitmap(iconSizeLarge);
            var rgbImageLarge = TextureData.ResizeBitmap(srcImageLarge, iconSizeLarge, rgbMatrix, true);
            rgbImageLarge = ColorUtility.ConvertBitmapWithGamma(
                rgbImageLarge, rgbMatrix, true, resultWithData.TextureData.PixelFormat.IsSrgb());
            var alphaImageLarge = TextureData.ResizeBitmap(srcImageLarge, iconSizeLarge, null, true);
            alphaImageLarge = ColorUtility.ConvertBitmapWithGamma(
                alphaImageLarge, alphaMatrix, false, resultWithData.TextureData.PixelFormat.IsSrgb());

            using (var g = Graphics.FromImage(combinedImageLarge))
            using (var grayBrush = new SolidBrush(Color.Gray))
            {
                g.FillRectangle(grayBrush, g.ClipBounds);
                g.DrawImage(rgbImageLarge, 0, 0);
                g.DrawImage(alphaImageLarge, iconSizeLarge.Width, 0);

                // 仕切り線
                g.FillRectangle(grayBrush, iconSizeLarge.Width - 1, 0, 2, iconSizeLarge.Height);

                // 非正方形時の上書き塗り
                if (ftxTarget.Width > ftxTarget.Height)
                {
                    int padding = (iconSizeLarge.Height - (int)((double)iconSizeLarge.Width * ((double)ftxTarget.Height / (double)ftxTarget.Width))) / 2;

                    g.FillRectangle(grayBrush, 0, 0, iconSizeLarge.Width, padding);
                    g.FillRectangle(grayBrush, 0, iconSizeLarge.Height - padding, iconSizeLarge.Width, padding);
                    g.FillRectangle(grayBrush, iconSizeLarge.Width, 0, iconSizeLarge.Width, padding);
                    g.FillRectangle(grayBrush, iconSizeLarge.Width, iconSizeLarge.Height - padding, iconSizeLarge.Width, padding);
                }
                else if (ftxTarget.Width < ftxTarget.Height)
                {
                    int padding = (iconSizeLarge.Width - (int)((double)iconSizeLarge.Height * ((double)ftxTarget.Width / (double)ftxTarget.Height))) / 2;

                    g.FillRectangle(grayBrush, 0, 0, padding, iconSizeLarge.Height);
                    g.FillRectangle(grayBrush, iconSizeLarge.Width - padding, 0, padding, iconSizeLarge.Height);
                    g.FillRectangle(grayBrush, iconSizeLarge.Width, 0, padding, iconSizeLarge.Height);
                    g.FillRectangle(grayBrush, iconSizeLarge.Width + iconSizeLarge.Width - padding, 0, padding, iconSizeLarge.Height);
                }
            }

            this.Icon = combinedImage;
            this.LargeIcon = combinedImageLarge;
            this.Comment = resultWithData.TextureData.Comment;
            this.LabelColor = GetLabelColorFromString(resultWithData.TextureData.LabelColor);

            return true;
        }

        /// <summary>
        /// The load primitive.
        /// </summary>
        private void LoadPrimitive()
        {
            // コメントとラベルカラーの読み込み
            var result = PrimitiveManager.Instance.LoadComment(this.FileFullPath);
            this.Comment = result.Comment;
            this.LabelColor = GetLabelColorFromString(result.Color);

            // サムネイル画像を読み込む
            {
                string dirPath = Path.GetDirectoryName(this.FileFullPath);
                string noExtName = Path.GetFileNameWithoutExtension(this.FileFullPath);
                string fileName = Path.GetFileName(this.FileFullPath);

                string jpgPath = Path.Combine(dirPath, noExtName + ".jpg");
                string pngPath = Path.Combine(dirPath, noExtName + ".png");

                string nwJpgPath = Path.Combine(dirPath, ".NwThumbs", fileName + ".jpg");
                string nwPngPath = Path.Combine(dirPath, ".NwThumbs", fileName + ".png");

                string targetPath = string.Empty;

                if (File.Exists(jpgPath))
                {
                    targetPath = jpgPath;
                }
                else if (File.Exists(pngPath))
                {
                    targetPath = pngPath;
                }
                else if (File.Exists(nwJpgPath))
                {
                    targetPath = nwJpgPath;
                }
                else if (File.Exists(nwPngPath))
                {
                    targetPath = nwPngPath;
                }

                if (string.IsNullOrEmpty(targetPath) == false)
                {
                    using (var fs = new FileStream(targetPath, FileMode.Open, FileAccess.Read))
                    {
                        Bitmap tmpBitmap = new Bitmap(fs);
                        Size iconSize = new Size(Constants.IconSize, Constants.IconSize);
                        Bitmap resizedBmp = TextureData.ResizeBitmap(tmpBitmap, iconSize);
                        this.Icon = tmpBitmap == resizedBmp ? (Bitmap)resizedBmp.Clone() : resizedBmp;
                        this.LargeIcon = tmpBitmap;
                        fs.Close();
                    }

                    this.ThumnailFilePath = targetPath;
                }
            }
        }

        /// <summary>
        /// エフェクトブラウザ独自実装した方法でPrevファイルを読む
        /// </summary>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        private bool LoadPrev()
        {
            if (Utilities.PathUtility.FileExists(this.FileFullPath) == false)
            {
                this.IsValid = false;
                return false;
            }

            try
            {
                using (var fs = new FileStream(this.FileFullPath, FileMode.Open, FileAccess.Read))
                {
                    var reader = XmlReader.Create(
                        fs,
                        new XmlReaderSettings { IgnoreComments = true, IgnoreWhitespace = true });

                    var inPreviewBasicFileData = false;
                    var inComment = false;
                    var inLabelColor = false;

                    while (reader.Read())
                    {
                        if (reader.IsEmptyElement)
                        {
                            if (reader.Name == "Comment")
                            {
                                // このXML読み込みだと<Comment />という記法から空の値を拾えない
                                // とりあえず特殊対応にしておく
                                this.Comment = string.Empty;
                            }

                            continue;
                        }

                        var name = reader.Name;

                        switch (name)
                        {
                            case "GameConfigDataXml":
                                // 旧形式であることのマーキング
                                this.IsOldData = true;
                                return true;
                            case "PreviewBasicFileData":
                                switch (reader.NodeType)
                                {
                                    case XmlNodeType.Element:
                                        inPreviewBasicFileData = true;
                                        break;

                                    case XmlNodeType.EndElement:
                                        inPreviewBasicFileData = false;
                                        break;
                                }

                                break;
                        }

                        if (inPreviewBasicFileData)
                        {
                            switch (name)
                            {
                                case "Comment":
                                    switch (reader.NodeType)
                                    {
                                        case XmlNodeType.Element:
                                            inComment = true;
                                            break;

                                        case XmlNodeType.EndElement:
                                            inComment = false;
                                            break;
                                    }

                                    break;

                                case "LabelColor":
                                    switch (reader.NodeType)
                                    {
                                        case XmlNodeType.Element:
                                            inLabelColor = true;
                                            break;

                                        case XmlNodeType.EndElement:
                                            inLabelColor = false;
                                            break;
                                    }

                                    break;
                            }

                            if (inComment)
                            {
                                if (reader.NodeType == XmlNodeType.Text)
                                {
                                    this.Comment = reader.Value;
                                }
                            }
                            else if (inLabelColor)
                            {
                                if (reader.NodeType == XmlNodeType.Text)
                                {
                                    this.LabelColor = (LabelColorType)int.Parse(reader.Value);
                                }
                            }
                        }
                    }
                }
            }
            catch
            {
                // 読み込み失敗時は失敗として続行
                Logger.Log("LogView", LogLevels.Error, Properties.Resources.ErrorEsetLoadFailed, this.FileFullPath);
                this.IsValid = false;
                return false;
            }

            return true;
        }

        /// <summary>
        /// エフェクトブラウザ独自実装した方法でWorkファイルを読む
        /// </summary>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        private bool LoadWork()
        {
            if (Utilities.PathUtility.FileExists(this.FileFullPath) == false)
            {
                this.IsValid = false;
                return false;
            }

            try
            {
                using (var fs = new FileStream(this.FileFullPath, FileMode.Open, FileAccess.Read))
                {
                    var reader = XmlReader.Create(
                        fs,
                        new XmlReaderSettings { IgnoreComments = true, IgnoreWhitespace = true });

                    var inViewerData = false;
                    var inComment = false;
                    var inLabelColor = false;

                    while (reader.Read())
                    {
                        if (reader.IsEmptyElement)
                        {
                            if (reader.Name == "Comment")
                            {
                                // このXML読み込みだと<Comment />という記法から空の値を拾えない
                                // とりあえず特殊対応にしておく
                                this.Comment = string.Empty;
                            }

                            continue;
                        }

                        var name = reader.Name;

                        switch (name)
                        {
                            case "ViewerData":
                                switch (reader.NodeType)
                                {
                                    case XmlNodeType.Element:
                                        inViewerData = true;
                                        break;

                                    case XmlNodeType.EndElement:
                                        inViewerData = false;
                                        break;
                                }

                                break;
                        }

                        if (inViewerData)
                        {
                            switch (name)
                            {
                                case "Comment":
                                    switch (reader.NodeType)
                                    {
                                        case XmlNodeType.Element:
                                            inComment = true;
                                            break;

                                        case XmlNodeType.EndElement:
                                            inComment = false;
                                            break;
                                    }

                                    break;

                                case "LabelColor":
                                    switch (reader.NodeType)
                                    {
                                        case XmlNodeType.Element:
                                            inLabelColor = true;
                                            break;

                                        case XmlNodeType.EndElement:
                                            inLabelColor = false;
                                            break;
                                    }

                                    break;
                            }

                            if (inComment)
                            {
                                if (reader.NodeType == XmlNodeType.Text)
                                {
                                    this.Comment = reader.Value;
                                }
                            }
                            else if (inLabelColor)
                            {
                                if (reader.NodeType == XmlNodeType.Text)
                                {
                                    this.LabelColor = (LabelColorType)int.Parse(reader.Value);
                                }
                            }
                        }
                    }
                }
            }
            catch
            {
                // 読み込み失敗時は失敗として続行
                Logger.Log("LogView", LogLevels.Error, Properties.Resources.ErrorEsetLoadFailed, this.FileFullPath);
                this.IsValid = false;
                return false;
            }

            return true;
        }

        #endregion
    }
}
