﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using System.Xml.Serialization;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.Foundation.Serialization;
using EffectMaker.UIControls.EffectBrowser.Config;
using EffectMaker.UIControls.EffectBrowser.Controls.FileTreeView;
using EffectMaker.UIControls.EffectBrowser.Utilities;

namespace EffectMaker.UIControls.EffectBrowser.Data.Favorites
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;
    using System.Xml.Serialization;

    using EffectMaker.UIControls.EffectBrowser.Config;
    using EffectMaker.UIControls.EffectBrowser.Controls.FileTreeView;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The favorites manager.
    /// </summary>
    public class FavoritesManager
    {
        #region Fields

        /// <summary>
        /// The view.
        /// </summary>
        private EBFavoritesFileTreeView view;

        #endregion

        #region Public Properties

        /// <summary>
        /// サンプルフォルダのパスを取得します。
        /// </summary>
        public static string SampleFolderPath
        {
            get
            {
                var sampleFolderPath = IOConstants.ToolsDirectoryPath;
                sampleFolderPath = Path.GetDirectoryName(sampleFolderPath);
                sampleFolderPath = Path.Combine(sampleFolderPath, Properties.Resources.FavoritesConfig_SampleDataFilepath);
                return sampleFolderPath;
            }
        }

        /// <summary>
        /// テンプレートフォルダのパスを取得します。
        /// </summary>
        public static string TemplateFolderPath
        {
            get
            {
                var templateFolderPath = IOConstants.ToolsDirectoryPath;
                templateFolderPath = Path.GetDirectoryName(templateFolderPath);
                templateFolderPath = Path.Combine(templateFolderPath, Properties.Resources.FavoritesConfig_TemplateDataFilepath);
                return templateFolderPath;
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The add bookmark.
        /// </summary>
        /// <param name="parent">
        /// The parent.
        /// </param>
        /// <param name="name">
        /// The name.
        /// </param>
        /// <param name="path">
        /// The path.
        /// </param>
        /// <returns>
        /// The <see cref="EBFileTreeNode"/>.
        /// </returns>
        public EBFileTreeNode AddBookmark(EBFileTreeNode parent, string name, string path)
        {
            return this.view.AddNode(
                parent,
                name,
                path,
                EBFileTreeView.IconKeyBookmark,
                EBFileTreeNode.NodeKindType.Bookmark);
        }

        /// <summary>
        /// The add directory.
        /// </summary>
        /// <param name="parent">
        /// The parent.
        /// </param>
        /// <param name="name">
        /// The name.
        /// </param>
        /// <param name="path">
        /// The path.
        /// </param>
        /// <returns>
        /// The <see cref="EBFileTreeNode"/>.
        /// </returns>
        public EBFileTreeNode AddDirectory(EBFileTreeNode parent, string name, string path)
        {
            return this.view.AddNode(
                parent,
                name,
                path,
                EBFileTreeView.IconKeyDirectory,
                EBFileTreeNode.NodeKindType.Directory);
        }

        /// <summary>
        /// The add fav folder.
        /// </summary>
        /// <param name="parent">
        /// The parent.
        /// </param>
        /// <param name="name">
        /// The name.
        /// </param>
        /// <returns>
        /// The <see cref="EBFileTreeNode"/>.
        /// </returns>
        public EBFileTreeNode AddFavFolder(EBFileTreeNode parent, string name)
        {
            return this.view.AddNode(
                parent,
                name,
                string.Empty,
                EBFileTreeView.IconKeyFavFolder,
                EBFileTreeNode.NodeKindType.FavFolder);
        }

        /// <summary>
        /// The delete.
        /// </summary>
        /// <param name="node">
        /// The node.
        /// </param>
        public void Delete(EBFileTreeNode node)
        {
            System.Diagnostics.Debug.Assert((node.NodeKind == EBFileTreeNode.NodeKindType.Bookmark) || (node.NodeKind == EBFileTreeNode.NodeKindType.FavFolder), "(node.NodeKind == EBFileTreeNode.NodeKindType.Bookmark) || (node.NodeKind == EBFileTreeNode.NodeKindType.FavFolder)");

            // フォルダの場合、フォルダ内に物があれば警告を出す
            if ((node.NodeKind == EBFileTreeNode.NodeKindType.FavFolder) && (node.Nodes.Count > 0))
            {
                var result = MessageBox.Show(
                    Properties.Resources.Favorites_WarningFolderDeleteMessage,
                    Properties.Resources.Favorites_WarningFolderDeleteTitle,
                    MessageBoxButtons.OKCancel,
                    MessageBoxIcon.Warning);

                if (result != DialogResult.OK)
                {
                    return;
                }
            }

            node.Remove();
        }

        /// <summary>
        /// The initialize.
        /// </summary>
        /// <param name="view">
        /// The view.
        /// </param>
        public void Initialize(EBFavoritesFileTreeView view)
        {
            this.view = view;
        }

        /// <summary>
        /// The load config.
        /// </summary>
        public void LoadConfig()
        {
            FavoritesConfig config = null;
            {
                try
                {
                    if (File.Exists(Constants.FavoritesFilepath))
                    {
                        config = SerializationHelper.LoadXmlDocSerializable<FavoritesConfig>(
                            Constants.FavoritesFilepath);
                    }
                }

                    // ReSharper disable once EmptyGeneralCatchClause
                catch
                {
                }
            }

            if (config == null)
            {
                config = this.CreateDefaultFavorites();
            }

            using (new UpdateBlock(this.view))
            {
                this.view.Nodes.Clear();
                this.CreateFavoritesFromConfig(null, config.Children);
            }
        }

        /// <summary>
        /// The rename.
        /// </summary>
        /// <param name="node">
        /// The node.
        /// </param>
        public void Rename(EBFileTreeNode node)
        {
            System.Diagnostics.Debug.Assert((node.NodeKind == EBFileTreeNode.NodeKindType.Bookmark) || (node.NodeKind == EBFileTreeNode.NodeKindType.FavFolder), "(node.NodeKind == EBFileTreeNode.NodeKindType.Bookmark) || (node.NodeKind == EBFileTreeNode.NodeKindType.FavFolder)");

            this.view.Rename(node);
        }

        /// <summary>
        /// The save config.
        /// </summary>
        public void SaveConfig()
        {
            var config = new FavoritesConfig();

            CreateConfig(config, this.view.Nodes.OfType<EBFileTreeNode>());

            try
            {
                SerializationHelper.SaveXmlDocSerializable(config, Constants.FavoritesFilepath);
            }
            catch
            {
                // Failed accessing the file, just skip saving the options.
                // (another EffectMaker instance is accessing it)
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The create config.
        /// </summary>
        /// <param name="dst">
        /// The dst.
        /// </param>
        /// <param name="sources">
        /// The sources.
        /// </param>
        private static void CreateConfig(Config.FavoriteFolder dst, IEnumerable<EBFileTreeNode> sources)
        {
            var fileTreeNodes = sources as EBFileTreeNode[] ?? sources.ToArray();

            dst.Children = fileTreeNodes.Where(x => x.IsConfigInfo).Select(CreateConfigNode).ToList();

            // フォルダの子を作る
            var dstFolders = dst.Children.OfType<Config.FavoriteFolder>();
            var srcFolders = fileTreeNodes.Where(x => x.NodeKind == EBFileTreeNode.NodeKindType.FavFolder);

            foreach (var v in dstFolders.Zip(srcFolders, (d, s) => new { Dst = d, Src = s }))
            {
                CreateConfig(v.Dst, v.Src.Nodes.OfType<EBFileTreeNode>());
            }
        }

        /// <summary>
        /// The create config node.
        /// </summary>
        /// <param name="src">
        /// The src.
        /// </param>
        /// <returns>
        /// The <see cref="FavoriteNodeBase"/>.
        /// </returns>
        private static Config.FavoriteNodeBase CreateConfigNode(EBFileTreeNode src)
        {
            Config.FavoriteNodeBase dst = null;
            {
                switch (src.NodeKind)
                {
                    case EBFileTreeNode.NodeKindType.FavFolder:
                        dst = new Config.FavoriteFolder { Name = src.Text, IsExpanded = src.IsExpanded };
                        break;

                    case EBFileTreeNode.NodeKindType.Bookmark:
                        dst = new Config.FavoriteBookmark { Name = src.Text, Path = src.Filepath };
                        break;

                    default:
                        System.Diagnostics.Debug.Assert(dst != null, "dst != null");
                        break;
                }
            }

            return dst;
        }

        /// <summary>
        /// The create favorites from config.
        /// </summary>
        /// <param name="parent">
        /// The parent.
        /// </param>
        /// <param name="children">
        /// The children.
        /// </param>
        private void CreateFavoritesFromConfig(EBFileTreeNode parent, IEnumerable<Config.FavoriteNodeBase> children)
        {
            if (children == null)
            {
                return;
            }

            foreach (var child in children)
            {
                if (child is Config.FavoriteBookmark)
                {
                    var boolmark = child as Config.FavoriteBookmark;

                    this.AddBookmark(parent, boolmark.Name, boolmark.Path);
                }
                else if (child is Config.FavoriteFolder)
                {
                    var folder = child as Config.FavoriteFolder;
                    var folderNode = this.AddFavFolder(parent, folder.Name);

                    if (folder.IsExpanded)
                    {
                        folderNode.Expand();
                    }
                    else
                    {
                        folderNode.Collapse();
                    }

                    this.CreateFavoritesFromConfig(folderNode, folder.Children);
                }
                else
                {
                    System.Diagnostics.Debug.Assert(false, "false");
                }
            }
        }

        /// <summary>
        /// サンプルフォルダを追加します。
        /// </summary>
        /// <returns>サンプルフォルダが見つかって追加したらtrue,見つからなかったらfalse.</returns>
        public bool AddSampleFolder()
        {
            if (!PathUtility.DirectoryExists(SampleFolderPath))
            {
                return false;
            }

            this.AddBookmark(
                null,
                Properties.Resources.FavoritesConfig_SampleDataName,
                SampleFolderPath);

            return true;
        }

        /// <summary>
        /// デフォルトのお気に入りを新規作成します。
        /// </summary>
        /// <returns>
        /// The <see cref="FavoritesConfig"/>.
        /// </returns>
        private FavoritesConfig CreateDefaultFavorites()
        {
            var config = new FavoritesConfig();

            var defaultFavorites = new List<Config.FavoriteNodeBase>();

            // サンプルフォルダを追加
            if (PathUtility.DirectoryExists(SampleFolderPath))
            {
                defaultFavorites.Add(new Config.FavoriteBookmark()
                {
                    Name = Properties.Resources.FavoritesConfig_SampleDataName,
                    Path = SampleFolderPath
                });
            }

            // テンプレートフォルダを追加
            if (PathUtility.DirectoryExists(TemplateFolderPath))
            {
                defaultFavorites.Add(new Config.FavoriteBookmark()
                {
                    Name = Properties.Resources.FavoritesConfig_TemplateDataName,
                    Path = TemplateFolderPath
                });
            }

            if (defaultFavorites.Any())
            {
                config.Children = defaultFavorites;

                // サンプルフォルダを選択状態にします。
                var firstFavorite = (Config.FavoriteBookmark)defaultFavorites[0];
                this.view.Directory = firstFavorite.Path;
            }

            return config;
        }

        #endregion
    }
}
