﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.SearchTab
{
    using System;
    using System.Diagnostics;
    using System.Windows.Forms;

    using EffectMaker.UIControls.EffectBrowser.Controls.Dialog;
    using EffectMaker.UIControls.EffectBrowser.Controls.FileListView;
    using EffectMaker.UIControls.EffectBrowser.Data;
    using EffectMaker.UIControls.EffectBrowser.Data.FileCache;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The search file panel.
    /// </summary>
    public partial class SearchFilePanel : UserControl
    {
        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="SearchFilePanel"/> class.
        /// </summary>
        public SearchFilePanel()
        {
            this.InitializeComponent();

            this.AutoScaleMode = AutoScaleMode.None;

            this.cmbSearchMode.SelectedIndex = 0;

            this.UpdateView();
        }

        #endregion

        #region Public Events

        /// <summary>
        /// The move all files button click.
        /// </summary>
        public event EventHandler MoveAllFilesButtonClick
        {
            add
            {
                this.btnMoveAllFiles.Click += value;
            }

            remove
            {
                this.btnMoveAllFiles.Click -= value;
            }
        }

        /// <summary>
        /// The move selected files button click.
        /// </summary>
        public event EventHandler MoveSelectedFilesButtonClick
        {
            add
            {
                this.btnMoveSelectedFiles.Click += value;
            }

            remove
            {
                this.btnMoveSelectedFiles.Click -= value;
            }
        }

        /// <summary>
        /// The reference file path changed.
        /// </summary>
        public event EventHandler ReferenceFilePathChanged
        {
            add
            {
                this.tbxReferenceFilePath.TextChanged += value;
            }

            remove
            {
                this.tbxReferenceFilePath.TextChanged -= value;
            }
        }

        /// <summary>
        /// The selected search mode changed.
        /// </summary>
        public event EventHandler SelectedSearchModeChanged
        {
            add
            {
                this.cmbSearchMode.SelectedIndexChanged += value;
            }

            remove
            {
                this.cmbSearchMode.SelectedIndexChanged -= value;
            }
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the directory.
        /// </summary>
        public string Directory { get; set; }

        /// <summary>
        /// Gets the file kind.
        /// </summary>
        public FileKindType FileKind { get; private set; }

        /// <summary>
        /// Gets the file path.
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.tbxReferenceFilePath.Text;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether move all files button enabled.
        /// </summary>
        public bool MoveAllFilesButtonEnabled
        {
            get
            {
                return this.btnMoveAllFiles.Enabled;
            }

            set
            {
                this.btnMoveAllFiles.Enabled = value;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether move selected files button enabled.
        /// </summary>
        public bool MoveSelectedFilesButtonEnabled
        {
            get
            {
                return this.btnMoveSelectedFiles.Enabled;
            }

            set
            {
                this.btnMoveSelectedFiles.Enabled = value;
            }
        }

        /// <summary>
        /// Gets the reference file path.
        /// </summary>
        public string ReferenceFilePath
        {
            get
            {
                return this.tbxReferenceFilePath.Text;
            }
        }

        /// <summary>
        /// Gets the search mode.
        /// </summary>
        public SearchFileInfo.SearchModeKind SearchMode
        {
            get
            {
                System.Diagnostics.Debug.Assert(
                    (this.cmbSearchMode.SelectedIndex == (int)SearchFileInfo.SearchModeKind.ReferenceFile)
                    || (this.cmbSearchMode.SelectedIndex == (int)SearchFileInfo.SearchModeKind.UnusedFile)
                    || (this.cmbSearchMode.SelectedIndex == (int)SearchFileInfo.SearchModeKind.EmitterSetUsedFile)
                    || (this.cmbSearchMode.SelectedIndex == (int)SearchFileInfo.SearchModeKind.Undefined),
                    "モードがおかしい");

                return (SearchFileInfo.SearchModeKind)this.cmbSearchMode.SelectedIndex;
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The initialize.
        /// </summary>
        /// <param name="type">
        /// The type.
        /// </param>
        public void Initialize(FileKindType type)
        {
            System.Diagnostics.Debug.Assert(
                (type == FileKindType.TextureFile) || (type == FileKindType.PrimitiveFile) || (type == FileKindType.CombinerFile),
                "(type == FileKindType.TextureFile) || (type == FileKindType.PrimitiveFile) || (type == FileKindType.CombinerFile)");

            this.FileKind = type;

            if (this.FileKind != FileKindType.TextureFile)
            {
                this.pnlTexturePreview.Parent.Controls.Remove(this.pnlTexturePreview);
            }

            var name = Constants.FileKindInfos[type].ShortName;

            for (var i = 0; i != this.cmbSearchMode.Items.Count; ++i)
            {
                System.Diagnostics.Debug.Assert(this.cmbSearchMode.Items[i] is string, "this.cmbSearchMode.Items[i] is string");
                this.cmbSearchMode.Items[i] = string.Format(this.cmbSearchMode.Items[i] as string, name);
            }

            this.lblMoveToUnUsed.Text = string.Format(this.lblMoveToUnUsed.Text, name);
            this.lblSelectFile.Text = string.Format(this.lblSelectFile.Text, name);

            this.ttpTooltip.SetToolTip(this.btnOpenFile, string.Format(Properties.Resources.Button_OpenFile, name));
            this.ttpTooltip.SetToolTip(this.btnClearFile, Properties.Resources.Button_ClearFile);
        }

        /// <summary>
        /// The refresh view.
        /// </summary>
        public void RefreshView()
        {
            this.UpdateView();
        }

        /// <summary>
        /// The reset.
        /// </summary>
        public void Reset()
        {
            this.cmbSearchMode.SelectedIndex = 0;
            this.tbxReferenceFilePath.Text = string.Empty;
        }

        #endregion

        #region Methods

        /// <summary>
        /// The update view.
        /// </summary>
        private void UpdateView()
        {
            using (new LockWindowUpdateBlock(this))
            using (new LayoutSuspendBlock(this))
            using (new ControlEventSuppressBlock())
            {
                switch (this.SearchMode)
                {
                    case SearchFileInfo.SearchModeKind.ReferenceFile:
                        this.pnlUnused.Visible = false;
                        this.pnlEmitterSet.Visible = false;
                        break;

                    case SearchFileInfo.SearchModeKind.UnusedFile:
                        this.pnlUnused.Visible = true;
                        this.pnlEmitterSet.Visible = false;
                        break;

                    case SearchFileInfo.SearchModeKind.EmitterSetUsedFile:
                        this.pnlEmitterSet.Visible = true;
                        this.pnlUnused.Visible = false;
                        break;
                }
            }
        }

        /// <summary>
        /// The btn clear file_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void BtnClearFile_Click(object sender, EventArgs e)
        {
            this.tbxReferenceFilePath.Text = string.Empty;
        }

        /// <summary>
        /// The btn open file_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void BtnOpenFile_Click(object sender, EventArgs e)
        {
            using (var dialog = new ChildFileSelectDialog(this.Directory, this.FileKind))
            {
                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    if (dialog.SelectedFile != null)
                    {
                        this.tbxReferenceFilePath.Text = dialog.SelectedFile.FileFullPath;
                    }
                }
            }
        }

        /// <summary>
        /// The btn open un used_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void BtnOpenUnUsed_Click(object sender, EventArgs e)
        {
            var dir =
                this.FileKind == FileKindType.TextureFile ? Constants.TexturesUnusedDirectory :
                this.FileKind == FileKindType.PrimitiveFile ? Constants.PrimitivesUnusedDirectory :
                this.FileKind == FileKindType.CombinerFile ? Constants.CombinersUnusedDirectory :
                string.Empty;

            PathUtility.SetupUnusedDirectory();
            Process.Start("EXPLORER.EXE", dir);
        }

        /// <summary>
        /// The cmb file kind_ selected index changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void CmbFileKind_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.UpdateView();
        }

        /// <summary>
        /// The tbx reference file path_ text changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void TbxReferenceFilePath_TextChanged(object sender, EventArgs e)
        {
            this.ttpTooltip.SetToolTip(this.tbxReferenceFilePath, this.tbxReferenceFilePath.Text);
            var cache = FileCacheManager.Instance.Get(this.tbxReferenceFilePath.Text, false);

            this.pnlTexture.BackgroundImage = (cache == null) ? null : cache.Icon;
        }

        #endregion
    }
}
