﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Runtime.InteropServices;
using System.Text;

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileTreeView
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Drawing;
    using System.Globalization;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;
    using System.DirectoryServices;

    using EffectMaker.UIControls.EffectBrowser.Controls.Basic;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb file tree view.
    /// </summary>
    public class EBFileTreeView : EBTreeView
    {
        #region Static Fields

        /// <summary>
        /// The icon key bookmark.
        /// </summary>
        public static readonly string IconKeyBookmark = ((int)Icon.Bookmark).ToString(CultureInfo.InvariantCulture);

        /// <summary>
        /// The icon key desktop.
        /// </summary>
        public static readonly string IconKeyDesktop = ((int)Icon.Desktop).ToString(CultureInfo.InvariantCulture);

        /// <summary>
        /// The icon key directory.
        /// </summary>
        public static readonly string IconKeyDirectory = ((int)Icon.Folder).ToString(CultureInfo.InvariantCulture);

        /// <summary>
        /// The icon key fav folder.
        /// </summary>
        public static readonly string IconKeyFavFolder = ((int)Icon.FavFolder).ToString(CultureInfo.InvariantCulture);

        /// <summary>
        /// The icon key my computer.
        /// </summary>
        public static readonly string IconKeyMyComputer = ((int)Icon.MyComputer).ToString(CultureInfo.InvariantCulture);

        /// <summary>
        /// The icon key my documents.
        /// </summary>
        public static readonly string IconKeyMyDocuments = ((int)Icon.MyDocuments).ToString(
            CultureInfo.InvariantCulture);

        /// <summary>
        /// The icon netowrk.
        /// </summary>
        public static readonly string IconKeyNetwork = ((int)Icon.Network).ToString(CultureInfo.InvariantCulture);

        /// <summary>
        /// The all image list.
        /// </summary>
        private static readonly ImageList AllImageList = new ImageList { ColorDepth = ColorDepth.Depth32Bit };

        /// <summary>
        /// The icon key max.
        /// </summary>
        private static readonly string IconKeyMax = ((int)Icon.Max).ToString(CultureInfo.InvariantCulture);

        #endregion

        #region Fields

        /// <summary>
        /// The directory.
        /// </summary>
        private string directory = string.Empty;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes static members of the <see cref="EBFileTreeView"/> class.
        /// </summary>
        static EBFileTreeView()
        {
            AllImageList.Images.Add(IconKeyDesktop, ImageUtility.GetShellIcon(ImageUtility.ShellIconIndices.DESKTOP));
            AllImageList.Images.Add(
                IconKeyMyDocuments,
                ImageUtility.GetShellIcon(ImageUtility.ShellIconIndices.DEFDOCUMENT));
            AllImageList.Images.Add(
                IconKeyMyComputer,
                ImageUtility.GetShellIcon(ImageUtility.ShellIconIndices.MYCOMPUTER));
            AllImageList.Images.Add(
                IconKeyDirectory,
                ImageUtility.GetShellIcon(ImageUtility.ShellIconIndices.FOLDEROPEN));
            AllImageList.Images.Add(IconKeyBookmark, Properties.Resources.Icon_Browser_Favorite);
            AllImageList.Images.Add(IconKeyFavFolder, Properties.Resources.Icon_Browser_FavoFolder);
            AllImageList.Images.Add(IconKeyNetwork, Properties.Resources.Icon_Browser_Network);

            foreach (var drivePath in Environment.GetLogicalDrives())
            {
                AllImageList.Images.Add(GetLogicalDriveIconKey(drivePath), ImageUtility.GetDirectoryIcon(drivePath));
            }
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="EBFileTreeView"/> class.
        /// </summary>
        public EBFileTreeView()
        {
            this.FullRowSelect = true;
            this.HideSelection = false;
            this.ShowLines = false;
            this.ImageList = AllImageList;
        }

        #endregion

        #region Public Events

        /// <summary>
        /// The directory changed.
        /// </summary>
        public event EventHandler DirectoryChanged;

        #endregion

        #region Enums

        /// <summary>
        /// The icon.
        /// </summary>
        private enum Icon
        {
            /// <summary>
            /// The desktop.
            /// </summary>
            Desktop,

            /// <summary>
            /// The my documents.
            /// </summary>
            MyDocuments,

            /// <summary>
            /// The my computer.
            /// </summary>
            MyComputer,

            /// <summary>
            /// The folder.
            /// </summary>
            Folder,

            /// <summary>
            /// The bookmark.
            /// </summary>
            Bookmark,

            /// <summary>
            /// The fav folder.
            /// </summary>
            FavFolder,

            /// <summary>
            /// The network.
            /// </summary>
            Network,

            /// <summary>
            /// The max.
            /// </summary>
            Max
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the directory.
        /// </summary>
        [Browsable(false)]
        public string Directory
        {
            get
            {
                return this.directory;
            }

            set
            {
                if (this.directory == value)
                {
                    return;
                }

                this.directory = value;

                if (this.DirectoryChanged != null)
                {
                    this.DirectoryChanged(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Gets or sets the selected node.
        /// </summary>
        [Browsable(false)]
        public new EBFileTreeNode SelectedNode
        {
            get
            {
                if (base.SelectedNode == null)
                {
                    return null;
                }

                System.Diagnostics.Debug.Assert(base.SelectedNode is EBFileTreeNode, "base.SelectedNode is EBFileTreeNode");
                return base.SelectedNode as EBFileTreeNode;
            }

            set
            {
                base.SelectedNode = value;
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The add node.
        /// </summary>
        /// <param name="parent">
        /// The parent.
        /// </param>
        /// <param name="name">
        /// The name.
        /// </param>
        /// <param name="path">
        /// The path.
        /// </param>
        /// <param name="iconKey">
        /// The icon key.
        /// </param>
        /// <param name="nodeKindType">
        /// The node kind type.
        /// </param>
        /// <returns>
        /// The <see cref="EBFileTreeNode"/>.
        /// </returns>
        public EBFileTreeNode AddNode(
            EBFileTreeNode parent,
            string name,
            string path,
            string iconKey,
            EBFileTreeNode.NodeKindType nodeKindType)
        {
            var newNode = this.CreateNode(name, path, iconKey, nodeKindType);

            if (parent != null)
            {
                parent.Nodes.Add(newNode);
            }
            else
            {
                this.Nodes.Add(newNode);
            }

            return newNode;
        }

        /// <summary>
        /// The expand current directory node.
        /// </summary>
        public virtual void ExpandCurrentDirectoryNode()
        {
        }

        /// <summary>
        /// The initialize.
        /// </summary>
        public virtual void Initialize()
        {
        }

        #endregion

        #region Methods

        /// <summary>
        /// The get logical drive icon key.
        /// </summary>
        /// <param name="drivePath">
        /// The drive path.
        /// </param>
        /// <returns>
        /// The <see cref="string"/>.
        /// </returns>
        protected static string GetLogicalDriveIconKey(string drivePath)
        {
            return IconKeyMax + drivePath;
        }

        /// <summary>
        /// The expand current directory node.
        /// </summary>
        /// <param name="targetPath">
        /// The target path.
        /// </param>
        /// <param name="targetNode">
        /// The target node.
        /// </param>
        /// <param name="dirs">
        /// The dirs.
        /// </param>
        /// <returns>
        /// The <see cref="EBFileTreeNode"/>.
        /// </returns>
        protected EBFileTreeNode ExpandCurrentDirectoryNode(
            string targetPath,
            EBFileTreeNode targetNode,
            IEnumerable<string> dirs)
        {
            foreach (var dir in dirs)
            {
                targetPath = Path.Combine(targetPath, dir);

                var node = targetNode.ChildNodes.FirstOrDefault(x => x.Filepath == targetPath)
                           ?? this.AddNode(
                               targetNode,
                               dir,
                               targetPath,
                               IconKeyDirectory,
                               EBFileTreeNode.NodeKindType.Directory);

                node.Expand();

                targetNode = node;
            }

            return targetNode;
        }

        /// <summary>
        /// The on after select.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnAfterSelect(TreeViewEventArgs e)
        {
            if (e.Node == null)
            {
                return;
            }

            System.Diagnostics.Debug.Assert(e.Node is EBFileTreeNode, "e.Node is EBFileTreeNode");

            var node = e.Node as EBFileTreeNode;

            if ((node.NodeKind == EBFileTreeNode.NodeKindType.Directory)
                || (node.NodeKind == EBFileTreeNode.NodeKindType.Bookmark))
            {
                this.Directory = node.Filepath;
            }
            else
            {
                // Directory = "*** Not Directory ***";
                this.Directory = string.Empty;
            }

            base.OnAfterSelect(e);
        }

        /// <summary>
        /// The on before expand.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnBeforeExpand(TreeViewCancelEventArgs e)
        {
            if ((e.Node is EBFileTreeNode) == false)
            {
                return;
            }

            var node = e.Node as EBFileTreeNode;

            if (node.IsNeedNodesUpdateOnTreeExpand == false)
            {
                return;
            }

            if (node.DisableNodesClearOnExpand)
            {
                return;
            }

            using (new UpdateBlock(this))
            {
                var roots = node.Nodes.OfType<EBFileTreeNode>().Where(x => x.IsRoot).ToArray();

                node.Nodes.Clear();

                // ReSharper disable once CoVariantArrayConversion
                node.Nodes.AddRange(roots);

                node.Nodes.AddRange(
                    PathUtility.EnumerateDirectories(node.Filepath)
                        .Select(
                            x =>
                            this.CreateNode(
                                Path.GetFileName(x),
                                x,
                                IconKeyDirectory,
                                EBFileTreeNode.NodeKindType.Directory))
                        .Cast<TreeNode>()
                        .ToArray());
            }

            base.OnBeforeExpand(e);
        }

        /// <summary>
        /// The on mouse down.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if ((e.Button == MouseButtons.Left) || (e.Button == MouseButtons.Right))
            {
                var newSelectedNode = this.GetNodeAt(new Point(e.X, e.Y)) as EBFileTreeNode;

                if (newSelectedNode != null)
                {
                    using (new Foundation.Debugging.Profiling.ProfileTimer("TreeNode selection."))
                    {
                        this.SelectedNode = newSelectedNode;
                    }
                }
            }

            base.OnMouseDown(e);
        }

        /// <summary>
        /// The create node.
        /// </summary>
        /// <param name="name">
        /// The name.
        /// </param>
        /// <param name="path">
        /// The path.
        /// </param>
        /// <param name="iconKey">
        /// The icon key.
        /// </param>
        /// <param name="nodeKindType">
        /// The node kind type.
        /// </param>
        /// <returns>
        /// The <see cref="EBFileTreeNode"/>.
        /// </returns>
        private EBFileTreeNode CreateNode(
            string name,
            string path,
            string iconKey,
            EBFileTreeNode.NodeKindType nodeKindType)
        {
            var newNode = new EBFileTreeNode(nodeKindType)
                              {
                                  Text = name,
                                  Filepath = path,
                                  ImageKey = iconKey,
                                  SelectedImageKey = iconKey
                              };

            // アクセス制限があるディレクトリに触ったら例外になるため
            try
            {
                // サブディレクトリを持っているか
                // 持っているなら [+]を表示
                if (PathUtility.HasSubDirectory(path))
                {
                    newNode.Nodes.Add(new EBFileTreeNode(EBFileTreeNode.NodeKindType.ExpanderDummy));
                }
            }

                // ReSharper disable once EmptyGeneralCatchClause
            catch
            {
            }

            return newNode;
        }

        #endregion
    }
}
