﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileListView
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.IO;
    using System.Windows.Forms;

    using EffectMaker.Foundation.FileMonitor;
    using EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base;
    using EffectMaker.UIControls.EffectBrowser.Data;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    using FileInfo = EffectMaker.UIControls.EffectBrowser.Data.FileInfo;

    /// <summary>
    /// The eb directory file list view.
    /// </summary>
    public class EBDirectoryFileListView : EBFileListView
    {
        #region Fields

        /// <summary>
        /// The monitor manager.
        /// </summary>
        private readonly FileMonitorManager monitorManager = new FileMonitorManager(Application.OpenForms["MainForm"]);

        /// <summary>
        /// The directory.
        /// </summary>
        private string directory = string.Empty;

        #endregion

        #region Public Events

        /// <summary>
        /// The directory changed.
        /// </summary>
        public event EventHandler DirectoryChanged;

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the directory.
        /// </summary>
        [Browsable(false)]
        public string Directory
        {
            get
            {
                return this.directory;
            }

            set
            {
                if (this.directory == value)
                {
                    return;
                }

                if (string.IsNullOrEmpty(this.directory) == false)
                {
                    this.monitorManager.UnregisterMonitor(this.directory, true);
                }

                this.directory = value;

                if (PathUtility.DirectoryExists(this.directory))
                {
                    this.monitorManager.RegisterMonitor(this.directory, null, this.OnFileChanged, true);
                }

                if (ControlEventSuppressBlock.Enabled == false)
                {
                    if (this.DirectoryChanged != null)
                    {
                        this.DirectoryChanged(this, EventArgs.Empty);
                    }
                }

                this.SelectedIndices.Clear();
                this.RequestUpdate();
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// リストビューで管理するファイルリストを取得します。
        /// 画面にはこのファイルリストをソートしてフィルタリングした結果が表示されます。
        /// </summary>
        /// <exception cref="NotSupportedException">NotSupportedException</exception>
        protected override IEnumerable<FileInfo> Files
        {
            get
            {
                return FileUtility.MakeFileInfosFromDirectory(this.Directory, true, false);
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on double click.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnDoubleClick(EventArgs e)
        {
            base.OnDoubleClick(e);

            if (this.SelectedFile != null)
            {
                if (this.SelectedFile.FileKind == FileKindType.Folder)
                {
                    this.Directory = this.SelectedFile.FileFullPath;
                }
            }
        }

        /// <summary>
        /// The on key down.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Return:
                    if (this.SelectedFile != null)
                    {
                        if (this.SelectedFile.FileKind == FileKindType.Folder)
                        {
                            this.Directory = this.SelectedFile.FileFullPath;
                            e.Handled = true;
                        }
                    }

                    break;

                case Keys.Back:
                    var fi = new System.IO.FileInfo(Path.Combine(this.Directory, ".."));
                    this.Directory = fi.FullName;
                    e.Handled = true;

                    break;
            }

            base.OnKeyDown(e);
        }

        /// <summary>
        /// The on file changed.
        /// </summary>
        /// <param name="path">
        /// The path.
        /// </param>
        /// <param name="userData">
        /// The user data.
        /// </param>
        /// <param name="changeType">
        /// The change type.
        /// </param>
        private void OnFileChanged(string path, object userData, WatcherChangeTypes changeType)
        {
            this.RequestUpdate();
        }

        #endregion
    }
}
