﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Runtime.InteropServices;
    using System.Text.RegularExpressions;

    using EffectMaker.Foundation.Utility;
    using EffectMaker.UIControls.EffectBrowser.Data;

    /// <summary>
    /// The eb file list view.
    /// </summary>
    public partial class EBFileListView
    {
        #region Fields

        /// <summary>
        /// The throttle executer.
        /// </summary>
        private readonly ThrottleExecuter throttleExecuter = new ThrottleExecuter();

        /// <summary>
        /// The matchers.
        /// </summary>
        private IMatcher[] matchers;

        /// <summary>
        /// The search words.
        /// </summary>
        private string searchWords = string.Empty;

        #endregion

        #region Interfaces

        /// <summary>
        /// The Matcher interface.
        /// </summary>
        private interface IMatcher
        {
            #region Public Methods and Operators

            /// <summary>
            /// The is match.
            /// </summary>
            /// <param name="src">
            /// The src.
            /// </param>
            /// <returns>
            /// The <see cref="bool"/>.
            /// </returns>
            bool IsMatch(string src);

            /// <summary>
            /// The match regions.
            /// </summary>
            /// <param name="src">
            /// The src.
            /// </param>
            /// <returns>
            /// The <see cref="IEnumerable"/>.
            /// </returns>
            IEnumerable<MatchRegionInternal> MatchRegions(string src);

            #endregion
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets a value indicating whether is filtering.
        /// </summary>
        public bool IsFiltering
        {
            get
            {
                return this.searchWords.Any();
            }
        }

        /// <summary>
        /// Gets the search words.
        /// </summary>
        public string SearchWords
        {
            get
            {
                return this.searchWords;
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The set search word.
        /// </summary>
        /// <param name="searchWords">
        /// The search words.
        /// </param>
        public void SetSearchWord(string searchWords)
        {
            searchWords = searchWords.Trim();

            if (searchWords == this.searchWords)
            {
                return;
            }

            this.searchWords = searchWords;

            // 検索結果の反映を遅延実行する
            // ThrottleExecuterに指定した処理はUIスレッドで実行される
            this.throttleExecuter.Execute(150, () =>
            {
                this.matchers = MakeMatchers(this.searchWords);
                this.RunUpdate(UpdateStateKind.Search);
            });
        }

        #endregion

        #region Methods

        /// <summary>
        /// The make matchers.
        /// </summary>
        /// <param name="searchWords">
        /// The search words.
        /// </param>
        /// <returns>
        /// The <see cref="IMatcher[]"/>.
        /// </returns>
        private static IMatcher[] MakeMatchers(string searchWords)
        {
            string[] splitWords = searchWords.Split(new [] { ' ', '　' }, StringSplitOptions.RemoveEmptyEntries);

            IMatcher[] matchers = new IMatcher[splitWords.Length];

            for (int i = 0; i < splitWords.Length; ++i)
            {
                string word = splitWords[i];

                if (word.IndexOfAny(new[] { '*', '?' }) != -1)
                {
                    matchers[i] = new WildcardMatcher(word);
                }
                else if (word[0] == '/')
                {
                    matchers[i] = new RegexMatcher(word);
                }
                else
                {
                    matchers[i] = new StringMatcher(word);
                }
            }

            return matchers;
        }

        /// <summary>
        /// 検索一致領域を作る
        /// </summary>
        /// <param name="src">もと文字列</param>
        /// <returns>マッチした領域情報</returns>
        private IEnumerable<MatchRegion> MakeMatchRegions(string src)
        {
            if (string.IsNullOrEmpty(src) || (this.matchers == null))
            {
                yield break;
            }

            var any = false;
            var matches = new bool[src.Length];
            {
                foreach (var region in this.matchers.SelectMany(match => match.MatchRegions(src)))
                {
                    for (var i = region.StartIndex; i != (region.StartIndex + region.Length); ++i)
                    {
                        if (i >= matches.Length)
                        {
                            break;
                        }

                        matches[i] = true;
                        any = true;
                    }
                }
            }

            if (any == false)
            {
                yield break;
            }

            var index = 0;
            do
            {
                var endIndex = index;

                for (; endIndex != src.Length; ++endIndex)
                {
                    if (matches[index] != matches[endIndex])
                    {
                        break;
                    }
                }

                yield return new MatchRegion { IsMatch = matches[index], Word = src.Substring(index, endIndex - index) };

                index = endIndex;

                if (index == src.Length)
                {
                    break;
                }
            }
            while (true);
        }

        /// <summary>
        /// The search by current word.
        /// </summary>
        /// <param name="sources">
        /// The sources.
        /// </param>
        /// <returns>
        /// The <see cref="IEnumerable"/>.
        /// </returns>
        private IEnumerable<FileInfo> SearchByCurrentWord(IEnumerable<FileInfo> sources)
        {
            return string.IsNullOrEmpty(this.searchWords)
                       ? sources
                       : sources.Where(
                           src =>
                           this.matchers.All(x => x.IsMatch(src.FileName))
                           || this.matchers.Any(x => x.IsMatch(src.Comment)));
        }

        #endregion

        /// <summary>
        /// The match region.
        /// </summary>
        public class MatchRegion
        {
            #region Public Properties

            /// <summary>
            /// Gets or sets a value indicating whether is match.
            /// </summary>
            public bool IsMatch { get; set; }

            /// <summary>
            /// Gets or sets the word.
            /// </summary>
            public string Word { get; set; }

            #endregion
        }

        /// <summary>
        /// The match region internal.
        /// </summary>
        private class MatchRegionInternal
        {
            #region Public Properties

            /// <summary>
            /// Gets or sets the length.
            /// </summary>
            public int Length { get; set; }

            /// <summary>
            /// Gets or sets the start index.
            /// </summary>
            public int StartIndex { get; set; }

            #endregion
        }

        /// <summary>
        /// 正規表現検索
        /// </summary>
        private class RegexMatcher : IMatcher
        {
            #region Fields

            /// <summary>
            /// The pattern.
            /// </summary>
            private readonly Regex pattern;

            #endregion

            #region Constructors and Destructors

            /// <summary>
            /// Initializes a new instance of the <see cref="RegexMatcher"/> class.
            /// </summary>
            /// <param name="pattern">
            /// The pattern.
            /// </param>
            public RegexMatcher(string pattern)
            {
                System.Diagnostics.Debug.Assert(pattern[0] == '/', "pattern[0] == '/'");

                try
                {
                    this.pattern = new Regex(pattern.Substring(1), RegexOptions.IgnoreCase);
                }

                    // ReSharper disable once EmptyGeneralCatchClause
                catch
                {
                }
            }

            #endregion

            #region Public Methods and Operators

            /// <summary>
            /// The is match.
            /// </summary>
            /// <param name="src">
            /// The src.
            /// </param>
            /// <returns>
            /// The <see cref="bool"/>.
            /// </returns>
            public bool IsMatch(string src)
            {
                if (string.IsNullOrEmpty(src))
                {
                    return false;
                }

                return (this.pattern != null) && this.pattern.IsMatch(src);
            }

            /// <summary>
            /// The match regions.
            /// </summary>
            /// <param name="src">
            /// The src.
            /// </param>
            /// <returns>
            /// The <see cref="IEnumerable"/>.
            /// </returns>
            public IEnumerable<MatchRegionInternal> MatchRegions(string src)
            {
                if (string.IsNullOrEmpty(src))
                {
                    return new MatchRegionInternal[0];
                }

                if (this.pattern != null)
                {
                    return
                        this.pattern.Matches(src)
                            .Cast<Match>()
                            .Select(x => new MatchRegionInternal { StartIndex = x.Index, Length = x.Length });
                }

                return new MatchRegionInternal[0];
            }

            #endregion
        }

        /// <summary>
        /// 文字列検索
        /// </summary>
        private class StringMatcher : IMatcher
        {
            #region Fields

            /// <summary>
            /// The pattern.
            /// </summary>
            private readonly string pattern;

            #endregion

            #region Constructors and Destructors

            /// <summary>
            /// Initializes a new instance of the <see cref="StringMatcher"/> class.
            /// </summary>
            /// <param name="pattern">
            /// The pattern.
            /// </param>
            public StringMatcher(string pattern)
            {
                this.pattern = pattern;
            }

            #endregion

            #region Public Methods and Operators

            /// <summary>
            /// The is match.
            /// </summary>
            /// <param name="src">
            /// The src.
            /// </param>
            /// <returns>
            /// The <see cref="bool"/>.
            /// </returns>
            public bool IsMatch(string src)
            {
                if (string.IsNullOrEmpty(src))
                {
                    return false;
                }

                return src.IndexOf(this.pattern, StringComparison.OrdinalIgnoreCase) != -1;
            }

            /// <summary>
            /// The match regions.
            /// </summary>
            /// <param name="src">
            /// The src.
            /// </param>
            /// <returns>
            /// The <see cref="IEnumerable"/>.
            /// </returns>
            public IEnumerable<MatchRegionInternal> MatchRegions(string src)
            {
                if (string.IsNullOrEmpty(src))
                {
                }
                else
                {
                    var startIndex = 0;

                    do
                    {
                        var index = src.IndexOf(this.pattern, startIndex, StringComparison.OrdinalIgnoreCase);
                        if (index == -1)
                        {
                            break;
                        }

                        yield return new MatchRegionInternal { StartIndex = index, Length = this.pattern.Length };

                        startIndex = index + this.pattern.Length;
                    }
                    while (true);
                }
            }

            #endregion
        }

        /// <summary>
        /// ワイルドカード検索
        /// </summary>
        private class WildcardMatcher : IMatcher
        {
            #region Fields

            /// <summary>
            /// The pattern.
            /// </summary>
            private readonly string pattern;

            #endregion

            #region Constructors and Destructors

            /// <summary>
            /// Initializes a new instance of the <see cref="WildcardMatcher"/> class.
            /// </summary>
            /// <param name="pattern">
            /// The pattern.
            /// </param>
            public WildcardMatcher(string pattern)
            {
                this.pattern = pattern;
            }

            #endregion

            #region Public Methods and Operators

            /// <summary>
            /// The is match.
            /// </summary>
            /// <param name="src">
            /// The src.
            /// </param>
            /// <returns>
            /// The <see cref="bool"/>.
            /// </returns>
            public bool IsMatch(string src)
            {
                if (string.IsNullOrEmpty(src))
                {
                    return false;
                }

                return PathMatchSpec(src, this.pattern);
            }

            /// <summary>
            /// The match regions.
            /// </summary>
            /// <param name="src">
            /// The src.
            /// </param>
            /// <returns>
            /// The <see cref="IEnumerable"/>.
            /// </returns>
            public IEnumerable<MatchRegionInternal> MatchRegions(string src)
            {
                if (string.IsNullOrEmpty(src))
                {
                }
                else
                {
                    if (this.IsMatch(src))
                    {
                        yield return new MatchRegionInternal { StartIndex = 0, Length = src.Length };
                    }
                }
            }

            #endregion

            #region Methods

            /// <summary>
            /// The path match spec.
            /// </summary>
            /// <param name="fileParam">
            /// The file param.
            /// </param>
            /// <param name="spec">
            /// The spec.
            /// </param>
            /// <returns>
            /// The <see cref="bool"/>.
            /// </returns>
            [DllImport("shlwapi.dll", CharSet = CharSet.Auto)]
            private static extern bool PathMatchSpec([In] string fileParam, [In] string spec);

            #endregion
        }
    }
}
