﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base
{
    using System;
    using System.Windows.Forms;

    using EffectMaker.UIControls.EffectBrowser.Controls.Basic;
    using EffectMaker.UIControls.EffectBrowser.Controls.Dialog;
    using EffectMaker.UIControls.EffectBrowser.Data;
    using EffectMaker.UIControls.EffectBrowser.Exceptions;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb file list view.
    /// </summary>
    public partial class EBFileListView
    {
        #region Methods

        /// <summary>
        /// The on h scroll.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnHScroll(EventArgs e)
        {
            this.Focus();
            this.Invalidate();
        }

        /// <summary>
        /// The on mouse wheel.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            this.Focus();
            this.Invalidate();
        }

        /// <summary>
        /// The on v scroll.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnVScroll(EventArgs e)
        {
            this.Focus();
            this.Invalidate();
        }

        /// <summary>
        /// The initialize comment editing.
        /// </summary>
        /// <param name="subItem">
        /// The sub item.
        /// </param>
        /// <param name="fileInfo">
        /// The file info.
        /// </param>
        private void InitializeCommentEditing(ListViewItem.ListViewSubItem subItem, FileInfo fileInfo)
        {
            var control = new EBTextBox { AutoSize = false, Bounds = subItem.Bounds, Text = fileInfo.Comment };

            control.GotFocus += (s, e) => control.SelectAll();

            control.Leave += (s, e) =>
            {
                this.Controls.Remove(control);

                if (fileInfo.Comment != control.Text)
                {
                    var oldComment = fileInfo.Comment;

                    try
                    {
                        using (new WaitCursorBlock(this))
                        {
                            fileInfo.Comment = control.Text;
                            fileInfo.SaveToFile(true);
                            this.Invalidate();
                        }
                    }
                    catch (InvalidEsetException ex)
                    {
                        using (
                            var dialog = new MultipleLineDialog
                                                {
                                                    Message =
                                                        Properties.Resources.Exception_InvalidEset,
                                                    Title =
                                                        Properties.Resources.Dialog_CantWriteFile,
                                                    Lines = ex.Filenames
                                                })
                        {
                            dialog.ShowDialog();
                        }

                        fileInfo.Comment = oldComment;
                    }
                }
            };

            control.KeyDown += (s, e) =>
            {
                if (e.KeyData == Keys.Enter)
                {
                    this.Focus();
                }
            };

            this.Controls.Add(control);
            control.Focus();
        }

        /// <summary>
        /// The initialize label color editing.
        /// </summary>
        /// <param name="subItem">
        /// The sub item.
        /// </param>
        /// <param name="fileInfo">
        /// The file info.
        /// </param>
        private void InitializeLabelColorEditing(ListViewItem.ListViewSubItem subItem, FileInfo fileInfo)
        {
            var control = new EBLabelColorComboBox
                              {
                                  AutoSize = false,
                                  Bounds = subItem.Bounds,
                                  SelectedIndex = (int)fileInfo.LabelColor
                              };

            control.Leave += (s, e) =>
            {
                this.Controls.Remove(control);

                if (fileInfo.LabelColor != (LabelColorType)control.SelectedIndex)
                {
                    var oldLabelColor = fileInfo.LabelColor;

                    try
                    {
                        using (new WaitCursorBlock(this))
                        {
                            fileInfo.LabelColor = (LabelColorType)control.SelectedIndex;
                            fileInfo.SaveToFile(true);
                            this.Invalidate();
                        }
                    }
                    catch (InvalidEsetException ex)
                    {
                        using (
                            var dialog = new MultipleLineDialog
                                                {
                                                    Message =
                                                        Properties.Resources.Exception_InvalidEset,
                                                    Title =
                                                        Properties.Resources.Dialog_CantWriteFile,
                                                    Lines = ex.Filenames
                                                })
                        {
                            dialog.ShowDialog();
                        }

                        fileInfo.LabelColor = oldLabelColor;
                    }
                }
            };

            control.KeyDown += (s, e) =>
            {
                if (e.KeyData == Keys.Enter)
                {
                    this.Focus();
                }
            };

            this.Controls.Add(control);
            control.Focus();
        }

        /// <summary>
        /// The start inline editing.
        /// </summary>
        private void StartInlineEditing()
        {
            if (this.LastMouseButton != MouseButtons.Left)
            {
                return;
            }

            if (this.HandleToolTipForBlockingEdit())
            {
                return;
            }

            if (this.CanEdit == false)
            {
                return;
            }

            var pos = this.PointToClient(Cursor.Position);

            var info = this.HitTest(pos.X, pos.Y);
            if ((info.Item == null) || (info.SubItem == null))
            {
                return;
            }

            System.Diagnostics.Debug.Assert(info.Item.Tag is FileInfo, "info.Item.Tag is FileInfo");
            var fileInfo = (FileInfo)info.Item.Tag;

            if (fileInfo.IsEditable == false)
            {
                return;
            }

            if (this.lastMouseDownFileInfo != fileInfo)
            {
                return;
            }

            System.Diagnostics.Debug.Assert(info.SubItem.Tag is ColumnKinds, "info.SubItem.Tag is ColumnKinds");
            switch ((ColumnKinds)info.SubItem.Tag)
            {
                case ColumnKinds.LabelColor:
                    this.InitializeLabelColorEditing(info.SubItem, fileInfo);
                    break;

                case ColumnKinds.Comment:
                    this.InitializeCommentEditing(info.SubItem, fileInfo);
                    break;
            }
        }

        #endregion
    }
}
