﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base
{
    using System;
    using System.Diagnostics.CodeAnalysis;
    using System.Drawing;
    using System.Drawing.Drawing2D;
    using System.Linq;
    using System.Windows.Forms;

    using EffectMaker.Foundation.Utility;
    using EffectMaker.UIControls.EffectBrowser.Data;

    /// <summary>
    /// The eb file list view.
    /// </summary>
    public partial class EBFileListView
    {
        #region Constants

        /// <summary>
        /// The left flag.
        /// </summary>
        private const TextFormatFlags LeftFlag = TextFormatFlags.Left | TextFormatFlags.VerticalCenter;

        /// <summary>
        /// The right flag.
        /// </summary>
        private const TextFormatFlags RightFlag = TextFormatFlags.Right | TextFormatFlags.VerticalCenter;

        #endregion

        #region Fields

        /// <summary>
        /// The with expander.
        /// </summary>
        private bool withExpander = true;

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets a value indicating whether with expander.
        /// </summary>
        public bool WithExpander
        {
            get
            {
                return this.withExpander;
            }

            set
            {
                if (this.withExpander == value)
                {
                    return;
                }

                this.withExpander = value;
                this.Invalidate();
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on draw column header.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnDrawColumnHeader(DrawListViewColumnHeaderEventArgs e)
        {
            e.DrawDefault = true;

            base.OnDrawColumnHeader(e);
        }

        /// <summary>
        /// The on draw sub item.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        /// <exception cref="NotSupportedException">
        /// NotSupportedException
        /// </exception>
        protected override void OnDrawSubItem(DrawListViewSubItemEventArgs e)
        {
            System.Diagnostics.Debug.Assert(e.Item.Tag is FileInfo, "e.Item.Tag is FileInfo");
            var info = e.Item.Tag as FileInfo;
            var kind = (ColumnKinds)this.Columns[e.ColumnIndex].Tag;

            var drawInfo = new DrawInfo { Bounds = e.Bounds };

            if ((kind == ColumnKinds.Name) && (info.LabelColor != LabelColorType.Color0))
            {
                drawInfo.TextColor = Color.Black;
            }
            else
            {
                if (e.Item.Selected)
                {
                    drawInfo.TextColor = this.Focused ?
                        SystemColors.HighlightText : SystemColors.ControlText;
                }
                else
                {
                    drawInfo.TextColor = (kind == ColumnKinds.Name) ?
                        Color.Black : Color.Gray;
                }
            }

            {
                // 領域補正
                // 列０は順序変更された時でも左位置が０のままになってしまう
                if (e.ColumnIndex == 0)
                {
                    drawInfo.Bounds.X = this.column0Left;
                }
            }

            if (this.GridLines)
            {
                drawInfo.Bounds.Height -= 1;
            }

            // 選択背景
            Brush tmpBrush = e.Item.Selected ?
                this.Focused ? SystemBrushes.Highlight : new SolidBrush(Color.FromArgb(240, 240, 240)) :
                new SolidBrush(Color.White);

            e.Graphics.FillRectangle(tmpBrush, drawInfo.Bounds);

            if (!e.Item.Selected || !this.Focused)
            {
                tmpBrush.Dispose();
            }

            // テキスト色
            switch (kind)
            {
                case ColumnKinds.Name:
                    this.DrawName(e, drawInfo, info, this.WithExpander);
                    break;

                case ColumnKinds.UpdateTimestamp:
                    this.DrawUpdateTimestamp(e, drawInfo, info);
                    break;

                case ColumnKinds.CreateTimestamp:
                    this.DrawCreateTimestamp(e, drawInfo, info);
                    break;

                case ColumnKinds.FileKind:
                    this.DrawFileKind(e, drawInfo, info);
                    break;

                case ColumnKinds.ByteSize:
                    this.DrawByteSize(e, drawInfo, info);
                    break;

                case ColumnKinds.LabelColor:
                    this.DrawLabelColor(e, drawInfo, info);
                    break;

                case ColumnKinds.Comment:
                    this.DrawComment(e, drawInfo, info);
                    break;

                case ColumnKinds.FileFullPath:
                    this.DrawPath(e, drawInfo, info);
                    break;

                default:
                    throw new NotSupportedException();
            }

            base.OnDrawSubItem(e);
        }

        /// <summary>
        /// The draw icon.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        /// <param name="bounds">
        /// The bounds.
        /// </param>
        /// <param name="icon">
        /// The icon.
        /// </param>
        /// <param name="x">
        /// The x.
        /// </param>
        private static void DrawIcon(DrawListViewSubItemEventArgs e, Rectangle bounds, Bitmap icon, int x)
        {
            var verticalCenter = (bounds.Height - icon.Height) / 2;
            var srcRect = new Rectangle(0, 0, icon.Width, icon.Height);
            var dstRect = new Rectangle(bounds.X + x, bounds.Y + verticalCenter, icon.Width, icon.Height);

            if (dstRect.Right > bounds.Width)
            {
                var diff = dstRect.Right - bounds.Right;

                srcRect.Width -= diff;
                dstRect.Width -= diff;
            }

            e.Graphics.DrawImage(icon, dstRect, srcRect, GraphicsUnit.Pixel);
        }

        /// <summary>
        /// The draw byte size.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        /// <param name="drawInfo">
        /// The draw info.
        /// </param>
        /// <param name="info">
        /// The info.
        /// </param>
        private void DrawByteSize(DrawListViewSubItemEventArgs e, DrawInfo drawInfo, FileInfo info)
        {
            if (info.FileKind != FileKindType.Folder)
            {
                TextRenderer.DrawText(
                    e.Graphics,
                    info.DisplayByteSize,
                    this.Font,
                    drawInfo.Bounds,
                    drawInfo.TextColor,
                    RightFlag);
            }
        }

        /// <summary>
        /// 大きいアイコンモード時のアイテムを描画します。
        /// </summary>
        /// <param name="e">描画するアイテムの情報</param>
        private void DrawLargeItem(DrawListViewItemEventArgs e)
        {
            var info = e.Item.Tag as FileInfo;
            System.Diagnostics.Debug.Assert(info != null, "e.Item.Tag must be FileInfo");

            // 折り返しを考慮したテキスト描画エリアを計算
            var textArea = this.MeasureTextInArea(
                e.Graphics,
                info.DisplayFileName,
                this.Font,
                new Size(Constants.LargeIconSize + 12, Constants.LargeIconSize),
                new Point());

            // 選択状態に応じて色を選択して背景色を描画
            var tmpBrush = e.Item.Selected ? this.Focused ? SystemBrushes.Highlight :
                new SolidBrush(Color.FromArgb(240, 240, 240)) :
                new SolidBrush(Color.White);
            var itemArea = new Rectangle(e.Bounds.X, e.Bounds.Y, e.Bounds.Width, Constants.LargeIconSize + textArea.Height + 2);
            e.Graphics.FillRectangle(tmpBrush, itemArea);

            if (tmpBrush != SystemBrushes.Highlight)
            {
                tmpBrush.Dispose();
            }

            // 画像をCenterTopで描画
            var imageBounds = e.Bounds;
            imageBounds.X += (imageBounds.Width - Constants.LargeIconSize) / 2;
            imageBounds.Width = imageBounds.Height = Constants.LargeIconSize;
            e.Graphics.DrawImage(info.LargeIcon, imageBounds);

            // 計算したエリアを配置する領域を再計算
            var drawingArea = new Rectangle(
                e.Bounds.X + ((e.Bounds.Width - Math.Min(Constants.LargeIconSize + 12, textArea.Width)) / 2),
                e.Bounds.Y + Constants.LargeIconSize,
                Constants.LargeIconSize + 12,
                Constants.LargeIconSize);

            // 折り返しこみでテキストを描画
            {
                var regions = this.MakeMatchRegions(info.DisplayFileName).ToArray();

                var textColor = e.Item.Selected ? this.Focused ? SystemColors.HighlightText : SystemColors.ControlText : Color.Black;

                if (regions.Any())
                {
                    Point startPos = drawingArea.Location;

                    foreach (var region in regions)
                    {
                        startPos = this.DrawTextInArea(e.Graphics, region.Word, this.Font, drawingArea, startPos, region.IsMatch ? Color.Red : textColor);
                    }
                }
                else
                {
                    this.DrawTextInArea(e.Graphics, info.DisplayFileName, this.Font, drawingArea, drawingArea.Location, textColor);
                }
            }
        }

        /// <summary>
        /// 文字列の描画領域を計算します。
        /// </summary>
        /// <param name="dc">デバイスコンテキスト</param>
        /// <param name="text">文字列</param>
        /// <param name="font">フォント</param>
        /// <param name="drawingArea">描画領域</param>
        /// <param name="startPos">文字の描画開始位置(左上)</param>
        /// <returns>文字列の描画領域を返します。</returns>
        private Size MeasureTextInArea(IDeviceContext dc, string text, Font font, Size drawingArea, Point startPos)
        {
            Point currPos = startPos;

            int maxX = 0;

            foreach (var str in text.ToCharArray())
            {
                Size strSize = TextRenderer.MeasureText(dc, str.ToString(), this.Font, new Size(int.MaxValue, int.MinValue), TextFormatFlags.NoPadding);

                if (strSize.Width > drawingArea.Width)
                {
                    continue;
                }

                // 描画領域から出そうになった場合は、改行
                if (currPos.X + strSize.Width > drawingArea.Width)
                {
                    currPos.X = 0;
                    currPos.Y += (int)font.GetHeight();
                }

                if (currPos.Y >= drawingArea.Height)
                {
                    break;
                }

                currPos.X += strSize.Width;

                maxX = Math.Max(maxX, currPos.X);
            }

            Size result = new Size(
                maxX,
                Math.Min(currPos.Y + (int)font.GetHeight(), drawingArea.Height));

            return result;
        }

        /// <summary>
        /// 指定した領域内に、文字列を描画します。
        /// </summary>
        /// <param name="dc">デバイスコンテキスト</param>
        /// <param name="text">文字列</param>
        /// <param name="font">フォント</param>
        /// <param name="drawingArea">描画領域</param>
        /// <param name="startPos">文字の描画開始位置(左上)</param>
        /// <param name="foreColor">文字色</param>
        /// <returns>最後の文字の描画位置(右上)を返します。</returns>
        private Point DrawTextInArea(IDeviceContext dc, string text, Font font, Rectangle drawingArea, Point startPos, Color foreColor)
        {
            Point currPos = startPos;

            foreach (var str in text.ToCharArray())
            {
                Size strSize = TextRenderer.MeasureText(dc, str.ToString(), this.Font, new Size(int.MaxValue, int.MinValue), TextFormatFlags.NoPadding);

                if (strSize.Width > drawingArea.Width)
                {
                    continue;
                }

                // 描画領域から出そうになった場合は、改行
                if (currPos.X + strSize.Width > drawingArea.X + drawingArea.Width)
                {
                    currPos.X = drawingArea.X;
                    currPos.Y += (int)font.GetHeight();
                }

                Rectangle strArea = new Rectangle(currPos, strSize);
                strArea.Height = Math.Min(strSize.Height, drawingArea.Y + drawingArea.Height - currPos.Y);

                if (strArea.Height <= 0)
                {
                    break;
                }

                TextRenderer.DrawText(dc, str.ToString(), this.Font, strArea, foreColor, TextFormatFlags.NoPadding);

                currPos.X += strSize.Width;
            }

            return currPos;
        }

        /// <summary>
        /// The draw comment.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        /// <param name="drawInfo">
        /// The draw info.
        /// </param>
        /// <param name="info">
        /// The info.
        /// </param>
        private void DrawComment(DrawListViewSubItemEventArgs e, DrawInfo drawInfo, FileInfo info)
        {
            if (!string.IsNullOrEmpty(info.DisplayComment))
            {
                this.DrawSearchString(drawInfo.Bounds, info.DisplayComment.Replace("\r\n", " ").Replace("\r", " ").Replace("\n", " "), e.Graphics, drawInfo.TextColor);
            }
        }

        /// <summary>
        /// The draw create timestamp.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        /// <param name="drawInfo">
        /// The draw info.
        /// </param>
        /// <param name="info">
        /// The info.
        /// </param>
        private void DrawCreateTimestamp(DrawListViewSubItemEventArgs e, DrawInfo drawInfo, FileInfo info)
        {
            TextRenderer.DrawText(
                e.Graphics,
                info.DisplayCreateTimestamp,
                this.Font,
                drawInfo.Bounds,
                drawInfo.TextColor,
                LeftFlag);
        }

        /// <summary>
        /// The draw file kind.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        /// <param name="drawInfo">
        /// The draw info.
        /// </param>
        /// <param name="info">
        /// The info.
        /// </param>
        private void DrawFileKind(DrawListViewSubItemEventArgs e, DrawInfo drawInfo, FileInfo info)
        {
            TextRenderer.DrawText(
                e.Graphics,
                info.DisplayFileKind,
                this.Font,
                drawInfo.Bounds,
                drawInfo.TextColor,
                LeftFlag);
        }

        /// <summary>
        /// The draw label color.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        /// <param name="drawInfo">
        /// The draw info.
        /// </param>
        /// <param name="info">
        /// The info.
        /// </param>
        private void DrawLabelColor(DrawListViewSubItemEventArgs e, DrawInfo drawInfo, FileInfo info)
        {
            // カラー０は描かない
            if (info.LabelColor == LabelColorType.Color0)
            {
                return;
            }

            var x = 0;
            {
                // アイコン
                var icon = Constants.LabelColorIcons[info.LabelColor];

                DrawIcon(e, drawInfo.Bounds, icon, x);

                x += icon.Width;
                x += Constants.IconMargin;
            }

            {
                // 名前
                var nameBounds = new Rectangle(
                    drawInfo.Bounds.X + x,
                    drawInfo.Bounds.Y,
                    drawInfo.Bounds.Width - x,
                    drawInfo.Bounds.Height);
                TextRenderer.DrawText(
                    e.Graphics,
                    info.DisplayLabelColor,
                    this.Font,
                    nameBounds,
                    drawInfo.TextColor,
                    LeftFlag);
            }
        }

        /// <summary>
        /// The draw name.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        /// <param name="drawInfo">
        /// The draw info.
        /// </param>
        /// <param name="info">
        /// The info.
        /// </param>
        /// <param name="withExpander">
        /// The with expander.
        /// </param>
        private void DrawName(DrawListViewSubItemEventArgs e, DrawInfo drawInfo, FileInfo info, bool withExpander)
        {
            // ラベルカラー
            if (info.LabelColor != LabelColorType.Color0)
            {
                var old = e.Graphics.SmoothingMode;
                {
                    e.Graphics.SmoothingMode = SmoothingMode.AntiAlias;

                    var icon = Constants.ChildExpander[info.IsOpend];

                    var x = info.Depth * Constants.IconSize;
                    x += icon.Width;
                    x += Constants.ExpanderMargin;

                    var rect = new RectangleF(
                        drawInfo.Bounds.X + x,
                        drawInfo.Bounds.Y,
                        drawInfo.Bounds.Width - x,
                        drawInfo.Bounds.Height);
                    rect.X -= Constants.ExpanderMargin;
                    rect.Y += 0;
                    rect.Height -= 1;

                    // セル幅に合わせて背景を塗りつぶす
                    using (var path = RenderUtility.MakeRoundRectangleGraphicsPath(rect, 10.0f))
                    using (var brush = new SolidBrush(Constants.LabelColorColors[info.LabelColor]))
                    {
                        e.Graphics.FillPath(brush, path);
                    }
                }

                e.Graphics.SmoothingMode = old;
            }

            {
                var x = 1 + (info.Depth * Constants.IconSize);

                // ノード開閉
                if (withExpander)
                {
                    var icon = Constants.ChildExpander[info.IsOpend];

                    if (info.HasChildren)
                    {
                        DrawIcon(e, drawInfo.Bounds, icon, x);
                    }

                    x += icon.Width;
                    x += Constants.ExpanderMargin;
                }

                {
                    // アイコン
                    var icon = info.Icon;

                    DrawIcon(e, drawInfo.Bounds, icon, x);

                    x += icon.Width;
                    x += Constants.IconMargin;
                }

                {
                    // 名前
                    var rightMargin = 10; // 背景色の右端を常に角丸とした場合に文字がはみ出てしまうため、サイズを調整する
                    var nameBounds = new Rectangle(
                        drawInfo.Bounds.X + x,
                        drawInfo.Bounds.Y,
                        drawInfo.Bounds.Width - x - rightMargin,
                        drawInfo.Bounds.Height);
                    this.DrawSearchString(nameBounds, info.DisplayFileName, e.Graphics, drawInfo.TextColor);
                }
            }
        }

        /// <summary>
        /// The draw path.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        /// <param name="drawInfo">
        /// The draw info.
        /// </param>
        /// <param name="info">
        /// The info.
        /// </param>
        private void DrawPath(DrawListViewSubItemEventArgs e, DrawInfo drawInfo, FileInfo info)
        {
            TextRenderer.DrawText(
                e.Graphics,
                info.DisplayPath,
                this.Font,
                drawInfo.Bounds,
                drawInfo.TextColor,
                LeftFlag);
        }

        /// <summary>
        /// The draw search string.
        /// </summary>
        /// <param name="bounds">
        /// The bounds.
        /// </param>
        /// <param name="s">
        /// The s.
        /// </param>
        /// <param name="g">
        /// The g.
        /// </param>
        /// <param name="textColor">
        /// The text color.
        /// </param>
        private void DrawSearchString(Rectangle bounds, string s, Graphics g, Color textColor)
        {
            var regions = this.MakeMatchRegions(s).ToArray();
            if (regions.Any())
            {
                foreach (var region in regions)
                {
                    TextRenderer.DrawText(
                        g,
                        region.Word,
                        this.Font,
                        bounds,
                        region.IsMatch ? Color.Red : textColor,
                        LeftFlag);

                    var size = TextRenderer.MeasureText(
                        g,
                        region.Word,
                        this.Font,
                        new Size(bounds.Width, bounds.Height),
                        TextFormatFlags.NoPadding);

                    bounds.X += size.Width;
                    bounds.Width -= size.Width;
                }
            }
            else
            {
                TextRenderer.DrawText(g, s, this.Font, bounds, textColor, LeftFlag);
            }
        }

        /// <summary>
        /// The draw update timestamp.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        /// <param name="drawInfo">
        /// The draw info.
        /// </param>
        /// <param name="info">
        /// The info.
        /// </param>
        private void DrawUpdateTimestamp(DrawListViewSubItemEventArgs e, DrawInfo drawInfo, FileInfo info)
        {
            TextRenderer.DrawText(
                e.Graphics,
                info.DisplayUpdateTimestamp,
                this.Font,
                drawInfo.Bounds,
                drawInfo.TextColor,
                LeftFlag);
        }

        #endregion

        /// <summary>
        /// The draw info.
        /// </summary>
        private class DrawInfo
        {
            #region Fields

            /// <summary>
            /// The bounds.
            /// </summary>
            [SuppressMessage("StyleCop.CSharp.MaintainabilityRules", "SA1401:FieldsMustBePrivate", Justification = "Reviewed. Suppression is OK here.")]
            public Rectangle Bounds;

            /// <summary>
            /// The text color.
            /// </summary>
            [SuppressMessage("StyleCop.CSharp.MaintainabilityRules", "SA1401:FieldsMustBePrivate", Justification = "Reviewed. Suppression is OK here.")]
            public Color TextColor;

            #endregion
        }
    }
}
