﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.UIControls.EffectBrowser.Controls.Basic;

namespace EffectMaker.UIControls.EffectBrowser.Controls.AddressBar
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Diagnostics.CodeAnalysis;
    using System.Linq;
    using System.Windows.Forms;

    using EffectMaker.Foundation.Disposables;

    /// <summary>
    /// The eb history address bar.
    /// </summary>
    public partial class EBHistoryAddressBar : EBAdressBar, IMessageFilter
    {
        // undos.Last() が現在
        #region Constants

        /// <summary>
        /// The max history.
        /// </summary>
        private const int MaxHistory = 10;

        #endregion

        #region Fields

        /// <summary>
        /// The last closed.
        /// </summary>
        private readonly Stopwatch lastClosed = new Stopwatch();

        /// <summary>
        /// The redos.
        /// </summary>
        private readonly List<string> redos = new List<string>();

        /// <summary>
        /// The undos.
        /// </summary>
        private readonly List<string> undos = new List<string>();

        /// <summary>
        /// The is mouse over.
        /// </summary>
        private bool isMouseOver;

        /// <summary>
        /// The menu.
        /// </summary>
        private ContextMenuStrip menu;

        /// <summary>
        /// The undoing.
        /// </summary>
        private bool undoing;

        /// <summary>
        /// ツールチップです。
        /// </summary>
        private readonly EBToolTip toolTip = new EBToolTip();

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="EBHistoryAddressBar"/> class.
        /// </summary>
        public EBHistoryAddressBar()
        {
            this.InitializeComponent();

            this.UpdateView();

            this.DirectoryChanged += this.OnDirectoryChanged;

            Application.AddMessageFilter(this);

            PutToolTip(this.toolTip, this.ibnUndo, Properties.Resources.EBHistoryAddressBar_Undo);
            PutToolTip(this.toolTip, this.ibnRedo, Properties.Resources.EBHistoryAddressBar_Redo);
            PutToolTip(this.toolTip, this.icbHistory, Properties.Resources.EBHistoryAddressBar_Recent);
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets a value indicating whether is enabled redo.
        /// </summary>
        private bool IsEnabledRedo
        {
            get
            {
                return this.redos.Any();
            }
        }

        /// <summary>
        /// Gets a value indicating whether is enabled undo.
        /// </summary>
        private bool IsEnabledUndo
        {
            get
            {
                return this.undos.Count() >= 2;
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The destory.
        /// </summary>
        public override void Destory()
        {
            Application.RemoveMessageFilter(this);
        }

        /// <summary>
        /// The pre filter message.
        /// </summary>
        /// <param name="m">
        /// The m.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1126:PrefixCallsCorrectly", Justification = "Reviewed. Suppression is OK here.")]
        public bool PreFilterMessage(ref Message m)
        {
            const int WmXbuttondown = 0x20B;
            const int WmSyskeydown = 0x104;
            const int MkXbutton1 = 0x20;
            const int MkXbutton2 = 0x40;
            const int VkLeft = 0x25;
            const int VkRight = 0x27;

            if (this.Visible)
            {
                switch (m.Msg)
                {
                    case WmXbuttondown:
                        if (((int)m.WParam & MkXbutton1) != 0)
                        {
                            this.Undo();
                            return true;
                        }

                        if (((int)m.WParam & MkXbutton2) != 0)
                        {
                            this.Redo();
                            return true;
                        }

                        break;

                    case WmSyskeydown:
                        if (ModifierKeys == Keys.Alt)
                        {
                            if ((int)m.WParam == VkLeft)
                            {
                                this.Undo();
                                return true;
                            }

                            if ((int)m.WParam == VkRight)
                            {
                                this.Redo();
                                return true;
                            }
                        }

                        break;
                }
            }

            return false;
        }

        /// <summary>
        /// The redo.
        /// </summary>
        /// <param name="count">
        /// The count.
        /// </param>
        public void Redo(int count = 1)
        {
            if (this.undoing)
            {
                return;
            }

            if (this.IsEnabledRedo == false)
            {
                return;
            }

            using (new AnonymousDisposable(() => this.undoing = false))
            {
                this.undoing = true;
                {
                    var dir = this.Directory;

                    for (var i = 0; i != count; ++i)
                    {
                        var first = this.redos.First();
                        this.redos.RemoveAt(0);
                        this.undos.Add(first);

                        dir = first;
                    }

                    this.Directory = dir;
                }

                this.UpdateView();
            }
        }

        /// <summary>
        /// The undo.
        /// </summary>
        /// <param name="count">
        /// The count.
        /// </param>
        public void Undo(int count = 1)
        {
            if (this.undoing)
            {
                return;
            }

            if (this.IsEnabledUndo == false)
            {
                return;
            }

            using (new AnonymousDisposable(() => this.undoing = false))
            {
                this.undoing = true;
                {
                    var dir = this.Directory;

                    for (var i = 0; i != count; ++i)
                    {
                        this.undos.RemoveAt(this.undos.Count() - 1);
                        this.redos.Insert(0, dir);

                        dir = this.undos.Last();
                    }

                    this.Directory = dir;
                }

                this.UpdateView();
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on directory changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void OnDirectoryChanged(object sender, EventArgs e)
        {
            if (this.undoing)
            {
                return;
            }

            this.redos.Clear();
            this.undos.Add(this.Directory);

            while (this.undos.Count() > MaxHistory)
            {
                this.undos.RemoveAt(0);
            }

            this.UpdateView();
        }

        /// <summary>
        /// The popup menu.
        /// </summary>
        private void PopupMenu()
        {
            if (this.isMouseOver && this.lastClosed.IsRunning && (this.lastClosed.ElapsedMilliseconds < 20))
            {
                this.menu.Close();
                this.lastClosed.Stop();
                this.icbHistory.Checked = false;
                return;
            }

            // メニュー作成
            this.menu = new ContextMenuStrip();
            {
                Action<ToolStripMenuItem> menuClick = item =>
                    {
                        var index = (int)item.Tag;

                        // undo
                        if (index > 0)
                        {
                            this.Undo(index);
                        }

                            // redo
                        else if (index < 0)
                        {
                            index *= -1;
                            this.Redo(index);
                        }
                    };

                var menuItems = new List<ToolStripMenuItem>();
                {
                    {
                        var index = this.undos.Count() - 1;
                        menuItems.AddRange(
                            this.undos.Select(
                                x =>
                                    {
                                        var item = new ToolStripMenuItem { Text = x, Tag = index-- };

                                        item.Click += (s, e) => menuClick(s as ToolStripMenuItem);

                                        return item;
                                    }));
                    }

                    {
                        var index = -1;
                        menuItems.AddRange(
                            this.redos.Select(
                                x =>
                                    {
                                        var item = new ToolStripMenuItem { Text = x, Tag = index-- };

                                        item.Click += (s, e) => menuClick(s as ToolStripMenuItem);

                                        return item;
                                    }));
                    }

                    menuItems[this.undos.Count() - 1].Checked = true;
                }

                this.menu.Items.Clear();
                this.menu.Items.AddRange(menuItems.Cast<ToolStripItem>().Reverse().ToArray());

                this.menu.Closed += (s, e) =>
                    {
                        this.icbHistory.Checked = false;
                        this.lastClosed.Restart();
                    };
            }

            this.menu.Show(this.icbHistory, 0, this.icbHistory.Height);
            this.icbHistory.Checked = true;
        }

        /// <summary>
        /// The update view.
        /// </summary>
        private void UpdateView()
        {
            this.ibnUndo.Enabled = this.undos.Count() >= 2;
            this.ibnRedo.Enabled = this.redos.Any();
            this.icbHistory.Enabled = this.ibnUndo.Enabled | this.ibnRedo.Enabled;
        }

        /// <summary>
        /// The btn redo_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void BtnRedo_Click(object sender, EventArgs e)
        {
            this.Redo();
        }

        /// <summary>
        /// The btn undo_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void BtnUndo_Click(object sender, EventArgs e)
        {
            this.Undo();
        }

        /// <summary>
        /// The cbx history_ mouse down.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void CbxHistory_MouseDown(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                Application.DoEvents();
                this.PopupMenu();
            }
        }

        /// <summary>
        /// The cbx history_ mouse enter.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void CbxHistory_MouseEnter(object sender, EventArgs e)
        {
            this.isMouseOver = true;
        }

        /// <summary>
        /// The cbx history_ mouse leave.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void CbxHistory_MouseLeave(object sender, EventArgs e)
        {
            this.isMouseOver = false;
        }

        private static void PutToolTip(ToolTip toolTip, Control ctl, string text)
        {
            ctl.MouseEnter += (s, e) => toolTip.Show(text, ctl, 20, 25, 2000);
            ctl.MouseLeave += (s, e) => toolTip.Hide(ctl);

            Action toolTipHanlder = () =>
            {
                if (!ctl.Enabled)
                {
                    if (toolTip.Tag == ctl &&
                        !ctl.ClientRectangle.Contains(ctl.PointToClient(Cursor.Position)))
                    {
                        toolTip.Hide(ctl);
                        toolTip.Tag = null;
                    }
                    else if (toolTip.Tag != ctl &&
                        ctl.ClientRectangle.Contains(ctl.PointToClient(Cursor.Position)))
                    {
                        toolTip.Show(text, ctl, 20, 25, 2000);
                        toolTip.Tag = ctl;
                    }
                }
            };
            ctl.Parent.MouseMove += (sender, args) => toolTipHanlder();
            ctl.Parent.MouseLeave += (sender, args) => toolTipHanlder();
        }

        #endregion
    }
}
