﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.AddressBar
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;

    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb adress bar.
    /// </summary>
    public partial class EBAdressBar : UserControl
    {
        #region Fields

        /// <summary>
        /// The directory.
        /// </summary>
        private string directory = string.Empty;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="EBAdressBar"/> class.
        /// </summary>
        public EBAdressBar()
        {
            this.InitializeComponent();
        }

        #endregion

        #region Public Events

        /// <summary>
        /// The directory changed.
        /// </summary>
        public event EventHandler DirectoryChanged;

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the directory.
        /// </summary>
        [Browsable(false)]
        public string Directory
        {
            get
            {
                return this.directory;
            }

            set
            {
                if (this.directory == value)
                {
                    return;
                }

                // ローカルドライブとネットワークパスのみ受け付ける
                if (PathUtility.IsLocalDrivePath(value) == false &&
                    PathUtility.IsNetworkPath(value) == false)
                {
                    return;
                }

                this.directory = value;

                this.dblAddress.Directory = this.directory;
                using (new ControlEventSuppressBlock())
                {
                    this.tbxAddress.Directory = this.directory;
                }

                if (ControlEventSuppressBlock.Enabled == false)
                {
                    if (this.DirectoryChanged != null)
                    {
                        this.DirectoryChanged(this, EventArgs.Empty);
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the get directory history.
        /// </summary>
        public Func<IEnumerable<string>> GetDirectoryHistory
        {
            get
            {
                return this.dblAddress.GetDirectoryHistory;
            }

            set
            {
                this.dblAddress.GetDirectoryHistory = value;
                this.tbxAddress.GetDirectoryHistory = value;
            }
        }

        /// <summary>
        /// Gets or sets the prompt text.
        /// </summary>
        public string PromptText
        {
            get
            {
                return this.tbxAddress.PromptText;
            }

            set
            {
                this.tbxAddress.PromptText = value;
                this.dblAddress.PromptText = value;
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The destory.
        /// </summary>
        public virtual void Destory()
        {
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on load.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            this.tbxAddress.Visible = false;
            this.DoubleBuffered = true;

            this.tbxAddress.LostFocus += (ss, ee) =>
                {
                    using (new LayoutSuspendBlock(this))
                    {
                        this.dblAddress.Visible = true;
                        this.tbxAddress.Visible = false;
                        this.dblAddress.Focus();
                    }
                };

            this.tbxAddress.GotFocus += (ss, ee) => this.tbxAddress.SelectAll();

            this.dblAddress.Click += (ss, ee) =>
                {
                    using (new LayoutSuspendBlock(this))
                    {
                        this.tbxAddress.Visible = true;
                        this.dblAddress.Visible = false;
                        this.tbxAddress.Focus();
                    }
                };
        }

        /// <summary>
        /// The get directory path.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        /// <returns>
        /// The <see cref="string"/>.
        /// </returns>
        private static string GetDirectoryPath(DragEventArgs e)
        {
            var filenames = e.Data.GetData(DataFormats.FileDrop) as IEnumerable<string>;
            if (filenames == null)
            {
                return null;
            }

            var filenameArray = filenames as string[] ?? filenames.ToArray();
            {
                // ディレクトリ指定
                var directory =
                    filenameArray.FirstOrDefault(x => (PathUtility.IsLocalDrivePath(x) && System.IO.Directory.Exists(x)) ||
                                                      (PathUtility.IsNetworkPath(x) && PathUtility.IsAccessibleDirectory(x)));
                if (directory != null)
                {
                    return directory;
                }
            }

            {
                // .lnkファイル指定
                var lnkFile = filenameArray.FirstOrDefault(x => Path.GetExtension(x) == ".lnk");
                if (lnkFile != null)
                {
                    var shell = new IWshRuntimeLibrary.WshShell();

                    var shortcut = shell.CreateShortcut(lnkFile) as IWshRuntimeLibrary.IWshShortcut;

                    if (shortcut != null)
                    {
                        var targetPath = shortcut.TargetPath;
                        if (PathUtility.IsLocalDrivePath(targetPath) && System.IO.Directory.Exists(targetPath))
                        {
                            return targetPath;
                        }
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// The btn folder_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void BtnFolder_Click(object sender, EventArgs e)
        {
            using (
                var fbd = new FolderBrowserDialog
                              {
                                  SelectedPath = this.Directory,
                                  RootFolder = Environment.SpecialFolder.MyComputer,
                                  ShowNewFolderButton = false
                              })
            {
                if (PathUtility.DirectoryExists(this.Directory))
                {
                    fbd.SelectedPath = this.Directory;
                }

                if (fbd.ShowDialog() == DialogResult.OK)
                {
                    this.Directory = fbd.SelectedPath;
                }
            }
        }

        /// <summary>
        /// The dbl address_ directory changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void DblAddress_DirectoryChanged(object sender, EventArgs e)
        {
            this.Directory = this.dblAddress.Directory;
        }

        /// <summary>
        /// The tbx address_ directory changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TbxAddress_DirectoryChanged(object sender, EventArgs e)
        {
            this.Directory = this.tbxAddress.Directory;
        }

        /// <summary>
        /// The tbx address_ drag drop.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TbxAddress_DragDrop(object sender, DragEventArgs e)
        {
            var dirName = GetDirectoryPath(e);
            if (dirName != null)
            {
                this.Directory = dirName;
            }
        }

        /// <summary>
        /// The tbx address_ drag enter.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TbxAddress_DragEnter(object sender, DragEventArgs e)
        {
            e.Effect = (GetDirectoryPath(e) != null) ? DragDropEffects.All : DragDropEffects.None;
        }

        /// <summary>
        /// The tbx address_ key down.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TbxAddress_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                this.Directory = this.tbxAddress.Text;
                if (this.Parent != null)
                {
                    this.Parent.Focus();
                }
            }
        }

        /// <summary>
        /// The tbx address_ validated.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// イベント引数
        /// </param>
        private void TbxAddress_Validated(object sender, EventArgs e)
        {
            this.Directory = this.tbxAddress.Text;
        }

        #endregion
    }
}
