﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.UIControls.EffectBrowser.Controls.Basic;

namespace EffectMaker.UIControls.EffectBrowser.Controls.AddressBar.DirBreadcrumbList
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Windows.Forms;

    using EffectMaker.UIControls.EffectBrowser.Controls.Interfaces;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb dir breadcrumb list history arrow.
    /// </summary>
    public class EBDirBreadcrumbListHistoryArrow : EBDirBreadcrumbListArrow
    {
        #region Fields

        /// <summary>
        /// ツールチップです。
        /// </summary>
        private readonly EBToolTip toolTip;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="EBDirBreadcrumbListHistoryArrow"/> class.
        /// </summary>
        public EBDirBreadcrumbListHistoryArrow()
        {
            this.PopupChanged += (s, e) => this.State = this.IsPopupped ? StateKind.MouseDown : StateKind.Wait;

            this.ForceOpendArrow = true;
            this.PopupType = PopupTypeKind.Wide;
            this.Width = 22;
            this.toolTip = new EBToolTip();
        }

        #endregion

        #region Public Events

        /// <summary>
        /// The menu clicked.
        /// </summary>
        public event EventHandler MenuClicked;

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the get directory history.
        /// </summary>
        public Func<IEnumerable<string>> GetDirectoryHistory { get; set; }

        /// <summary>
        /// ツールチップで表示するテキストを取得または設定します。
        /// </summary>
        public string ToolTipText { get; set; }

        #endregion

        #region Methods

        /// <summary>
        /// The create dir list.
        /// </summary>
        /// <param name="param">
        /// The param.
        /// </param>
        /// <returns>
        /// The <see cref="ContextMenuStrip"/>.
        /// </returns>
        protected override ContextMenuStrip CreateDirList(object param)
        {
            System.Diagnostics.Debug.Assert(this.GetDirectoryHistory != null, "this.GetDirectoryHistory != null");

            if (this.GetDirectoryHistory().Any() == false)
            {
                return null;
            }

            var currentDir = ((IHasDirectory)this.Parent).Directory;

            var cms = new ContextMenuStrip();

            cms.Items.AddRange(
                this.GetDirectoryHistory().Where(x => x != currentDir).Select(
                    x =>
                        {
                            var item = new ToolStripMenuItem
                                           {
                                               Text = x,
                                               Image = ImageUtility.GetDirectoryIcon(x),
                                               Enabled = PathUtility.DirectoryExists(x),
                                               Tag = x
                                           };

                            item.Click += (s, e) => this.MenuClicked(s, e);

                            return item;
                        }).Cast<ToolStripItem>().ToArray());

            return cms;
        }

        /// <summary>
        /// The on mouse enter.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnMouseEnter(EventArgs e)
        {
            base.OnMouseEnter(e);

            this.State = StateKind.MouseOver;
            if (!string.IsNullOrEmpty(this.ToolTipText))
            {
                this.toolTip.Show(this.ToolTipText, this, 20, 25, 2000);
            }
        }

        /// <summary>
        /// The on mouse leave.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);

            this.State = StateKind.Wait;
            if (!string.IsNullOrEmpty(this.ToolTipText))
            {
                this.toolTip.Hide(this);
            }
        }

        #endregion
    }
}
