﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.AddressBar.DirBreadcrumbList
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;
    using System.Windows.Forms.VisualStyles;

    using EffectMaker.UIControls.EffectBrowser.Controls.Interfaces;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb dir breadcrumb list.
    /// </summary>
    public partial class EBDirBreadcrumbList : UserControl, IHasDirectory
    {
        #region Constants

        /// <summary>
        /// The empty click width reserve.
        /// </summary>
        private const int EmptyClickWidthReserve = 50;

        #endregion

        #region Fields

        /// <summary>
        /// The history arrow.
        /// </summary>
        private readonly EBDirBreadcrumbListHistoryArrow historyArrow = new EBDirBreadcrumbListHistoryArrow
                                                                             {
                                                                                 Dock =
                                                                                     DockStyle
                                                                                     .Right
                                                                             };

        /// <summary>
        /// The directory.
        /// </summary>
        private string directory = string.Empty;

        /// <summary>
        /// The prompt font.
        /// </summary>
        private Font promptFont;

        /// <summary>
        /// The prompt text.
        /// </summary>
        private string promptText = string.Empty;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="EBDirBreadcrumbList"/> class.
        /// </summary>
        public EBDirBreadcrumbList()
        {
            this.InitializeComponent();

            this.ResizeRedraw = true;
            this.DoubleBuffered = true;
            this.Padding = new Padding(1);
        }

        #endregion

        #region Public Events

        /// <summary>
        /// The directory changed.
        /// </summary>
        public event EventHandler DirectoryChanged;

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the directory.
        /// </summary>
        [Browsable(false)]
        public string Directory
        {
            get
            {
                return this.directory;
            }

            set
            {
                if (this.directory == value)
                {
                    return;
                }

                this.directory = value;

                this.UpdateItemSets();
                this.UpdateHistory();

                if (ControlEventSuppressBlock.Enabled == false)
                {
                    if (this.DirectoryChanged != null)
                    {
                        this.DirectoryChanged(this, EventArgs.Empty);
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the get directory history.
        /// </summary>
        public Func<IEnumerable<string>> GetDirectoryHistory
        {
            get
            {
                return this.historyArrow.GetDirectoryHistory;
            }

            set
            {
                this.historyArrow.GetDirectoryHistory = value;
                this.UpdateHistory();
            }
        }

        /// <summary>
        /// ヒストリー表示ボタンのツールチップテキストを取得または設定します。
        /// </summary>
        [Category("Appearance"), Description("Specifies the tool-tip of text on history button.")]
        public string HistoryButtonToolTipText
        {
            get { return this.historyArrow.ToolTipText; }
            set { this.historyArrow.ToolTipText = value; }
        }

        /// <summary>
        /// Gets or sets the prompt text.
        /// </summary>
        [Browsable(false)]
        public string PromptText
        {
            get
            {
                return this.promptText;
            }

            set
            {
                this.promptText = value;
                this.Invalidate();
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on paint.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            if (VisualStyleRenderer.IsSupported)
            {
                // http://social.msdn.microsoft.com/Forums/ja-JP/538c2ab2-93ef-4751-b98c-3632978cd484/textbox?forum=csharpgeneralja
                (new VisualStyleRenderer("EDIT", 6, 1)).DrawBackground(e.Graphics, e.ClipRectangle);
            }
            else if (TextBoxRenderer.IsSupported)
            {
                TextBoxRenderer.DrawTextBox(e.Graphics, e.ClipRectangle, TextBoxState.Normal);
            }

            if (string.IsNullOrEmpty(this.Directory) && (string.IsNullOrEmpty(this.PromptText) == false))
            {
                if (this.promptFont == null)
                {
                    this.promptFont = new Font(this.Font, FontStyle.Italic);
                }

                const TextFormatFlags LeftFlag = TextFormatFlags.Left | TextFormatFlags.VerticalCenter;

                TextRenderer.DrawText(
                    e.Graphics,
                    this.PromptText,
                    this.promptFont,
                    e.ClipRectangle,
                    Color.Gray,
                    LeftFlag);
            }
        }

        /// <summary>
        /// The on parent changed.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnParentChanged(EventArgs e)
        {
            base.OnParentChanged(e);

            if (this.Parent != null)
            {
                this.historyArrow.MenuClicked += (ss, ee) => this.Directory = ((ToolStripMenuItem)ss).Tag as string;
                this.Controls.Add(this.historyArrow);
            }
        }

        /// <summary>
        /// The on resize.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            this.UpdateItemSetVisible();
        }

        /// <summary>
        /// The on visible changed.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);

            this.UpdateHistory();
        }

        /// <summary>
        /// The update history.
        /// </summary>
        private void UpdateHistory()
        {
            if (this.DesignMode)
            {
                return;
            }

            this.historyArrow.Enabled = (this.GetDirectoryHistory != null) && this.GetDirectoryHistory().Any();
        }

        /// <summary>
        /// The update item set visible.
        /// </summary>
        private void UpdateItemSetVisible()
        {
            var maxWidth = this.Width;
            var width = EmptyClickWidthReserve; // 空クリックできる領域を予約

            foreach (var control in this.Controls.OfType<Control>())
            {
                width += control.Width;
                control.Visible = width < maxWidth;
            }
        }

        /// <summary>
        /// The update item sets.
        /// </summary>
        private void UpdateItemSets()
        {
            using (new LockWindowUpdateBlock(this))
            {
                this.Controls.Clear();
                this.Controls.Add(this.historyArrow);

                this.Padding = new Padding(1, 1, 1, 1);

                var itemSets = new List<Control>();
                {
                    var fullPath = string.Empty;

                    // 最初が@"\\"で始まっていたら、ネットワーク上のパスなので、
                    // ファイルパスを\\形式にする
                    if (fullPath == string.Empty && this.Directory.StartsWith(@"\\"))
                    {
                        fullPath = @"\\";
                    }

                    EBDirBreadcrumbListItemSet priorDepthItemSet = null;

                    foreach (var name in this.Directory.Split('\\').Where(x => string.IsNullOrEmpty(x) == false))
                    {
                        if (fullPath == @"\\")
                        {
                            fullPath = Path.Combine(fullPath, name + @"\");
                            continue;
                        }

                        fullPath = Path.Combine(fullPath, name + @"\");

                        var itemSet = new EBDirBreadcrumbListItemSet(fullPath, this.Font) { Dock = DockStyle.Left };
                        itemSets.Insert(0, itemSet);

                        if (priorDepthItemSet != null)
                        {
                            priorDepthItemSet.NextDepthDirectory = fullPath;
                        }

                        priorDepthItemSet = itemSet;
                    }
                }

                this.Controls.AddRange(itemSets.ToArray());
            }

            this.UpdateItemSetVisible();
            this.Invalidate();
        }

        #endregion
    }
}
