﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Windows.Forms.VisualStyles;
using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.UIDialogs.SearchDialog
{
    public partial class UISearchCondition : UIUserControl
    {
        public UISearchCondition()
        {
            InitializeComponent();
        }

        /// <summary>
        /// ConditionValues プロパティのバッキングフィールドです。
        /// </summary>
        private Dictionary<string, string> conditionValues;

        /// <summary>
        /// Operators プロパティのバッキングフィールドです。
        /// </summary>
        private Dictionary<string, int> operators;

        public SearchDialog ParentDialog { get; set; }

        /// <summary>
        /// 置換するかどうかを取得または設定します。
        /// </summary>
        public bool IsReplace { get; set; }

        /// <summary>
        /// 検索条件の識別名を取得または設定します。
        /// （例）A, B, C, D, E
        /// </summary>
        public string UniqueKey
        {
            get { return this.txbID.Text; }
            set { this.txbID.Text = value; }
        }

        /// <summary>
        /// 検索対象のノードの値を取得または設定します。
        /// （例）<AlphaFadeTime>10</AlphaFadeTime> の 10
        /// </summary>
        public string Value
        {
            get
            {
                if (this.cbxValue.DropDownStyle == ComboBoxStyle.DropDownList)
                {
                    return (string)this.cbxValue.SelectedValue;
                }
                else
                {
                    return this.cbxValue.Text;
                }
            }

            set { this.cbxValue.SelectedValue = value; }
        }

        /// <summary>
        /// 検索条件のラベルを取得または設定します。
        /// （例）"基本設定/基本設定/パーティクル追従タイプ"
        /// </summary>
        public string Label
        {
            get { return this.txbText.Text; }
            set { this.txbText.Text = value; }
        }

        /// <summary>
        /// 比較方法または置換方法を取得または設定します。
        /// </summary>
        public int SelectedComparison
        {
            get { return (int)this.cbxComparison.SelectedValue; }
            set { this.cbxComparison.SelectedValue = value; }
        }

        /// <summary>
        /// 検索条件の値または置換する値の選択肢を取得または設定します。
        /// </summary>
        public Dictionary<string, string> ConditionValues
        {
            get
            {
                return this.conditionValues;
            }

            set
            {
                this.conditionValues = value;

                // コンボボックスの選択肢を更新
                if (this.conditionValues != null && this.conditionValues.Any())
                {
                    this.cbxValue.DisplayMember = "Key";
                    this.cbxValue.ValueMember   = "Value";
                    this.cbxValue.DataSource    = this.conditionValues.Select(x => new ConditionValueItem(x.Key, x.Value)).ToArray();

                    this.cbxValue.Enabled       = true;
                    this.cbxValue.DropDownStyle = ComboBoxStyle.DropDownList;

                    this.cbxValue.SelectedIndex = 0;
                }
            }
        }

        /// <summary>
        /// 条件値の選択肢を取得または設定します。
        /// </summary>
        public Dictionary<string, int> Operators
        {
            get
            {
                return this.operators;
            }

            set
            {
                this.operators = value;

                // コンボボックスの選択肢を更新
                if (this.operators != null && operators.Any())
                {
                    this.cbxComparison.DisplayMember = "Key";
                    this.cbxComparison.ValueMember   = "Value";
                    this.cbxComparison.DataSource    = this.operators.Select(x => new OperatorItem(x.Key, x.Value)).ToArray();

                    this.cbxComparison.Enabled       = true;
                    this.cbxComparison.DropDownStyle = ComboBoxStyle.DropDownList;

                    this.cbxComparison.SelectedIndex = 0;
                }
            }
        }

        /// <summary>
        /// Condition プロパティのバッキングフィールドです。
        /// </summary>
        private SearchCondition condition;

        /// <summary>
        /// 検索条件を取得または設定します。
        /// </summary>
        public SearchCondition Condition
        {
            get
            {
                return this.condition;
            }

            set
            {
                this.condition = value;

                this.txbText.Text = this.condition.Label;
            }
        }

        private void btnTrashBox_Click(object sender, EventArgs e)
        {
            if (this.ParentDialog == null)
            {
                return;
            }
            if (!this.IsReplace)
            {
                this.ParentDialog.DeleteCondition(this.UniqueKey);
            }
            else
            {
                this.ParentDialog.DeleteConditionReplace(this.UniqueKey);
            }
        }

        /// <summary>
        /// 条件値選択コンボボックスに表示する項目です。
        /// ComboBox.DataSource に Generic な Dictionary を直接指定できないため、
        /// 一度この値に変換してから DataSource に指定します。
        /// </summary>
        private class ConditionValueItem
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public ConditionValueItem()
            {
            }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="key">キー</param>
            /// <param name="value">値</param>
            public ConditionValueItem(string key, string value)
            {
                this.Key = key;
                this.Value = value;
            }

            /// <summary>
            /// キーを取得または設定します。
            /// </summary>
            public string Key { get; set; }

            /// <summary>
            /// 値を取得または設定します。
            /// </summary>
            public string Value { get; set; }

            /// <summary>
            /// 現在のオブジェクトを表す文字列を返します。
            /// この値がコンボボックスのドロップダウンリストに表示されるため、キーを返します。
            /// </summary>
            /// <returns>現在のオブジェクトを表す文字列を返します。</returns>
            public override string ToString()
            {
                return this.Key;
            }
        }

        /// <summary>
        /// 比較方法選択コンボボックスに表示する項目です。
        /// ComboBox.DataSource に Generic な Dictionary を直接指定できないため、
        /// 一度この値に変換してから DataSource に指定します。
        /// </summary>
        private class OperatorItem
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public OperatorItem()
            {
            }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="key">キー</param>
            /// <param name="value">値</param>
            public OperatorItem(string key, int value)
            {
                this.Key = key;
                this.Value = value;
            }

            /// <summary>
            /// キーを取得または設定します。
            /// </summary>
            public string Key { get; set; }

            /// <summary>
            /// 値を取得または設定します。
            /// </summary>
            public int Value { get; set; }

            /// <summary>
            /// 現在のオブジェクトを表す文字列を返します。
            /// この値がコンボボックスのドロップダウンリストに表示されるため、キーを返します。
            /// </summary>
            /// <returns>現在のオブジェクトを表す文字列を返します。</returns>
            public override string ToString()
            {
                return this.Key;
            }
        }
    }
}
