﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml;
using EffectMaker.BusinessLogic.Search;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIDialogs.SearchDialog.Plugins;

namespace EffectMaker.UIDialogs.SearchDialog
{
    public partial class SearchDialog : Form
    {
        private const int maxConditionNum = 5;
        private const int uiConditionHeight = 32;
        private readonly SearchConditions conditions = new SearchConditions();

        private readonly List<UISearchCondition> uiConditions = new List<UISearchCondition>();
        private readonly List<UISearchCombination> uiCombinations = new List<UISearchCombination>();
        private readonly List<UISearchCondition> uiReplaces = new List<UISearchCondition>();

        private SearchLogger logger;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SearchDialog()
        {
            this.InitializeComponent();
        }

        /// <summary>
        /// XPath テーブルをクリアします。
        /// </summary>
        public void ClearXPathTable()
        {
            this.conditions.Clear();
        }

        /// <summary>
        /// XPath テーブルを追加します。
        /// </summary>
        /// <param name="stream">XPath テーブルのストリーム</param>
        public void AddXPathTable(Stream stream)
        {
            SearchTableByDataModelMaker table = SearchTableByDataModelMakerSerializer.Load(stream);

            this.conditions.Add(table);
        }

        /// <summary>
        /// XPathTable に登録できない検索条件をコードで追加します。
        /// </summary>
        public void AddXPathTablePlugin()
        {
            {
                SearchCondition c = new SearchCondition();
                c.IsAttribute = true;
                c.Category1 = "その他";
                c.Category2 = "存在判定";
                c.Category3 = "チャイルドエミッタ";
                c.Value = "";
                c.Plugin = new SearchPluginFindChildEmitter();
                this.conditions.Items.Add(c);
            }

            {
                SearchCondition c = new SearchCondition();
                c.IsAttribute = true;
                c.Category1 = "その他";
                c.Category2 = "存在判定";
                c.Category3 = "ストライプ";
                c.Value = "";
                c.Plugin = new SearchPluginFindStripe();
                this.conditions.Items.Add(c);
            }

            {
                SearchCondition c = new SearchCondition();
                c.IsAttribute = true;
                c.Category1 = "その他";
                c.Category2 = "存在判定";
                c.Category3 = "カスタムアクション";
                c.Value = "";
                c.Plugin = new SearchPluginFindCustomAction();
                this.conditions.Items.Add(c);
            }
        }

        /// <summary>
        /// 列挙値定義データを追加します。
        /// </summary>
        /// <param name="stream">XPath テーブルのストリーム</param>
        public void AddEnumDefinitions(Stream stream)
        {
            SearchTableByDataModelMaker table = SearchTableByDataModelMakerSerializer.Load(stream);

            foreach (SearchItem searchItem in table.SearchItems)
            {
                if (searchItem.EnumDefinitionItems != null && searchItem.EnumDefinitionItems.Any())
                {
                    SearchCondition condition = this.conditions.Items.FirstOrDefault(x => x.XPath == searchItem.XPath);

                    if (condition != null)
                    {
                        condition.Processor.OverrideConditionValues = searchItem.EnumDefinitionItems.ToDictionary(x => x.Key, x => x.Value);
                    }
                    else
                    {
                        Debug.Print(string.Format("XPath \"{0}\" is not found.\n", searchItem.XPath));
                    }
                }
            }
        }

        private void SearchReplaceDialog_Load(object sender, EventArgs e)
        {
            this.logger = new SearchLogger(this.txbLog, txbSearchResult);

            // カルチャが日本語のときは英語未対応の警告を非表示にする
            if (Thread.CurrentThread.CurrentCulture.Name == "ja-JP")
            {
                int sub = this.uiTabControl1.Top - this.lblWarningCulture.Top;

                this.uiTabControl1.Top -= sub;
                this.uiTabControl1.Height += sub;

                this.lblWarningCulture.Visible = false;
            }
        }

        public void SetupUI()
        {
            this.cbxCategory1.Items.Clear();
            this.cbxCategory1R.Items.Clear();
            this.cbxCategory2.Items.Clear();
            this.cbxCategory2R.Items.Clear();
            this.cbxCategory3.Items.Clear();
            this.cbxCategory3R.Items.Clear();

            var category1 = this.CreateCategory1List();
            foreach (var item in category1)
            {
                this.cbxCategory1.Items.Add(item);
                this.cbxCategory1R.Items.Add(item);
            }

            this.cbxCategory1.SelectedIndex = 0;
            this.cbxCategory1R.SelectedIndex = 0;
        }

        public void DeleteCondition(string id)
        {
            this.DeleteCondition(id, this.pnlSearchConditionList, this.uiConditions, false);
        }

        public void DeleteConditionReplace(string id)
        {
            this.DeleteCondition(id, this.pnlReplaceConditionList, this.uiReplaces, true);
        }

        private void DeleteCondition(
            string id,
            UIControl control,
            List<UISearchCondition> uiConditions,
            bool isReplace)
        {
            List<SearchCondition> newConditions = new List<SearchCondition>();

            foreach (var item in uiConditions)
            {
                if (item.UniqueKey == id)
                {
                    continue;
                }
                newConditions.Add(item.Condition);
            }

            uiConditions.Clear();
            control.Controls.Clear();

            for (int i = 0; i < newConditions.Count; i++)
            {
                var condition = newConditions[i];
                var uiCondition = SearchUtility.CreateUICondition(i, condition, this);
                uiCondition.IsReplace = isReplace;
                control.Controls.Add(uiCondition);
                uiConditions.Add(uiCondition);
            }
        }

        private void txbFolder_DragDrop(object sender, DragEventArgs e)
        {
            //ドロップされたファイルの一覧を取得
            string[] fileName = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            if (fileName.Length <= 0)
            {
                return;
            }

            // ドロップ先がTextBoxであるかチェック
            TextBox txtTarget = sender as TextBox;
            if (txtTarget == null)
            {
                return;
            }

            //TextBoxの内容をファイル名に変更
            txtTarget.Text = fileName[0];
        }

        private void txbFolder_DragEnter(object sender, DragEventArgs e)
        {
            //ファイルがドラッグされている場合、カーソルを変更する
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                e.Effect = DragDropEffects.Copy;
            }
        }

        private void btnOpenFolderDialog_Click(object sender, EventArgs e)
        {
            FolderBrowserDialog fbd = new FolderBrowserDialog();

            //上部に表示する説明テキストを指定する
            fbd.Description = "フォルダを指定してください。";
            //ルートフォルダを指定する
            //デフォルトでDesktop
            fbd.RootFolder = Environment.SpecialFolder.Desktop;
            //最初に選択するフォルダを指定する
            //RootFolder以下にあるフォルダである必要がある
            fbd.SelectedPath = @"C:\Windows";
            //ユーザーが新しいフォルダを作成できるようにする
            //デフォルトでTrue
            fbd.ShowNewFolderButton = true;

            //ダイアログを表示する
            if (fbd.ShowDialog(this) == DialogResult.OK)
            {
                //選択されたフォルダを表示する
                this.txbFolder.Text = fbd.SelectedPath;
            }
        }

        private void btnClearResultLog_Click(object sender, EventArgs e)
        {
            this.txbSearchResult.Text = string.Empty;
        }

        private void btnClearLog_Click(object sender, EventArgs e)
        {
            this.txbLog.Text = string.Empty;
        }

        private void btnCopySearchResult_Click(object sender, EventArgs e)
        {
            if(string.IsNullOrEmpty(this.txbSearchResult.Text))
            {
                return;
            }

            Clipboard.SetText(this.txbSearchResult.Text);
        }

        private void btnCopyLog_Click(object sender, EventArgs e)
        {
            if (string.IsNullOrEmpty(this.txbLog.Text))
            {
                return;
            }

            Clipboard.SetText(this.txbLog.Text);
        }

        /// <summary>
        /// ユーザーが入力した値を反映させます。
        /// </summary>
        private void UpdateCondition()
        {
            for (int i = 0; i < uiConditions.Count; i++)
            {
                var ui = this.pnlSearchConditionList.Controls[i] as UISearchCondition;
                uiConditions[i].Condition.Value = ui.Value;
                uiConditions[i].Condition.Operator = ui.SelectedComparison;
            }

            for (int i = 0; i < uiReplaces.Count; i++)
            {
                var ui = this.pnlReplaceConditionList.Controls[i] as UISearchCondition;
                uiReplaces[i].Condition.Value = ui.Value;
                uiConditions[i].Condition.Operator = ui.SelectedComparison;
            }
        }

        private void btnLogReplaceConditions_Click(object sender, EventArgs e)
        {
            foreach (var panel in uiReplaces)
            {
                this.logger.LogBar();
                this.logger.Log("Replace condition.");
                this.logger.Log(panel.Label);
                this.logger.Log(panel.Value);
                this.logger.Log(panel.SelectedComparison.ToString());
                this.logger.Log("xpath : " + panel.Condition.XPath);
            }
        }

        private void btnAddCombination_Click(object sender, EventArgs e)
        {
            if (this.uiCombinations.Count >= maxConditionNum)
            {
                MessageBox.Show("合成式は最大５つです。");
                return;
            }

            var uiCombination = new UISearchCombination();

            int count = this.uiCombinations.Count;
            uiCombination.Top = uiConditionHeight * count;

            for (int i = 0; i < maxConditionNum; i++)
            {
                uiCombination.AddLeftItem(SearchUtility.GetID(i));
                uiCombination.AddRightItem(SearchUtility.GetID(i));
            }

            for (int i = 0; i < count; i++)
            {
                uiCombination.AddLeftItem("式" + (i + 1));
                uiCombination.AddRightItem("式" + (i + 1));
            }

            uiCombination.SetTitle("式" + (count + 1));

            uiCombination.Setup();

            this.pnlCombinationList.Controls.Add(uiCombination);
            this.uiCombinations.Add(uiCombination);
        }

        private void btnLogSearchConditions_Click(object sender, EventArgs e)
        {
            foreach (var panel in uiConditions)
            {
                this.logger.LogBar();
                this.logger.Log("Search condition.");
                this.logger.Log(panel.Condition.Label);
                this.logger.Log(panel.Value);
                this.logger.Log(panel.SelectedComparison.ToString());
                this.logger.Log("xpath : " + panel.Condition.XPath);
            }

            foreach (var panel in uiCombinations)
            {
                this.logger.LogBar();
                this.logger.Log("binomial condition.");
                this.logger.Log("Lhs : " + panel.Lhs);
                this.logger.Log("Rhs : " + panel.Rhs);
                this.logger.Log("Operator : " + panel.Operator);
            }
        }

        /// <summary>
        /// 検索または一括編集を行います。
        /// </summary>
        /// <param name="args">検索または一括編集の引数です。</param>
        private void DoIt(SearchArgs args)
        {
            if (uiConditions.Count == 0)
            {
                this.logger.Log("検索条件が入力されていません。");
                return;
            }

            if (uiConditions.Count == 1)
            {
                this.logger.Log("検索条件が１つなのでこの検索条件に一致するものを検索します。");
            }

            if (uiConditions.Count >= 2 && uiCombinations.Count == 0)
            {
                this.logger.Log("条件式を入力してください。");
                return;
            }

            if (args.IsDoReplace && this.uiReplaces.Count == 0)
            {
                this.logger.Log("置換式を入力してください。");
                return;
            }

            // ユーザーが入力した内容を反映させる
            this.UpdateCondition();

            UISearchCombination combination = null;

            // root となる合成式を設定する
            if (uiCombinations.Count >= 1)
            {
                combination = uiCombinations[uiCombinations.Count - 1];
            }

            SearchTree tree = new SearchTree();
            tree.Logger = this.logger;

            SearchUtility.SetupTree(tree, combination, uiConditions, uiCombinations);
            tree.Dump();

            args.Logger = this.logger;
            args.Folder = this.txbFolder.Text;
            args.SearchPanels = this.uiConditions;
            args.ReplacePanels = this.uiReplaces;
            args.Tree = tree;

            SearchProgressDialog pd = new SearchProgressDialog();
            pd.Show(this);

            SearchUtility.DoSearch(args, pd);
            pd.Close();
        }

        private void btnDoReplace_Click(object sender, EventArgs e)
        {
            SearchArgs args = new SearchArgs();
            args.IsDoReplace = true;
            this.DoIt(args);
        }

        private void btnDoSearch_Click(object sender, EventArgs e)
        {
            SearchArgs args = new SearchArgs();
            this.DoIt(args);
        }

        private void btnClearConditions_Click(object sender, EventArgs e)
        {
            this.uiConditions.Clear();
            this.pnlSearchConditionList.Controls.Clear();
        }

        private void btnClearCombinations_Click(object sender, EventArgs e)
        {
            this.uiCombinations.Clear();
            this.pnlCombinationList.Controls.Clear();

        }

        private string GetSelectedLabel()
        {
            string label1 = SearchUtility.GetSelectedString(cbxCategory1);
            string label2 = SearchUtility.GetSelectedString(cbxCategory2);
            string label3 = SearchUtility.GetSelectedString(cbxCategory3);

            return label1 + "/" + label2 + "/" + label3;
        }

        private string GetSelectedLabelR()
        {
            string label1 = SearchUtility.GetSelectedString(cbxCategory1R);
            string label2 = SearchUtility.GetSelectedString(cbxCategory2R);
            string label3 = SearchUtility.GetSelectedString(cbxCategory3R);

            return label1 + "/" + label2 + "/" + label3;
        }

        private IList<string> CreateCategory1List()
        {
            List<string> result = new List<string>();

            foreach (var item in this.conditions.Items)
            {
                var category = item.Category1;

                if (!result.Contains(category))
                {
                    result.Add(category);
                }
            }

            return result;
        }

        private IList<string> CreateCategory2List(string selectedCategory)
        {
            List<string> result = new List<string>();

            foreach (var condition in this.conditions.Items)
            {
                if (condition.Category1 != selectedCategory)
                {
                    continue;
                }

                if (result.Contains(condition.Category2))
                {
                    continue;
                }

                result.Add(condition.Category2);
            }

            return result;
        }

        private IList<string> CreateCategory3List(string selectedCategory)
        {
            List<string> result = new List<string>();

            foreach (var condition in this.conditions.Items)
            {
                if (condition.Category2 != selectedCategory)
                {
                    continue;
                }

                if (result.Contains(condition.Category3))
                {
                    continue;
                }

                result.Add(condition.Category3);
            }

            return result;
        }

        private void cbxCategory1_SelectedIndexChanged(object sender, EventArgs e)
        {
            string selectedCategory = SearchUtility.GetSelectedString(this.cbxCategory1);
            IList<string> list = this.CreateCategory2List(selectedCategory);
            UIComboBox target = this.cbxCategory2;
            SearchUtility.SetupCombobox(list, target);
        }

        private void cbxCategory2_SelectedIndexChanged(object sender, EventArgs e)
        {
            string selectedCategory = SearchUtility.GetSelectedString(this.cbxCategory2);
            IList<string> list = this.CreateCategory3List(selectedCategory);
            UIComboBox target = this.cbxCategory3;
            SearchUtility.SetupCombobox(list, target);
        }


        private void btnAddSearchCondition_Click(object sender, EventArgs e)
        {
            if (this.uiConditions.Count >= maxConditionNum)
            {
                MessageBox.Show("条件は最大５つです。");
                return;
            }

            string label = this.GetSelectedLabel();
            var condition = SearchUtility.GetSearchCondition(label, this.conditions);

            if (condition == null)
            {
                MessageBox.Show("条件が見つかりませんでした。");
                return;
            }

            var uiCondition = SearchUtility.CreateUICondition(this.uiConditions.Count, condition, this);
            uiCondition.IsReplace = false;
            this.pnlSearchConditionList.Controls.Add(uiCondition);
            this.uiConditions.Add(uiCondition);
        }

        private void btnAddReplaceCondition2_Click(object sender, EventArgs e)
        {
            if (this.uiReplaces.Count >= maxConditionNum)
            {
                MessageBox.Show("条件は最大５つです。");
                return;
            }

            string label = this.GetSelectedLabelR();
            var condition = SearchUtility.GetSearchCondition(label, this.conditions);

            if (condition == null)
            {
                MessageBox.Show("条件が見つかりませんでした。");
                return;
            }

            var uiCondition = SearchUtility.CreateReplaceCondition(this.uiReplaces.Count, condition, this);
            uiCondition.IsReplace = true;
            this.pnlReplaceConditionList.Controls.Add(uiCondition);
            this.uiReplaces.Add(uiCondition);
        }

        private void cbxCategory1R_SelectedIndexChanged(object sender, EventArgs e)
        {
            string selectedCategory = SearchUtility.GetSelectedString(this.cbxCategory1R);
            IList<string> list = this.CreateCategory2List(selectedCategory);
            UIComboBox target = this.cbxCategory2R;
            SearchUtility.SetupCombobox(list, target);
        }

        private void cbxCategory2R_SelectedIndexChanged(object sender, EventArgs e)
        {
            string selectedCategory = SearchUtility.GetSelectedString(this.cbxCategory2R);
            IList<string> list = this.CreateCategory3List(selectedCategory);
            UIComboBox target = this.cbxCategory3R;
            SearchUtility.SetupCombobox(list, target);
        }
    }
}
