﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.UIDialogs.Properties;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIDialogs.MessageDialogs
{
    /// <summary>
    /// 名前とプリセット入力ダイアログクラスです。
    /// </summary>
    public partial class NameAndPresetInputDialog : Form
    {
        /// <summary>
        /// プリセットリストの情報です。
        /// </summary>
        private List<PresetItem> presetList;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public NameAndPresetInputDialog()
        {
            this.InitializeComponent();

            // ボタンに表示する文字を設定
            this.btnOk.Text = Properties.Resources.ControlsDialogResultOk;
            this.btnCancel.Text = Properties.Resources.ControlsDialogResultCancel;

            // プリセットのラベルを設定
            this.lblPreset.Text = Resources.NameAndPresetInputDialogPresetComboBoxCaption;

            // ラベルに合わせてコンボボックスの位置を修正
            this.cbPreset.Location = new Point(this.lblPreset.Right + 3, this.cbPreset.Top);

            // プリセット選択処理は未実装
            this.cbPreset.Enabled = false;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="caption">キャプション</param>
        /// <param name="defaultName">デフォルトの名前</param>
        /// <param name="argPresetList">プリセットリスト</param>
        /// <param name="notGoodNameList">被ってはいけない名前のリスト</param>
        public NameAndPresetInputDialog(
            string caption,
            string defaultName,
            List<PresetItem> argPresetList,
            IEnumerable<string> notGoodNameList)
        {
            this.InitializeComponent();
            this.presetList = argPresetList;

            // ボタンに表示する文字を設定
            this.btnOk.Text = Properties.Resources.ControlsDialogResultOk;
            this.btnCancel.Text = Properties.Resources.ControlsDialogResultCancel;

            // キャプション, デフォルト名を設定
            this.Caption = caption;
            this.tbInputedName.Text = defaultName;

            // プリセットのラベルを設定
            this.lblPreset.Text = Resources.NameAndPresetInputDialogPresetComboBoxCaption;

            // ラベルに合わせてコンボボックスの位置を修正
            this.cbPreset.Location = new Point(this.lblPreset.Right + 3, this.cbPreset.Top);

            Image itemImageUser = null;
            Image itemImageProj = null;
            if (this.presetList.Count > 1)
            {
                if (this.presetList[1].Value is KeyValuePair<string, string>)
                {
                    itemImageUser = Resources.Icon_Emitter_Simple_Add;
                    itemImageProj = Resources.Icon_Emitter_Simple_Add_Proj;
                }
                else
                {
                    itemImageUser = Resources.Icon_EmitterSet_Add;
                    itemImageProj = Resources.Icon_EmitterSet_Add_Proj;
                }
            }

            foreach (var item in this.presetList)
            {
                if (item.IsUnselected)
                {
                    // プリセット未選択
                    this.cbPreset.Items.Add(new KeyValuePair<Image, string>(Resources.Icon_View_OFF, item.Key));
                }
                else if (item.SettingType == PresetSettingTypes.User)
                {
                    // ユーザー設定
                    this.cbPreset.Items.Add(new KeyValuePair<Image, string>(itemImageUser, item.Key));
                }
                else if (item.SettingType == PresetSettingTypes.Project)
                {
                    // プロジェクト設定
                    this.cbPreset.Items.Add(new KeyValuePair<Image, string>(itemImageProj, item.Key));
                }
            }

            this.cbPreset.SelectedIndex = 0;
            this.cbPreset.DropDownStyle = ComboBoxStyle.DropDownList;
            this.cbPreset.Enabled = this.presetList.Count > 1;

            this.cbPreset.SelectedIndexChanged += (s, e) =>
            {
                int index = this.cbPreset.SelectedIndex;
                if (index != 0)
                {
                    if (presetList.Count > 1)
                    {
                        string name;

                        if (presetList[1].Value is KeyValuePair<string, string>)
                        {
                            name = ((KeyValuePair<string, string>)this.presetList[index].Value).Value;
                            int count = 1;
                            while (notGoodNameList.Contains(name))
                            {
                                if (name.EndsWith("_" + count))
                                {
                                    string postFix = "_" + count;
                                    ++count;
                                    name = name.Substring(0, name.Length - postFix.Length) + "_" + count;
                                }
                                else
                                {
                                    name = name + "_1";
                                }
                            }
                        }
                        else
                        {
                            name = (string)this.presetList[index].Key;
                            int count = 1;
                            while (notGoodNameList.Contains(name))
                            {
                                if (name.EndsWith("_" + count))
                                {
                                    string postFix = "_" + count;
                                    ++count;
                                    name = name.Substring(0, name.Length - postFix.Length) + "_" + count;
                                }
                                else
                                {
                                    name = name + "_1";
                                }
                            }
                        }

                        this.InputedName = name;
                    }
                }
            };
        }

        /// <summary>
        /// キャプションを取得または設定します。
        /// </summary>
        public string Caption
        {
            get
            {
                return this.Text;
            }

            set
            {
                this.Text = value;
            }
        }

        /// <summary>
        /// 入力した名前を取得または設定します。
        /// </summary>
        public string InputedName
        {
            get
            {
                return this.tbInputedName.Text;
            }

            set
            {
                this.tbInputedName.Text = value;
            }
        }

        /// <summary>
        /// コンボボックスで選択したプリセットのインデックス
        /// </summary>
        public int SelectedPreset
        {
            get
            {
                return this.cbPreset.SelectedIndex;
            }

            set
            {
                this.cbPreset.SelectedIndex = value;
            }
        }
    }
}
