﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Media;
using System.Windows.Forms;

namespace EffectMaker.UIDialogs.GeneralControls
{
    /// <summary>
    /// ダイアログサウンド効果。
    /// </summary>
    public enum DialogSoundEffect
    {
        /// <summary>なし。</summary>
        None,

        /// <summary>情報。</summary>
        Information,

        /// <summary>応答。</summary>
        Question,

        /// <summary>警告。</summary>
        Warning,

        /// <summary>エラー。</summary>
        Error
    }

    /// <summary>
    /// モーダルダイアログクラス。
    /// </summary>
    public partial class ModalDialog : Form
    {
        /// <summary>
        /// サウンド効果です。
        /// </summary>
        private DialogSoundEffect soundEffect = DialogSoundEffect.None;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ModalDialog()
        {
            this.InitializeComponent();
        }

        /// <summary>
        /// サウンド効果を設定または取得します。
        /// </summary>
        [DefaultValue(DialogSoundEffect.None)]
        [Description("ダイアログ表示時に鳴らすサウンド効果です。")]
        public DialogSoundEffect SoundEffect
        {
            get
            {
                return this.soundEffect;
            }

            set
            {
                this.soundEffect = value;
            }
        }

        #region デザイナ制御

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool IsMdiContainer
        {
            get
            {
                return base.IsMdiContainer;
            }

            set
            {
                base.IsMdiContainer = value;
            }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool MaximizeBox
        {
            get
            {
                return base.MaximizeBox;
            }

            set
            {
                base.MaximizeBox = value;
            }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool MinimizeBox
        {
            get
            {
                return base.MinimizeBox;
            }

            set
            {
                base.MinimizeBox = value;
            }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool TopMost
        {
            get
            {
                return base.TopMost;
            }

            set
            {
                base.TopMost = value;
            }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new FormWindowState WindowState
        {
            get
            {
                return base.WindowState;
            }

            set
            {
               base.WindowState = value;
            }
        }

        #endregion

        /// <summary>
        /// ダイアログを表示（再定義しています）。
        /// 表示終了後にオーナー側の無効領域更新を行います。
        /// </summary>
        /// <returns>ユーザーの入力を返します。</returns>
        public new DialogResult ShowDialog()
        {
            DialogResult result = base.ShowDialog();
            Application.DoEvents();

            return result;
        }

        /// <summary>
        /// ダイアログを表示（再定義しています）。
        /// 表示終了後にオーナー側の無効領域更新を行います。
        /// </summary>
        /// <param name="owner">オーナーコントロール</param>
        /// <returns>ユーザーの入力を返します。</returns>
        public new DialogResult ShowDialog(IWin32Window owner)
        {
            DialogResult result = base.ShowDialog(owner);
            Application.DoEvents();

            return result;
        }

        #region オーバーライド

        /// <summary>
        /// オーバーライド。
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnLoad(EventArgs e)
        {
            Screen screen = Screen.FromControl(this);

            if (this.Height > screen.WorkingArea.Height)
            {
                this.Height = screen.WorkingArea.Height;
            }

            if (!this.DesignMode)
            {
                // サイズ変更モード時の設定
                if (this.FormBorderStyle == FormBorderStyle.Sizable)
                {
                    // TODO:アイコン設定
                    // アイコン
//                    this.Image = TheApp.MainFrame.Image;
                }
            }

            base.OnLoad(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnShown(EventArgs e)
        {
            if (!this.DesignMode)
            {
                // サウンド効果
                switch (this.soundEffect)
                {
                    case DialogSoundEffect.Information:
                        SystemSounds.Asterisk.Play();
                        break;
                    case DialogSoundEffect.Question:
                        // WindowsXP 標準設定では問い合わせサウンドは未設定なので警告音を出しておく
                        // 特に違和感はないので気づかれないでしょ...
                        // SystemSounds.Question.Play();
                        SystemSounds.Exclamation.Play();
                        break;
                    case DialogSoundEffect.Warning:
                        SystemSounds.Exclamation.Play();
                        break;
                    case DialogSoundEffect.Error:
                        SystemSounds.Hand.Play();
                        break;
                    default:
                        break;
                }
            }

            base.OnShown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            // 結果コード別ハンドラ呼び出し
            switch (this.DialogResult)
            {
                case DialogResult.OK:
                    e.Cancel = !this.OnResultOk();
                    break;
                case DialogResult.Cancel:
                    e.Cancel = !this.OnResultCancel();
                    break;
                case DialogResult.Yes:
                    e.Cancel = !this.OnResultYes();
                    break;
                case DialogResult.No:
                    e.Cancel = !this.OnResultNo();
                    break;
                default:
                    e.Cancel = !this.OnResultCancel();
                    break;
            }

            base.OnFormClosing(e);
        }

        #endregion

        #region 派生クラスオーバーライド用

        /// <summary>
        /// 「ＯＫ」ハンドラ。
        /// ダイアログを閉じる場合は true を返します。
        /// 基本クラスのメソッドを呼び出す必要はありません。
        /// </summary>
        /// <returns>ダイアログを閉じる場合は true を返します。</returns>
        protected virtual bool OnResultOk()
        {
            return true;
        }

        /// <summary>
        /// 「キャンセル」ハンドラ。
        /// 基本クラスのメソッドを呼び出す必要はありません。
        /// </summary>
        /// <returns>ダイアログを閉じる場合は true を返します。</returns>
        protected virtual bool OnResultCancel()
        {
            return true;
        }

        /// <summary>
        /// 「はい」ハンドラ。
        /// ダイアログを閉じる場合は true を返します。
        /// 基本クラスのメソッドを呼び出す必要はありません。
        /// </summary>
        /// <returns>ダイアログを閉じる場合は true を返します。</returns>
        protected virtual bool OnResultYes()
        {
            return true;
        }

        /// <summary>
        /// 「いいえ」ハンドラ。
        /// ダイアログを閉じる場合は true を返します。
        /// 基本クラスのメソッドを呼び出す必要はありません。
        /// </summary>
        /// <returns>ダイアログを閉じる場合は true を返します。</returns>
        protected virtual bool OnResultNo()
        {
            return true;
        }

        #endregion
    }
}
