﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Layout;
using Orientation = EffectMaker.UIControls.Layout.Orientation;

namespace EffectMaker.UIDialogs.CurveEditorDialog
{
    /// <summary>
    /// フォーカスを受け取れるスタックパネルです。
    /// </summary>
    internal class CurveListPanel : StackPanel
    {
        /// <summary>
        /// ボタンクリック時の処理
        /// </summary>
        private Action<UICheckBox, UIImageButton, int> onButtonClick;

        /// <summary>
        /// チェックボックス変更時の処理
        /// </summary>
        private Action<UICheckBox, int> onCheckClick;

        /// <summary>
        /// チャンネル選択処理
        /// </summary>
        private Action<int, bool> onSelect;

        /// <summary>
        /// 現在の選択チャンネル取得処理
        /// </summary>
        private Func<int> getSelectedChannel;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CurveListPanel()
        {
            this.SetStyle(ControlStyles.Selectable, true);
            this.TabStop = true;
        }

        /// <summary>
        /// コールバック処理をセットします。
        /// </summary>
        /// <param name="buttonClick">ボタンクリック時</param>
        /// <param name="checkClick">チェック変更時</param>
        /// <param name="select">選択時</param>
        /// <param name="selectedChannel">選択チャンネル取得</param>
        public void SetCallbacks(
            Action<UICheckBox, UIImageButton, int> buttonClick,
            Action<UICheckBox, int> checkClick,
            Action<int, bool> select,
            Func<int> selectedChannel)
        {
            this.onButtonClick = buttonClick;
            this.onCheckClick = checkClick;
            this.onSelect = select;
            this.getSelectedChannel = selectedChannel;
        }

        /// <summary>
        /// リストに項目を表すセクション区切りを追加.
        /// </summary>
        /// <param name="name">項目名</param>
        public void AddSection(string name)
        {
            var separator = new UILabel();
            var label = new UILabel();
            var panel = new UIPanel();

            panel.SuspendLayout();
            label.AutoSize = true;
            label.Text = name;
            label.Location = new Point(3, 0);
            separator.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
            separator.AutoSize = false;
            separator.Location = new Point(3 + label.Location.X, 6);
            separator.Size = new Size(310 - label.Width, 2);
            panel.BackColor = System.Drawing.Color.Transparent;
            panel.Controls.Add(label);
            panel.Controls.Add(separator);
            panel.ResumeLayout(true);
            this.Controls.Add(panel);
            this.PerformLayout();
        }

        /// <summary>
        /// リストにアニメーション項目を追加.
        /// </summary>
        /// <param name="name">タブ名</param>
        /// <param name="subName">アニメーション名</param>
        /// <param name="channelId">チャンネルID</param>
        public void AddChannel(
            string name,
            string subName,
            int channelId)
        {
            var panel = new StackPanel
            {
                AutoSize = true,
                MinimumSize = new Size(this.Width - 2, 26),
                VerticalAlignment = VAlignment.Center,
                Orientation = Orientation.Horizontal,
                Tag = channelId,
            };
            var checkBox = new UICheckBox
            {
                AutoSize = false,
                Size = new Size(26, 26),
                BitMask = 1u,
                MaskedValue = 0u,
                OnCheckedChangedExecutable = null,
                OnCheckedChangedExecutableParameter = null,
                TabStop = false,
                UseVisualStyleBackColor = true,
                Padding = new Padding(9, 0, 0, 0),
                Margin = new Padding(0),
                VerticalAlignment = VAlignment.Center,
                HorizontalAlignment = HAlignment.Center,
                Tag = channelId,
            };
            var button = new UIImageButton
            {
                Image = Properties.Resources.Icon_Anim_Off,
                NormalImage = Properties.Resources.Icon_Anim_Off,
                DownImage = Properties.Resources.Icon_Anim_On,
                MouseOverImage = Properties.Resources.Icon_Anim_Off_Over,
                DownMouseOverImage = Properties.Resources.Icon_Anim_On_Over,
                ToggleType = true,
                AutoSize = false,
                Size = new Size(26, 26),
                VerticalAlignment = VAlignment.Center,
                HorizontalAlignment = HAlignment.Center,
                Padding = new Padding(0, 0, 0, 4),
                Margin = new Padding(0),
                Tag = channelId,
            };
            var subPanel = new StackPanel
            {
                AutoSize = true,
                MinimumSize = new Size(panel.Width - 52 - 12, 26),
                VerticalAlignment = VAlignment.Center,
                Orientation = Orientation.Horizontal,
                Padding = new Padding(0),
                Margin = new Padding(0),
                Tag = channelId,
            };
            var label = new UILabel
            {
                AutoSize = true,
                Text = string.Format(" {0} /", name),
                VerticalAlignment = VAlignment.Center,
                Padding = new Padding(0),
                Margin = new Padding(0),
            };
            var subLabel = new UILabel
            {
                AutoSize = true,
                Font = new Font(FontFamily.GenericSansSerif, 9.0f, FontStyle.Bold),
                Text = subName,
                VerticalAlignment = VAlignment.Center,
                Padding = new Padding(0, 0, 0, 4),
                Margin = new Padding(0),
            };

            panel.SuspendLayout();
            panel.Controls.Add(checkBox);
            panel.Controls.Add(button);
            subPanel.SuspendLayout();
            subPanel.Controls.Add(label);
            subPanel.Controls.Add(subLabel);
            subPanel.ResumeLayout(true);
            panel.Controls.Add(subPanel);
            panel.ResumeLayout(true);

            this.Controls.Add(panel);
            this.PerformLayout();

            button.Click += (s, e) => this.onButtonClick(checkBox, button, channelId);
            checkBox.CheckedChanged += (s, e) => this.onCheckClick(checkBox, channelId);

            subPanel.MouseDown += (s, e) => OnSubPanelMouseDown(e, this.onSelect, channelId);
            label.MouseDown += (s, e) => OnSubPanelMouseDown(e, this.onSelect, channelId);
            subLabel.MouseDown += (s, e) => OnSubPanelMouseDown(e, this.onSelect, channelId);

            subPanel.MouseEnter += (s, e) => OnHover(subPanel, Control.DefaultBackColor, this.getSelectedChannel);
            label.MouseEnter += (s, e) => OnHover(subPanel, Control.DefaultBackColor, this.getSelectedChannel);
            subLabel.MouseEnter += (s, e) => OnHover(subPanel, Control.DefaultBackColor, this.getSelectedChannel);

            subPanel.MouseLeave += (s, e) => OnHover(subPanel, Color.Transparent, this.getSelectedChannel);
            label.MouseLeave += (s, e) => OnHover(subPanel, Color.Transparent, this.getSelectedChannel);
            subLabel.MouseLeave += (s, e) => OnHover(subPanel, Color.Transparent, this.getSelectedChannel);
        }

        /// <summary>
        /// ON/OFFボタンの状態を設定する
        /// 併せてチェックボックスの状態も更新する
        /// </summary>
        /// <param name="id">チャンネルID</param>
        /// <param name="state">状態</param>
        /// <param name="enabled">ボタンの有効無効フラグ(省略時有効)</param>
        public void SetButtonState(int id, bool state, bool enabled = true)
        {
            var panel = this.Controls.OfType<StackPanel>().First(x => (int)x.Tag == id);
            panel.Controls.OfType<UIImageButton>().First().Checked = state;
            panel.Controls.OfType<UICheckBox>().First().Checked = state;
            panel.Controls.OfType<UIImageButton>().First().Enabled = enabled;
        }

        /// <summary>
        /// 指定したIDのON/OFFボタンの状態を取得する
        /// </summary>
        /// <param name="id">チャンネル</param>
        /// <returns>状態</returns>
        public bool GetButtonState(int id)
        {
            var panel = this.Controls.OfType<StackPanel>().First(x => (int)x.Tag == id);
            return panel.Controls.OfType<UIImageButton>().First().Checked;
        }

        /// <summary>
        /// マウスダウンイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            this.Focus();
            base.OnMouseDown(e);
        }

        /// <summary>
        /// 入力キーのハンドリング設定
        /// </summary>
        /// <param name="keyData">キーデータ</param>
        /// <returns>ハンドリングするか否か</returns>
        protected override bool IsInputKey(Keys keyData)
        {
            // 上下キーをハンドリング対象にする
            if (keyData == Keys.Up || keyData == Keys.Down)
            {
                return true;
            }

            return base.IsInputKey(keyData);
        }

        /// <summary>
        /// マウスがリストの項目をホバーした際の処理
        /// </summary>
        /// <param name="panel">背景色を変えるパネル</param>
        /// <param name="color">変更したい背景色</param>
        /// <param name="getSelectedChannel">選択チャンネル取得関数</param>
        private static void OnHover(StackPanel panel, Color color, Func<int> getSelectedChannel)
        {
            if (getSelectedChannel() != (int)panel.Tag)
            {
                panel.BackColor = color;
            }
        }

        /// <summary>
        /// チャンネルパネル上でマウスダウンした際の処理
        /// </summary>
        /// <param name="args">マウスイベント引数</param>
        /// <param name="onSelect">選択処理</param>
        /// <param name="id">ID</param>
        private static void OnSubPanelMouseDown(MouseEventArgs args, Action<int, bool> onSelect, int id)
        {
            if (args.Button == MouseButtons.Left)
            {
                onSelect(id, true);
            }
        }
    }
}
