﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Foundation.Command;
using EffectMaker.UIControls.Input;
using WeifenLuo.WinFormsUI.Docking;

namespace EffectMaker.UIDialogs.CommandHistoryDialog
{
    /// <summary>
    /// Class for the command history dialog.
    /// </summary>
    /// WeifenLuo.WinFormsUI.Docking.DockContent
    public partial class CommandHistoryDialog : WeifenLuo.WinFormsUI.Docking.DockContent
    {
        /// <summary>
        /// リストビューのアップデートを一時的に停止するフラグ.
        /// </summary>
        private bool suppressUpdate = false;

        /// <summary>
        /// コンストラクタ.
        /// </summary>
        public CommandHistoryDialog()
        {
            this.DoubleBuffered = true;

            this.InitializeComponent();

            this.CreateInitialCommand();

            CommandManager.CommandExecuted += this.OnChangedCommand;
            CommandManager.ActiveTargetChanged += this.OnChangedActiveCommand;
        }

        /// <summary>
        /// コマンド履歴を更新する.
        /// </summary>
        public void UpdateCommandHistory()
        {
            if (this.suppressUpdate == true || this.Visible == false)
            {
                return;
            }

            this.commandListView.BeginUpdate();

            // リストアイテムのカーソル位置を復元するための情報を保持.
            // # この情報は、スクロールバーの位置の復元に利用される.
            int firstItemIndex = 0;
            int numItems = this.commandListView.Items.Count;
            if (this.commandListView.TopItem != null)
            {
                firstItemIndex = this.commandListView.TopItem.Index;
            }

            // リストビューをクリアする.
            this.commandListView.Items.Clear();

            CommandStack stack = CommandManager.GetActiveStack();
            if (stack == null)
            {
                // アクティブスタックが見つからない.
                this.CreateInitialCommand();

                this.commandListView.EndUpdate();
                return;
            }

            IEnumerable<CommandBase> redo = stack.RedoBuffer.Reverse();
            IEnumerable<CommandBase> undo = stack.UndoBuffer;

            int redoCount = redo.Count();
            int undoCount = undo.Count();
            int count = redoCount + undoCount;

            // Redoバッファの情報をリストビューへ追加.
            foreach (CommandBase cmd in redo)
            {
                string[] itemstr =
                {
                    string.Empty + count,
                    string.Format("{0}{1}", cmd.KeepExecuting ? "|   " : string.Empty, cmd.ToString()),
                    string.Empty,
                };

                var target = cmd.GetTarget();
                if (target != null)
                {
                    itemstr[2] = target.GetType().Name;
                }

                var lvi = new ListViewItem(itemstr)
                {
                    ForeColor = Color.LightGray,
                    Tag = new CommandData(redoCount--, ExecutionType.Redo),
                };

                this.commandListView.Items.Add(lvi);

                --count;
            }

            // Undoバッファの情報をリストビューへ追加.
            foreach (CommandBase cmd in undo)
            {
                string[] itemstr =
                {
                    string.Empty + count,
                    string.Format("{0}{1}", cmd.KeepExecuting ? "|   " : string.Empty, cmd.ToString()),
                    string.Empty
                };

                var target = cmd.GetTarget();
                if (target != null)
                {
                    itemstr[2] = target.GetType().Name;
                }

                var listItem = new ListViewItem(itemstr)
                {
                    Tag = new CommandData(undo.Count() - undoCount--, ExecutionType.Undo),
                };

                this.commandListView.Items.Add(listItem);

                --count;
            }

            this.CreateInitialCommand();

            this.commandListView.EndUpdate();

            // スクロールバーの位置を復元.
            if (numItems == this.commandListView.Items.Count &&
                firstItemIndex >= 0 &&
                firstItemIndex < this.commandListView.Items.Count)
            {
                this.commandListView.TopItem = this.commandListView.Items[firstItemIndex];
            }
        }

        /// <summary>
        /// コマンドスタックが変更された時のイベント.
        /// </summary>
        /// <param name="e">イベント引数.</param>
        protected void OnChangedCommand(CommandManager.CommandEventArgs e)
        {
            this.UpdateCommandHistory();
        }

        /// <summary>
        /// アクティブなターゲットが変更された時のイベント.
        /// </summary>
        /// <param name="e">イベント引数.</param>
        protected void OnChangedActiveCommand(CommandManager.CommandActiveTargetEventArgs e)
        {
            this.UpdateCommandHistory();
        }

        /// <summary>
        /// フローティングウィンドウ時のショートカットキーハンドリング
        /// </summary>
        /// <param name="msg">メッセージ</param>
        /// <param name="keyData">キーデータ</param>
        /// <returns>処理済みならtrueを返す。</returns>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            if (this.DockState == DockState.Float)
            {
                GlobalKeyEventHandler.Instance.ProcessShortcut(keyData);
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        /// <summary>
        /// コマンド選択イベントの処理.
        /// </summary>
        /// <param name="sender">Event sender.</param>
        /// <param name="e">Event Arguments.</param>
        private void OnCommandSelected(object sender, MouseEventArgs e)
        {
            if (this.suppressUpdate == true)
            {
                return;
            }

            ListViewItem item = this.commandListView.GetItemAt(e.X, e.Y);

            if ((item == null) || (item.Tag is CommandData) == false)
            {
                return;
            }

            var data = item.Tag as CommandData;

            this.suppressUpdate = true; // リストビュー再描画を抑止.

            if (data.CommandExecType == ExecutionType.Undo)
            {
                // 元に戻す処理.
                for (int i = 0; i < data.ExecutionCount; i++)
                {
                    int undoCount = CommandManager.GetActiveStack().UndoBuffer.Count();
                    if (CommandManager.Undo() == false)
                    {
                        break;
                    }

                    int exeCount = undoCount - CommandManager.GetActiveStack().UndoBuffer.Count();
                    i += exeCount - 1;

                    if (i >= data.ExecutionCount)
                    {
                        CommandManager.Redo();
                        break;
                    }
                }
            }
            else if (data.CommandExecType == ExecutionType.Redo)
            {
                // やり直し処理.
                for (int i = 0; i < data.ExecutionCount; i++)
                {
                    int redoCount = CommandManager.GetActiveStack().RedoBuffer.Count();
                    if (CommandManager.Redo() == false)
                    {
                        break;
                    }

                    int exeCount = redoCount - CommandManager.GetActiveStack().RedoBuffer.Count();
                    i += exeCount - 1;
                }
            }

            this.suppressUpdate = false; // リストビュー再描画の抑止を解除.

            // コマンド履歴リストを再描画する.
            this.UpdateCommandHistory();
        }

        /// <summary>
        /// 初期コマンドを作成して、リストへ追加する.
        /// </summary>
        private void CreateInitialCommand()
        {
            var item = new ListViewItem("0");

            item.ForeColor = Color.LightGray;

            item.SubItems.Add(Properties.Resources.CommandHistoryDialogInitialState);
            item.SubItems.Add(string.Empty);

            CommandStack stack = CommandManager.GetActiveStack();
            if (stack != null)
            {
                item.Tag = new CommandData(stack.UndoBuffer.Count(), ExecutionType.Undo);
            }

            this.commandListView.Items.Add(item);
        }

        /// <summary>
        /// 閉じるボタンが押された時のイベント.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event.</param>
        private void OnCloseButtonClicked(object sender, EventArgs e)
        {
            // フォームを非表示にする.
            this.Hide();
        }

        /// <summary>
        /// フォームが閉じる前に呼び出される.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event.</param>
        private void OnFormClosing(object sender, FormClosingEventArgs e)
        {
            // フォームはクローズしないで、非表示にする.
            this.Hide();

            e.Cancel = true; // キャンセルしないとダイアログが閉じてしまう.
        }

        #region Class for storing command data.

        /// <summary>
        /// Class for storing command data.
        /// </summary>
        private class CommandData
        {
            /// <summary>
            /// The constructor.
            /// </summary>
            /// <param name="execCount">実行のカウント.</param>
            /// <param name="cmdType">コマンドの実行タイプ.</param>
            public CommandData(int execCount, ExecutionType cmdType)
            {
                this.ExecutionCount = execCount;
                this.CommandExecType = cmdType;
            }

            /// <summary>
            /// 実行のカウント.
            /// </summary>
            public int ExecutionCount { get; set; }

            /// <summary>
            /// コマンドの実行タイプ.
            /// </summary>
            public ExecutionType CommandExecType { get; set; }
        }

        #endregion // Class for storing command data.
    }
}
